# -*- coding: utf-8 -*-
import logging

from django.contrib import admin
from django.utils.html import mark_safe
from django.utils.translation import gettext_lazy as _

from cms.plugin_pool import plugin_pool

from cms_forms.cms_plugins import FormParentPlugin

from .models import EmailNotification, EmailNotificationFormPlugin
from .notification import DefaultNotificationConf


logger = logging.getLogger(__name__)


class NewEmailNotificationInline(admin.StackedInline):
    extra = 1
    fields = ['theme']
    model = EmailNotification
    verbose_name = _('new email notification')
    verbose_name_plural = _('new email notifications')

    fieldsets = (
        (None, {
            'fields': (
                'theme',
            )
        }),
    )

    def get_queryset(self, request):
        queryset = super(NewEmailNotificationInline, self).get_queryset(
            request)
        return queryset.none()


class ExistingEmailNotificationInline(admin.StackedInline):
    model = EmailNotification

    fieldsets = (
        (None, {
            'fields': (
                'theme',
            )
        }),
        (_('Recipients'), {
            'classes': ('collapse',),
            'fields': (
                'text_variables',
                'to_user',
                ('to_name', 'to_email'),
                ('from_name', 'from_email'),
            )
        }),
    )

    readonly_fields = ['text_variables']
    text_variables_help_text = _('variables can be used with by '
                                 'wrapping with "${variable}" like ${variable}')

    def has_module_permission(self, request):
        return False

    def get_fieldsets(self, request, obj=None):
        fieldsets = super(ExistingEmailNotificationInline, self).get_fieldsets(
            request, obj)

        if obj is None:
            return fieldsets

        email_fieldset = self.get_email_fieldset(obj)
        fieldsets = list(fieldsets) + email_fieldset
        return fieldsets

    def get_email_fieldset(self, obj):
        fields = ['subject']

        notification_conf = obj.get_notification_conf()

        if notification_conf.txt_email_format_configurable:
            # add the body_text field only if it's configurable
            fields.append('body_text')

        if notification_conf.html_email_format_enabled:
            # add the body_html field only if email is allowed
            # to be sent in html version.
            fields.append('body_html')
        return [(_('Email'), {
            'classes': ('collapse',),
            'fields': fields
        })]

    def text_variables(self, obj):
        if obj.pk is None:
            return ''

        # list of tuples - [('category', [('value', 'label')])]
        choices_by_category = obj.form.get_notification_text_context_keys_as_choices()

        li_items = []

        for category, choices in choices_by_category:
            # <li>field_1</li><li>field_2</li>
            fields_li = ''.join(
                ('<li>{0} | {1}</li>'.format(*var) for var in choices))

            if fields_li:
                li_item = '<li>{0}</li><ul>{1}</ul>'.format(category,
                                                            fields_li)
                li_items.append(li_item)
        unordered_list = '<ul>{0}</ul>'.format(''.join(li_items))
        help_text = '<p class="help">{0}</p>'.format(
            self.text_variables_help_text)
        return mark_safe(unordered_list + '\n' + help_text)

    text_variables.allow_tags = True
    text_variables.short_description = _('available text variables')


class EmailNotificationForm(FormParentPlugin):
    name = _('Form (Advanced)')
    model = EmailNotificationFormPlugin
    inlines = [
        ExistingEmailNotificationInline,
        NewEmailNotificationInline
    ]
    notification_conf_class = DefaultNotificationConf

    fieldsets = (
        (None, {
            'fields': (
                'name',
                'redirect_type',
                ('redirect_page', 'url'),
            )
        }),
        (_('Advanced Settings'), {
            'classes': ('collapse',),
            'fields': (
                'form_template',
                'error_message',
                'success_message',
                'custom_classes',
                'action_backend',
            )
        }),
    )

    def get_inline_instances(self, request, obj=None):
        inlines = super(EmailNotificationForm, self).get_inline_instances(
            request, obj)

        if obj is None:
            # remove ExistingEmailNotificationInline inline instance
            # if we're first creating this object.
            inlines = [inline for inline in inlines
                       if
                       not isinstance(inline, ExistingEmailNotificationInline)]
        return inlines


plugin_pool.register_plugin(EmailNotificationForm)
