"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GlobalTableCoordinator = void 0;
const path = require("path");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
/**
 * A stack that will make a Lambda that will launch a lambda to glue
 * together all the DynamoDB tables into a global table
 */
class GlobalTableCoordinator extends cdk.Stack {
    constructor(scope, id, props) {
        super(scope, id, props);
        const lambdaFunction = new lambda.SingletonFunction(this, 'SingletonLambda', {
            code: lambda.Code.fromAsset(path.resolve(__dirname, '../', 'lambda-packages', 'aws-global-table-coordinator', 'lib')),
            description: 'Lambda to make DynamoDB a global table',
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_14_X,
            timeout: cdk.Duration.minutes(5),
            uuid: 'D38B65A6-6B54-4FB6-9BAD-9CD40A6DAC12',
        });
        grantCreateGlobalTableLambda(lambdaFunction.role);
        new cdk.CustomResource(this, 'CfnCustomResource', {
            serviceToken: lambdaFunction.functionArn,
            pascalCaseProperties: true,
            properties: {
                regions: props.regions,
                resourceType: 'Custom::DynamoGlobalTableCoordinator',
                tableName: props.tableName,
            },
            removalPolicy: props.removalPolicy,
        });
    }
}
exports.GlobalTableCoordinator = GlobalTableCoordinator;
/**
 * Permits an IAM Principal to create a global dynamodb table.
 * @param principal The principal (no-op if undefined)
 */
function grantCreateGlobalTableLambda(principal) {
    if (principal) {
        principal.addToPrincipalPolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: [
                'iam:CreateServiceLinkedRole',
                'application-autoscaling:DeleteScalingPolicy',
                'application-autoscaling:DeregisterScalableTarget',
                'dynamodb:CreateGlobalTable', 'dynamodb:DescribeLimits',
                'dynamodb:DeleteTable', 'dynamodb:DescribeGlobalTable',
                'dynamodb:UpdateGlobalTable',
            ],
        }));
    }
}
//# sourceMappingURL=data:application/json;base64,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