"use strict";
var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PodConnections = exports.PodConnectionsIsolation = exports.PodScheduling = exports.Topology = exports.Node = exports.NamedNode = exports.TaintedNode = exports.LabeledNode = exports.Pods = exports.NodeTaintQuery = exports.TaintEffect = exports.LabelExpression = exports.NodeLabelQuery = exports.DnsPolicy = exports.FsGroupChangePolicy = exports.RestartPolicy = exports.PodSecurityContext = exports.PodDns = exports.Pod = exports.LabelSelector = exports.AbstractPod = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const constructs_1 = require("constructs");
const base = require("./base");
const container = require("./container");
const k8s = require("./imports/k8s");
const networkpolicy = require("./network-policy");
const utils_1 = require("./utils");
class AbstractPod extends base.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this._containers = [];
        this._initContainers = [];
        this._hostAliases = [];
        this._volumes = new Map();
        this.restartPolicy = props.restartPolicy;
        this.serviceAccount = props.serviceAccount;
        this.securityContext = new PodSecurityContext(props.securityContext);
        this.dns = new PodDns(props.dns);
        this.dockerRegistryAuth = props.dockerRegistryAuth;
        this.automountServiceAccountToken = props.automountServiceAccountToken ?? true;
        if (props.containers) {
            props.containers.forEach(c => this.addContainer(c));
        }
        if (props.volumes) {
            props.volumes.forEach(v => this.addVolume(v));
        }
        if (props.initContainers) {
            props.initContainers.forEach(c => this.addInitContainer(c));
        }
        if (props.hostAliases) {
            props.hostAliases.forEach(c => this.addHostAlias(c));
        }
    }
    get containers() {
        return [...this._containers];
    }
    get initContainers() {
        return [...this._initContainers];
    }
    get volumes() {
        return Array.from(this._volumes.values());
    }
    get hostAliases() {
        return [...this._hostAliases];
    }
    /**
     * @see IPodSelector.toPodSelectorConfig()
     */
    toPodSelectorConfig() {
        const podAddress = this.podMetadata.getLabel(Pod.ADDRESS_LABEL);
        if (!podAddress) {
            // shouldn't happen because we add this label automatically in both pods and workloads.
            throw new Error(`Unable to create a label selector since ${Pod.ADDRESS_LABEL} label is missing`);
        }
        return {
            labelSelector: LabelSelector.of({ labels: { [Pod.ADDRESS_LABEL]: podAddress } }),
            namespaces: this.metadata.namespace ? {
                names: [this.metadata.namespace],
            } : undefined,
        };
    }
    /**
     * @see INetworkPolicyPeer.toNetworkPolicyPeerConfig()
     */
    toNetworkPolicyPeerConfig() {
        return { podSelector: this.toPodSelectorConfig() };
    }
    /**
     * @see INetworkPolicyPeer.toPodSelector()
     */
    toPodSelector() {
        return this;
    }
    addContainer(cont) {
        const impl = new container.Container(cont);
        this._containers.push(impl);
        return impl;
    }
    addInitContainer(cont) {
        // https://kubernetes.io/docs/concepts/workloads/pods/init-containers/#differences-from-regular-containers
        if (cont.readiness) {
            throw new Error('Init containers must not have a readiness probe');
        }
        if (cont.liveness) {
            throw new Error('Init containers must not have a liveness probe');
        }
        if (cont.startup) {
            throw new Error('Init containers must not have a startup probe');
        }
        const impl = new container.Container({
            ...cont,
            name: cont.name ?? `init-${this._initContainers.length}`,
        });
        this._initContainers.push(impl);
        return impl;
    }
    addHostAlias(hostAlias) {
        this._hostAliases.push(hostAlias);
    }
    addVolume(vol) {
        const existingVolume = this._volumes.get(vol.name);
        if (existingVolume) {
            throw new Error(`Volume with name ${vol.name} already exists`);
        }
        this._volumes.set(vol.name, vol);
    }
    /**
     * @see ISubect.toSubjectConfiguration()
     */
    toSubjectConfiguration() {
        if (!this.serviceAccount && !this.automountServiceAccountToken) {
            throw new Error(`${this.name} cannot be converted to a role binding subject:`
                + ' You must either assign a service account to it, or use \'automountServiceAccountToken: true\'');
        }
        // 'default' is assumed to be the name of the default service account
        // in the cluster.
        const serviceAccountName = this.serviceAccount?.name ?? 'default';
        return {
            kind: 'ServiceAccount',
            name: serviceAccountName,
            apiGroup: '',
        };
    }
    /**
     * @internal
     */
    _toPodSpec() {
        if (this.containers.length === 0) {
            throw new Error('PodSpec must have at least 1 container');
        }
        const volumes = new Map();
        const containers = [];
        const initContainers = [];
        for (const cont of this.containers) {
            // automatically add volume from the container mount
            // to this pod so thats its available to the container.
            for (const mount of cont.mounts) {
                addVolume(mount.volume);
            }
            containers.push(cont._toKube());
        }
        for (const cont of this.initContainers) {
            // automatically add volume from the container mount
            // to this pod so thats its available to the container.
            for (const mount of cont.mounts) {
                addVolume(mount.volume);
            }
            initContainers.push(cont._toKube());
        }
        for (const vol of this.volumes) {
            addVolume(vol);
        }
        function addVolume(vol) {
            const existingVolume = volumes.get(vol.name);
            // its ok to call this function twice on the same volume, but its not ok to
            // call it twice on a different volume with the same name.
            if (existingVolume && existingVolume !== vol) {
                throw new Error(`Invalid mount configuration. At least two different volumes have the same name: ${vol.name}`);
            }
            volumes.set(vol.name, vol);
        }
        const dns = this.dns._toKube();
        return {
            restartPolicy: this.restartPolicy,
            serviceAccountName: this.serviceAccount?.name,
            containers: containers,
            securityContext: utils_1.undefinedIfEmpty(this.securityContext._toKube()),
            initContainers: utils_1.undefinedIfEmpty(initContainers),
            hostAliases: utils_1.undefinedIfEmpty(this.hostAliases),
            volumes: utils_1.undefinedIfEmpty(Array.from(volumes.values()).map(v => v._toKube())),
            dnsPolicy: dns.policy,
            dnsConfig: utils_1.undefinedIfEmpty(dns.config),
            hostname: dns.hostname,
            subdomain: dns.subdomain,
            setHostnameAsFqdn: dns.hostnameAsFQDN,
            imagePullSecrets: this.dockerRegistryAuth ? [{ name: this.dockerRegistryAuth.name }] : undefined,
            automountServiceAccountToken: this.automountServiceAccountToken,
        };
    }
}
exports.AbstractPod = AbstractPod;
_a = JSII_RTTI_SYMBOL_1;
AbstractPod[_a] = { fqn: "cdk8s-plus-22.AbstractPod", version: "2.0.0-rc.37" };
/**
 * Match a resource by labels.
 */
class LabelSelector {
    constructor(expressions, labels) {
        this.expressions = expressions;
        this.labels = labels;
    }
    static of(options = {}) {
        return new LabelSelector(options.expressions ?? [], options.labels ?? {});
    }
    isEmpty() {
        return this.expressions.length === 0 && Object.keys(this.labels).length === 0;
    }
    /**
     * @internal
     */
    _toKube() {
        if (this.isEmpty()) {
            return {};
        }
        return {
            matchExpressions: utils_1.undefinedIfEmpty(this.expressions.map(q => ({ key: q.key, operator: q.operator, values: q.values }))),
            matchLabels: utils_1.undefinedIfEmpty(this.labels),
        };
    }
}
exports.LabelSelector = LabelSelector;
_b = JSII_RTTI_SYMBOL_1;
LabelSelector[_b] = { fqn: "cdk8s-plus-22.LabelSelector", version: "2.0.0-rc.37" };
/**
 * Pod is a collection of containers that can run on a host. This resource is
 * created by clients and scheduled onto hosts.
 */
class Pod extends AbstractPod {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.resourceType = 'pods';
        this.apiObject = new k8s.KubePod(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
        this.metadata.addLabel(Pod.ADDRESS_LABEL, cdk8s_1.Names.toLabelValue(this));
        this.scheduling = new PodScheduling(this);
        this.connections = new PodConnections(this);
    }
    get podMetadata() {
        return this.metadata;
    }
    /**
     * @internal
     */
    _toKube() {
        const scheduling = this.scheduling._toKube();
        return {
            ...this._toPodSpec(),
            affinity: scheduling.affinity,
            nodeName: scheduling.nodeName,
            tolerations: scheduling.tolerations,
        };
    }
}
exports.Pod = Pod;
_c = JSII_RTTI_SYMBOL_1;
Pod[_c] = { fqn: "cdk8s-plus-22.Pod", version: "2.0.0-rc.37" };
/**
 * This label is autoamtically added by cdk8s to any pod. It provides
 * a unique and stable identifier for the pod.
 */
Pod.ADDRESS_LABEL = 'cdk8s.io/metadata.addr';
/**
 * Holds dns settings of the pod.
 */
class PodDns {
    constructor(props = {}) {
        this.hostname = props.hostname;
        this.subdomain = props.subdomain;
        this.policy = props.policy ?? DnsPolicy.CLUSTER_FIRST;
        this.hostnameAsFQDN = props.hostnameAsFQDN ?? false;
        this._nameservers = props.nameservers ?? [];
        this._searches = props.searches ?? [];
        this._options = props.options ?? [];
    }
    /**
     * Nameservers defined for this pod.
     */
    get nameservers() {
        return [...this._nameservers];
    }
    /**
     * Search domains defined for this pod.
     */
    get searches() {
        return [...this._searches];
    }
    /**
     * Custom dns options defined for this pod.
     */
    get options() {
        return [...this._options];
    }
    /**
     * Add a nameserver.
     */
    addNameserver(...nameservers) {
        this._nameservers.push(...nameservers);
    }
    /**
     * Add a search domain.
     */
    addSearch(...searches) {
        this._searches.push(...searches);
    }
    /**
     * Add a custom option.
     */
    addOption(...options) {
        this._options.push(...options);
    }
    /**
     * @internal
     */
    _toKube() {
        if (this.policy === DnsPolicy.NONE && this.nameservers.length === 0) {
            throw new Error('When dns policy is set to NONE, at least one nameserver is required');
        }
        if (this.nameservers.length > 3) {
            throw new Error('There can be at most 3 nameservers specified');
        }
        if (this.searches.length > 6) {
            throw new Error('There can be at most 6 search domains specified');
        }
        return {
            hostname: this.hostname,
            subdomain: this.subdomain,
            hostnameAsFQDN: this.hostnameAsFQDN,
            policy: this.policy,
            config: {
                nameservers: utils_1.undefinedIfEmpty(this.nameservers),
                searches: utils_1.undefinedIfEmpty(this.searches),
                options: utils_1.undefinedIfEmpty(this.options),
            },
        };
    }
}
exports.PodDns = PodDns;
_d = JSII_RTTI_SYMBOL_1;
PodDns[_d] = { fqn: "cdk8s-plus-22.PodDns", version: "2.0.0-rc.37" };
/**
 * Holds pod-level security attributes and common container settings.
 */
class PodSecurityContext {
    constructor(props = {}) {
        this._sysctls = [];
        this.ensureNonRoot = props.ensureNonRoot ?? false;
        this.fsGroupChangePolicy = props.fsGroupChangePolicy ?? FsGroupChangePolicy.ALWAYS;
        this.user = props.user;
        this.group = props.group;
        this.fsGroup = props.fsGroup;
        for (const sysctl of props.sysctls ?? []) {
            this._sysctls.push(sysctl);
        }
    }
    get sysctls() {
        return [...this._sysctls];
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            runAsGroup: this.group,
            runAsUser: this.user,
            fsGroup: this.fsGroup,
            runAsNonRoot: this.ensureNonRoot,
            fsGroupChangePolicy: this.fsGroupChangePolicy,
            sysctls: utils_1.undefinedIfEmpty(this._sysctls),
        };
    }
}
exports.PodSecurityContext = PodSecurityContext;
_e = JSII_RTTI_SYMBOL_1;
PodSecurityContext[_e] = { fqn: "cdk8s-plus-22.PodSecurityContext", version: "2.0.0-rc.37" };
/**
 * Restart policy for all containers within the pod.
 */
var RestartPolicy;
(function (RestartPolicy) {
    /**
     * Always restart the pod after it exits.
     */
    RestartPolicy["ALWAYS"] = "Always";
    /**
     * Only restart if the pod exits with a non-zero exit code.
     */
    RestartPolicy["ON_FAILURE"] = "OnFailure";
    /**
     * Never restart the pod.
     */
    RestartPolicy["NEVER"] = "Never";
})(RestartPolicy = exports.RestartPolicy || (exports.RestartPolicy = {}));
var FsGroupChangePolicy;
(function (FsGroupChangePolicy) {
    /**
     * Only change permissions and ownership if permission and ownership of root directory does
     * not match with expected permissions of the volume.
     * This could help shorten the time it takes to change ownership and permission of a volume
     */
    FsGroupChangePolicy["ON_ROOT_MISMATCH"] = "OnRootMismatch";
    /**
     * Always change permission and ownership of the volume when volume is mounted.
     */
    FsGroupChangePolicy["ALWAYS"] = "Always";
})(FsGroupChangePolicy = exports.FsGroupChangePolicy || (exports.FsGroupChangePolicy = {}));
/**
 * Pod DNS policies.
 */
var DnsPolicy;
(function (DnsPolicy) {
    /**
     * Any DNS query that does not match the configured cluster domain suffix,
     * such as "www.kubernetes.io", is forwarded to the
     * upstream nameserver inherited from the node.
     * Cluster administrators may have extra stub-domain and upstream DNS servers configured.
     */
    DnsPolicy["CLUSTER_FIRST"] = "ClusterFirst";
    /**
     * For Pods running with hostNetwork, you should
     * explicitly set its DNS policy "ClusterFirstWithHostNet".
     */
    DnsPolicy["CLUSTER_FIRST_WITH_HOST_NET"] = "ClusterFirstWithHostNet";
    /**
     * The Pod inherits the name resolution configuration
     * from the node that the pods run on.
     */
    DnsPolicy["DEFAULT"] = "Default";
    /**
     * It allows a Pod to ignore DNS settings from the Kubernetes environment.
     * All DNS settings are supposed to be provided using the dnsConfig
     * field in the Pod Spec.
     */
    DnsPolicy["NONE"] = "None";
})(DnsPolicy = exports.DnsPolicy || (exports.DnsPolicy = {}));
/**
 * Represents a query that can be performed against nodes with labels.
 */
class NodeLabelQuery {
    constructor(key, operator, values) {
        this.key = key;
        this.operator = operator;
        this.values = values;
    }
    /**
     * Requires value of label `key` to equal `value`.
     */
    static is(key, value) {
        return NodeLabelQuery.in(key, [value]);
    }
    /**
     * Requires value of label `key` to be one of `values`.
     */
    static in(key, values) {
        return new NodeLabelQuery(key, 'In', values);
    }
    /**
     * Requires value of label `key` to be none of `values`.
     */
    static notIn(key, values) {
        return new NodeLabelQuery(key, 'NotIn', values);
    }
    /**
     * Requires label `key` to exist.
     */
    static exists(key) {
        return new NodeLabelQuery(key, 'Exists', undefined);
    }
    /**
     * Requires label `key` to not exist.
     */
    static doesNotExist(key) {
        return new NodeLabelQuery(key, 'DoesNotExist', undefined);
    }
    /**
     * Requires value of label `key` to greater than all elements in `values`.
     */
    static gt(key, values) {
        return new NodeLabelQuery(key, 'Gt', values);
    }
    /**
     * Requires value of label `key` to less than all elements in `values`.
     */
    static lt(key, values) {
        return new NodeLabelQuery(key, 'Lt', values);
    }
}
exports.NodeLabelQuery = NodeLabelQuery;
_f = JSII_RTTI_SYMBOL_1;
NodeLabelQuery[_f] = { fqn: "cdk8s-plus-22.NodeLabelQuery", version: "2.0.0-rc.37" };
/**
 * Represents a query that can be performed against resources with labels.
 */
class LabelExpression {
    constructor(key, operator, values) {
        this.key = key;
        this.operator = operator;
        this.values = values;
    }
    /**
     * Requires value of label `key` to be one of `values`.
     */
    static in(key, values) {
        return new LabelExpression(key, 'In', values);
    }
    /**
     * Requires value of label `key` to be none of `values`.
     */
    static notIn(key, values) {
        return new LabelExpression(key, 'NotIn', values);
    }
    /**
     * Requires label `key` to exist.
     */
    static exists(key) {
        return new LabelExpression(key, 'Exists', undefined);
    }
    /**
     * Requires label `key` to not exist.
     */
    static doesNotExist(key) {
        return new LabelExpression(key, 'DoesNotExist', undefined);
    }
}
exports.LabelExpression = LabelExpression;
_g = JSII_RTTI_SYMBOL_1;
LabelExpression[_g] = { fqn: "cdk8s-plus-22.LabelExpression", version: "2.0.0-rc.37" };
/**
 * Taint effects.
 */
var TaintEffect;
(function (TaintEffect) {
    /**
     * This means that no pod will be able to schedule
     * onto the node unless it has a matching toleration.
     */
    TaintEffect["NO_SCHEDULE"] = "NoSchedule";
    /**
     * This is a "preference" or "soft" version of `NO_SCHEDULE` -- the system
     * will try to avoid placing a pod that does not tolerate the taint on the node,
     * but it is not required
     */
    TaintEffect["PREFER_NO_SCHEDULE"] = "PreferNoSchedule";
    /**
     * This affects pods that are already running on the node as follows:
     *
     * - Pods that do not tolerate the taint are evicted immediately.
     * - Pods that tolerate the taint without specifying `duration` remain bound forever.
     * - Pods that tolerate the taint with a specified `duration` remain bound for
     *   the specified amount of time.
     */
    TaintEffect["NO_EXECUTE"] = "NoExecute";
})(TaintEffect = exports.TaintEffect || (exports.TaintEffect = {}));
/**
 * Taint queries that can be perfomed against nodes.
 */
class NodeTaintQuery {
    constructor(operator, key, value, effect, evictAfter) {
        this.operator = operator;
        this.key = key;
        this.value = value;
        this.effect = effect;
        this.evictAfter = evictAfter;
        if (evictAfter && effect !== TaintEffect.NO_EXECUTE) {
            throw new Error('Only \'NO_EXECUTE\' effects can specify \'evictAfter\'');
        }
    }
    /**
     * Matches a taint with a specific key and value.
     */
    static is(key, value, options = {}) {
        return new NodeTaintQuery('Equal', key, value, options.effect, options.evictAfter);
    }
    /**
     * Matches a tain with any value of a specific key.
     */
    static exists(key, options = {}) {
        return new NodeTaintQuery('Exists', key, undefined, options.effect, options.evictAfter);
    }
    /**
     * Matches any taint.
     */
    static any() {
        return new NodeTaintQuery('Exists');
    }
}
exports.NodeTaintQuery = NodeTaintQuery;
_h = JSII_RTTI_SYMBOL_1;
NodeTaintQuery[_h] = { fqn: "cdk8s-plus-22.NodeTaintQuery", version: "2.0.0-rc.37" };
/**
 * Represents a group of pods.
 */
class Pods extends constructs_1.Construct {
    constructor(scope, id, expressions, labels, namespaces) {
        super(scope, id);
        this.expressions = expressions;
        this.labels = labels;
        this.namespaces = namespaces;
    }
    /**
     * Select pods in the cluster with various selectors.
     */
    static select(scope, id, options) {
        return new Pods(scope, id, options.expressions, options.labels, options.namespaces);
    }
    /**
     * Select all pods.
     */
    static all(scope, id, options = {}) {
        return Pods.select(scope, id, { namespaces: options.namespaces });
    }
    /**
     * @see IPodSelector.toPodSelectorConfig()
     */
    toPodSelectorConfig() {
        return {
            labelSelector: LabelSelector.of({ expressions: this.expressions, labels: this.labels }),
            namespaces: this.namespaces?.toNamespaceSelectorConfig(),
        };
    }
    /**
     * @see INetworkPolicyPeer.toNetworkPolicyPeerConfig()
     */
    toNetworkPolicyPeerConfig() {
        return { podSelector: this.toPodSelectorConfig() };
    }
    /**
     * @see INetworkPolicyPeer.toPodSelector()
     */
    toPodSelector() {
        return this;
    }
}
exports.Pods = Pods;
_j = JSII_RTTI_SYMBOL_1;
Pods[_j] = { fqn: "cdk8s-plus-22.Pods", version: "2.0.0-rc.37" };
/**
 * A node that is matched by label selectors.
 */
class LabeledNode {
    constructor(labelSelector) {
        this.labelSelector = labelSelector;
    }
    ;
}
exports.LabeledNode = LabeledNode;
_k = JSII_RTTI_SYMBOL_1;
LabeledNode[_k] = { fqn: "cdk8s-plus-22.LabeledNode", version: "2.0.0-rc.37" };
/**
 * A node that is matched by taint selectors.
 */
class TaintedNode {
    constructor(taintSelector) {
        this.taintSelector = taintSelector;
    }
    ;
}
exports.TaintedNode = TaintedNode;
_l = JSII_RTTI_SYMBOL_1;
TaintedNode[_l] = { fqn: "cdk8s-plus-22.TaintedNode", version: "2.0.0-rc.37" };
/**
 * A node that is matched by its name.
 */
class NamedNode {
    constructor(name) {
        this.name = name;
    }
    ;
}
exports.NamedNode = NamedNode;
_m = JSII_RTTI_SYMBOL_1;
NamedNode[_m] = { fqn: "cdk8s-plus-22.NamedNode", version: "2.0.0-rc.37" };
/**
 * Represents a node in the cluster.
 */
class Node {
    /**
     * Match a node by its labels.
     */
    static labeled(...labelSelector) {
        return new LabeledNode(labelSelector);
    }
    /**
     * Match a node by its name.
     */
    static named(nodeName) {
        return new NamedNode(nodeName);
    }
    /**
     * Match a node by its taints.
     */
    static tainted(...taintSelector) {
        return new TaintedNode(taintSelector);
    }
}
exports.Node = Node;
_o = JSII_RTTI_SYMBOL_1;
Node[_o] = { fqn: "cdk8s-plus-22.Node", version: "2.0.0-rc.37" };
/**
 * Available topology domains.
 */
class Topology {
    constructor(key) {
        this.key = key;
    }
    /**
     * Custom key for the node label that the system uses to denote the topology domain.
     */
    static custom(key) {
        return new Topology(key);
    }
    ;
}
exports.Topology = Topology;
_p = JSII_RTTI_SYMBOL_1;
Topology[_p] = { fqn: "cdk8s-plus-22.Topology", version: "2.0.0-rc.37" };
/**
 * A hostname represents a single node in the cluster.
 *
 * @see https://kubernetes.io/docs/reference/labels-annotations-taints/#kubernetesiohostname
 */
Topology.HOSTNAME = new Topology('kubernetes.io/hostname');
/**
 * A zone represents a logical failure domain. It is common for Kubernetes clusters to
 * span multiple zones for increased availability. While the exact definition of a zone is
 * left to infrastructure implementations, common properties of a zone include very low
 * network latency within a zone, no-cost network traffic within a zone, and failure
 * independence from other zones. For example, nodes within a zone might share a network
 * switch, but nodes in different zones should not.
 *
 * @see https://kubernetes.io/docs/reference/labels-annotations-taints/#topologykubernetesiozone
 */
Topology.ZONE = new Topology('topology.kubernetes.io/zone');
/**
 * A region represents a larger domain, made up of one or more zones. It is uncommon
 * for Kubernetes clusters to span multiple regions. While the exact definition of a
 * zone or region is left to infrastructure implementations, common properties of a region
 * include higher network latency between them than within them, non-zero cost for network
 * traffic between them, and failure independence from other zones or regions.
 *
 * For example, nodes within a region might share power infrastructure (e.g. a UPS or generator), but
 * nodes in different regions typically would not.
 *
 * @see https://kubernetes.io/docs/reference/labels-annotations-taints/#topologykubernetesioregion
 */
Topology.REGION = new Topology('topology.kubernetes.io/region');
/**
 * Controls the pod scheduling strategy.
 */
class PodScheduling {
    constructor(instance) {
        this.instance = instance;
        this._nodeAffinityPreferred = [];
        this._nodeAffinityRequired = [];
        this._podAffinityPreferred = [];
        this._podAffinityRequired = [];
        this._podAntiAffinityPreferred = [];
        this._podAntiAffinityRequired = [];
        this._tolerations = [];
    }
    /**
     * Assign this pod a specific node by name.
     *
     * The scheduler ignores the Pod, and the kubelet on the named node
     * tries to place the Pod on that node. Overrules any affinity rules of the pod.
     *
     * Some limitations of static assignment are:
     *
     * - If the named node does not exist, the Pod will not run, and in some
     *   cases may be automatically deleted.
     * - If the named node does not have the resources to accommodate the Pod,
     *   the Pod will fail and its reason will indicate why, for example OutOfmemory or OutOfcpu.
     * - Node names in cloud environments are not always predictable or stable.
     *
     * Will throw is the pod is already assigned to named node.
     *
     * Under the hood, this method utilizes the `nodeName` property.
     */
    assign(node) {
        if (this._nodeName) {
            // disallow overriding an static node assignment
            throw new Error(`Cannot assign ${this.instance.podMetadata.name} to node ${node.name}. It is already assigned to node ${this._nodeName}`);
        }
        else {
            this._nodeName = node.name;
        }
    }
    /**
     * Allow this pod to tolerate taints matching these tolerations.
     *
     * You can put multiple taints on the same node and multiple tolerations on the same pod.
     * The way Kubernetes processes multiple taints and tolerations is like a filter: start with
     * all of a node's taints, then ignore the ones for which the pod has a matching toleration;
     * the remaining un-ignored taints have the indicated effects on the pod. In particular:
     *
     * - if there is at least one un-ignored taint with effect NoSchedule then Kubernetes will
     *   not schedule the pod onto that node
     * - if there is no un-ignored taint with effect NoSchedule but there is at least one un-ignored
     *   taint with effect PreferNoSchedule then Kubernetes will try to not schedule the pod onto the node
     * - if there is at least one un-ignored taint with effect NoExecute then the pod will be evicted from
     *   the node (if it is already running on the node), and will not be scheduled onto the node (if it is
     *   not yet running on the node).
     *
     * Under the hood, this method utilizes the `tolerations` property.
     *
     * @see https://kubernetes.io/docs/concepts/scheduling-eviction/taint-and-toleration/
     */
    tolerate(node) {
        for (const query of node.taintSelector) {
            this._tolerations.push({
                key: query.key,
                value: query.value,
                effect: query.effect,
                operator: query.operator,
                tolerationSeconds: query.evictAfter?.toSeconds(),
            });
        }
    }
    /**
     * Attract this pod to a node matched by selectors.
     * You can select a node by using `Node.labeled()`.
     *
     * Attracting to multiple nodes (i.e invoking this method multiple times) acts as
     * an OR condition, meaning the pod will be assigned to either one of the nodes.
     *
     * Under the hood, this method utilizes the `nodeAffinity` property.
     *
     * @see https://kubernetes.io/docs/concepts/scheduling-eviction/assign-pod-node/#node-affinity
     */
    attract(node, options = {}) {
        const term = this.createNodeAffinityTerm(node);
        if (options.weight) {
            this.validateWeight(options.weight);
            this._nodeAffinityPreferred.push({ weight: options.weight, preference: term });
        }
        else {
            this._nodeAffinityRequired.push(term);
        }
    }
    /**
     * Co-locate this pod with a scheduling selection.
     *
     * A selection can be one of:
     *
     * - An instance of a `Pod`.
     * - An instance of a `Workload` (e.g `Deployment`, `StatefulSet`).
     * - An un-managed pod that can be selected via `Pods.select()`.
     *
     * Co-locating with multiple selections ((i.e invoking this method multiple times)) acts as
     * an AND condition. meaning the pod will be assigned to a node that satisfies all
     * selections (i.e runs at least one pod that satisifies each selection).
     *
     * Under the hood, this method utilizes the `podAffinity` property.
     *
     * @see https://kubernetes.io/docs/concepts/scheduling-eviction/assign-pod-node/#inter-pod-affinity-and-anti-affinity
     */
    colocate(selector, options = {}) {
        const topology = options.topology ?? Topology.HOSTNAME;
        const term = this.createPodAffinityTerm(topology, selector);
        if (options.weight) {
            this.validateWeight(options.weight);
            this._podAffinityPreferred.push({ weight: options.weight, podAffinityTerm: term });
        }
        else {
            this._podAffinityRequired.push(term);
        }
    }
    /**
     * Seperate this pod from a scheduling selection.
     *
     * A selection can be one of:
     *
     * - An instance of a `Pod`.
     * - An instance of a `Workload` (e.g `Deployment`, `StatefulSet`).
     * - An un-managed pod that can be selected via `Pods.select()`.
     *
     * Seperating from multiple selections acts as an AND condition. meaning the pod
     * will not be assigned to a node that satisfies all selections (i.e runs at least one pod that satisifies each selection).
     *
     * Under the hood, this method utilizes the `podAntiAffinity` property.
     *
     * @see https://kubernetes.io/docs/concepts/scheduling-eviction/assign-pod-node/#inter-pod-affinity-and-anti-affinity
     */
    separate(selector, options = {}) {
        const topology = options.topology ?? Topology.HOSTNAME;
        const term = this.createPodAffinityTerm(topology, selector);
        if (options.weight) {
            this.validateWeight(options.weight);
            this._podAntiAffinityPreferred.push({ weight: options.weight, podAffinityTerm: term });
        }
        else {
            this._podAntiAffinityRequired.push(term);
        }
    }
    createPodAffinityTerm(topology, selector) {
        const config = selector.toPodSelectorConfig();
        return {
            topologyKey: topology.key,
            labelSelector: config.labelSelector._toKube(),
            namespaceSelector: config.namespaces?.labelSelector?._toKube(),
            namespaces: config.namespaces?.names,
        };
    }
    createNodeAffinityTerm(node) {
        return { matchExpressions: node.labelSelector.map(s => ({ key: s.key, operator: s.operator, values: s.values })) };
    }
    validateWeight(weight) {
        if (weight < 1 || weight > 100) {
            // https://kubernetes.io/docs/concepts/scheduling-eviction/assign-pod-node/#node-affinity-weight
            throw new Error(`Invalid affinity weight: ${weight}. Must be in range 1-100`);
        }
    }
    /**
     * @internal
     */
    _toKube() {
        const atLeastOne = (...arrays) => {
            return arrays.flat().length > 0;
        };
        const hasNodeAffinity = atLeastOne(this._nodeAffinityPreferred, this._nodeAffinityRequired);
        const hasPodAffinity = atLeastOne(this._podAffinityPreferred, this._podAffinityRequired);
        const hasPodAntiAffinty = atLeastOne(this._podAntiAffinityPreferred, this._podAntiAffinityRequired);
        const hasAffinity = hasNodeAffinity || hasPodAffinity || hasPodAntiAffinty;
        return {
            affinity: hasAffinity ? {
                nodeAffinity: hasNodeAffinity ? {
                    preferredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._nodeAffinityPreferred),
                    requiredDuringSchedulingIgnoredDuringExecution: this._nodeAffinityRequired.length > 0 ? {
                        nodeSelectorTerms: this._nodeAffinityRequired,
                    } : undefined,
                } : undefined,
                podAffinity: hasPodAffinity ? {
                    preferredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._podAffinityPreferred),
                    requiredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._podAffinityRequired),
                } : undefined,
                podAntiAffinity: hasPodAntiAffinty ? {
                    preferredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._podAntiAffinityPreferred),
                    requiredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._podAntiAffinityRequired),
                } : undefined,
            } : undefined,
            nodeName: this._nodeName,
            tolerations: utils_1.undefinedIfEmpty(this._tolerations),
        };
    }
}
exports.PodScheduling = PodScheduling;
_q = JSII_RTTI_SYMBOL_1;
PodScheduling[_q] = { fqn: "cdk8s-plus-22.PodScheduling", version: "2.0.0-rc.37" };
/**
 * Isolation determines which policies are created
 * when allowing connections from a a pod / workload to peers.
 */
var PodConnectionsIsolation;
(function (PodConnectionsIsolation) {
    /**
     * Only creates network policies that select the pod.
     */
    PodConnectionsIsolation["POD"] = "POD";
    /**
     * Only creates network policies that select the peer.
     */
    PodConnectionsIsolation["PEER"] = "PEER";
})(PodConnectionsIsolation = exports.PodConnectionsIsolation || (exports.PodConnectionsIsolation = {}));
/**
 * Controls network isolation rules for inter-pod communication.
 */
class PodConnections {
    constructor(instance) {
        this.instance = instance;
    }
    /**
     * Allow network traffic from this pod to the peer.
     *
     * By default, this will create an egress network policy for this pod, and an ingress
     * network policy for the peer. This is required if both sides are already isolated.
     * Use `options.isolation` to control this behavior.
     *
     * @example
     *
     * // create only an egress policy that selects the 'web' pod to allow outgoing traffic
     * // to the 'redis' pod. this requires the 'redis' pod to not be isolated for ingress.
     * web.connections.allowTo(redis, { isolation: Isolation.POD })
     *
     * // create only an ingress policy that selects the 'redis' peer to allow incoming traffic
     * // from the 'web' pod. this requires the 'web' pod to not be isolated for egress.
     * web.connections.allowTo(redis, { isolation: Isolation.PEER })
     *
     */
    allowTo(peer, options = {}) {
        return this.allow('Egress', peer, { ports: this.extractPorts(peer), ...options });
    }
    /**
     * Allow network traffic from the peer to this pod.
     *
     * By default, this will create an ingress network policy for this pod, and an egress
     * network policy for the peer. This is required if both sides are already isolated.
     * Use `options.isolation` to control this behavior.
     *
     * @example
     *
     * // create only an egress policy that selects the 'web' pod to allow outgoing traffic
     * // to the 'redis' pod. this requires the 'redis' pod to not be isolated for ingress.
     * redis.connections.allowFrom(web, { isolation: Isolation.PEER })
     *
     * // create only an ingress policy that selects the 'redis' peer to allow incoming traffic
     * // from the 'web' pod. this requires the 'web' pod to not be isolated for egress.
     * redis.connections.allowFrom(web, { isolation: Isolation.POD })
     *
     */
    allowFrom(peer, options = {}) {
        return this.allow('Ingress', peer, { ports: this.extractPorts(this.instance), ...options });
    }
    allow(direction, peer, options = {}) {
        const config = peer.toNetworkPolicyPeerConfig();
        networkpolicy.validatePeerConfig(config);
        const peerAddress = utils_1.address(peer);
        if (!options.isolation || options.isolation === PodConnectionsIsolation.POD) {
            const src = new networkpolicy.NetworkPolicy(this.instance, `Allow${direction}${peerAddress}`, {
                selector: this.instance,
                // the policy must be defined in the namespace of the pod
                // so it can select it.
                metadata: { namespace: this.instance.metadata.namespace },
            });
            switch (direction) {
                case 'Egress':
                    src.addEgressRule(peer, options.ports);
                    break;
                case 'Ingress':
                    src.addIngressRule(peer, options.ports);
            }
        }
        if (!options.isolation || options.isolation === PodConnectionsIsolation.PEER) {
            if (config.ipBlock) {
                // for an ip block we don't need to create the opposite policies
                return;
            }
            const podSelector = peer.toPodSelector();
            if (!podSelector) {
                throw new Error(`Unable to create policies for peer '${peer.node.addr}' since its not a pod selector`);
            }
            const oppositeDirection = direction === 'Egress' ? 'Ingress' : 'Egress';
            const podSelectorConfig = podSelector.toPodSelectorConfig();
            let namespaces;
            if (!podSelectorConfig.namespaces) {
                // if the peer doesn't specify namespaces, we assume the same namespace.
                namespaces = [this.instance.metadata.namespace];
            }
            else {
                // a peer cannot specify namespaces by labels because
                // we won't be able to extract the names of those namespaces.
                if (podSelectorConfig.namespaces.labelSelector && !podSelectorConfig.namespaces.labelSelector.isEmpty()) {
                    throw new Error(`Unable to create an ${oppositeDirection} policy for peer '${peer.node.path}' (pod=${this.instance.name}). Peer must specify namespaces only by name`);
                }
                // a peer must specify namespaces by name.
                if (!podSelectorConfig.namespaces.names) {
                    throw new Error(`Unable to create an ${oppositeDirection} policy for peer '${peer.node.path}' (pod=${this.instance.name}). Peer must specify namespace names`);
                }
                namespaces = podSelectorConfig.namespaces.names;
            }
            for (const name of namespaces) {
                switch (direction) {
                    case 'Egress':
                        new networkpolicy.NetworkPolicy(this.instance, `AllowIngress${name}${peerAddress}`, {
                            selector: podSelector,
                            metadata: { namespace: name },
                            ingress: { rules: [{ peer: this.instance, ports: options.ports }] },
                        });
                        break;
                    case 'Ingress':
                        new networkpolicy.NetworkPolicy(this.instance, `AllowEgress${name}${peerAddress}`, {
                            selector: podSelector,
                            metadata: { namespace: name },
                            egress: { rules: [{ peer: this.instance, ports: options.ports }] },
                        });
                        break;
                    default:
                        throw new Error(`Unsupported direction: ${direction}`);
                }
            }
        }
    }
    extractPorts(selector) {
        return container.extractContainerPorts(selector).map(n => networkpolicy.NetworkPolicyPort.tcp(n));
    }
}
exports.PodConnections = PodConnections;
_r = JSII_RTTI_SYMBOL_1;
PodConnections[_r] = { fqn: "cdk8s-plus-22.PodConnections", version: "2.0.0-rc.37" };
//# sourceMappingURL=data:application/json;base64,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