import numpy as np
from ase import Atoms
from networkx import cycle_basis

from tscode.algebra import dihedral, norm_of
from tscode.calculators._xtb import xtb_opt, xtb_pre_opt
from tscode.graph_manipulations import neighbors
from tscode.hypermolecule_class import align_structures
from tscode.mep_relaxer import interpolate_structures
from tscode.optimization_methods import optimize
from tscode.utils import write_xyz
from tscode.rmsd_pruning import rmsd_and_max_numba


def automep(embedder, n_images=9):

    assert embedder.options.calculator == "XTB"
    assert len(embedder.objects) == 2, "Provide two molecules as start/endpoints."

    mol = embedder.objects[0]
    coords = mol.atomcoords[0]

    # Get cycle indices between 7 and 9
    # graph = graphize(coords, mol.atomnos)
    cycles = [l for l in cycle_basis(mol.graph) if len(l) in (7, 8, 9)]
    assert len(cycles) == 1, "Automep only works for 7/8/9-membered ring flips at the moment"

    embedder.log(f'--> AutoMEP - Building MEP for {len(cycles[0])}-membered ring inversion')
    embedder.log(f'    Preoptimizing starting point at {embedder.options.calculator}/{embedder.options.theory_level}({embedder.options.solvent}) level')

    print("    - Optimizing starting point...", end="\r")
    coords, _, _ = xtb_opt(
                            coords,
                            mol.atomnos,
                            method=embedder.options.theory_level,
                            solvent=embedder.options.solvent,
                            procs=embedder.procs
                        )

    dihedrals = cycle_to_dihedrals(cycles[0])
    exocyclic = get_exocyclic_dihedrals(mol.graph, cycles[0])

    # start_angles = np.array([dihedral(coords[d]) for d in dihedrals+exocyclic])
    target_angles = np.array([0 for _ in dihedrals] + [180 for _ in exocyclic])
    print("    - Optimizing planar TS guess...", end="\r")
    ts_guess, _, _ = xtb_opt(
                            coords,
                            mol.atomnos,
                            constrained_dihedrals=dihedrals+exocyclic,
                            constrained_dih_angles=target_angles,
                            method=embedder.options.theory_level,
                            solvent=embedder.options.solvent,
                            procs=embedder.procs
                        )
    # multipliers = np.linspace(1, -1, n_images)

    # mep_angles = [(start_angles * m + target_angles * (1-m)) % 360 for m in multipliers]
    # mep = []
    # for i, m_a in enumerate(mep_angles):
    #     t_start = time.perf_counter()
    #     coords, _, _ = xtb_opt(coords,
    #                             mol.atomnos,
    #                             constrained_dihedrals=dihedrals+exocyclic,
    #                             constrained_dih_angles=m_a,
    #                             method=embedder.options.theory_level,
    #                             solvent=embedder.options.solvent,
    #                             procs=embedder.procs)
    #     embedder.log(f'    - optimized image {i+1}/{len(mep_angles)} ({round(time.perf_counter()-t_start, 3)} s)')
    #     mep.append(coords)

    mep = interpolate_structures(align_structures(np.array([coords,
                                                            ts_guess,
                                                            embedder.objects[1].atomcoords[0]])),
                                 mol.atomnos,
                                 n=n_images,
                                 method='linear')
    
    constrained_indices = [[a, b] for (a, b) in mol.graph.edges if a!=b]
    constrained_distances = [norm_of(coords[a]-coords[b]) for (a, b) in constrained_indices]

    for g, geom in enumerate(mep):
        if g not in (0, n_images-1):
            print(f"    - Relaxing image {g+1}/{n_images}...", end="\r")
            positions = geom.get_positions()
            opt_geom, _, _ = xtb_pre_opt(
                                positions,
                                mol.atomnos,
                                graphs=[mol.graph],
                                constrained_indices=constrained_indices,
                                constrained_distances=constrained_distances,
                                constrained_dihedrals=dihedrals+exocyclic,
                                constrained_dih_angles=[dihedral(positions[quadruplet]) for quadruplet in dihedrals+exocyclic],
                                method=embedder.options.theory_level,
                                solvent=embedder.options.solvent,
                                procs=embedder.procs
                            )
            mep[g] = Atoms(mol.atomnos, positions=opt_geom)

    mep_array = np.array([c.get_positions() for c in mep], dtype=float)
    mep_array = align_structures(mep_array)
    with open(f"{mol.rootname}_automep.xyz", "w") as f:
        for c in mep_array:
            write_xyz(c, mol.atomnos, f)

    embedder.log(f"\n--> Saved autogenerated MEP as {mol.rootname}_automep.xyz\n")

    return f"{mol.rootname}_automep.xyz"

def get_exocyclic_dihedrals(graph, cycle):
    '''
    '''
    exo_dihs = []
    for index in cycle:
        for exo_id in neighbors(graph, index):
            if exo_id not in cycle:
                dummy1 = next(i for i in cycle if i not in (exo_id, index) and i in neighbors(graph, index))
                dummy2 = next(i for i in cycle if i not in (exo_id, index, dummy1) and i in neighbors(graph, dummy1))
                exo_dihs.append([exo_id, index, dummy1, dummy2])

    return exo_dihs    

def cycle_to_dihedrals(cycle):
    '''
    '''
    dihedrals = []
    for i in range(len(cycle)):

        a = cycle[i % len(cycle)]
        b = cycle[(i+1) % len(cycle)]
        c = cycle[(i+2) % len(cycle)]
        d = cycle[(i+3) % len(cycle)]
        dihedrals.append([a, b, c, d])
    return dihedrals