"use strict";
var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PodConnections = exports.PodConnectionsIsolation = exports.PodScheduling = exports.Topology = exports.Node = exports.NamedNode = exports.TaintedNode = exports.LabeledNode = exports.Pods = exports.NodeTaintQuery = exports.TaintEffect = exports.LabelExpression = exports.NodeLabelQuery = exports.DnsPolicy = exports.FsGroupChangePolicy = exports.RestartPolicy = exports.PodSecurityContext = exports.PodDns = exports.Pod = exports.LabelSelector = exports.AbstractPod = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const constructs_1 = require("constructs");
const base = require("./base");
const container = require("./container");
const k8s = require("./imports/k8s");
const networkpolicy = require("./network-policy");
const utils_1 = require("./utils");
class AbstractPod extends base.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this._containers = [];
        this._initContainers = [];
        this._hostAliases = [];
        this._volumes = new Map();
        this.restartPolicy = props.restartPolicy ?? RestartPolicy.ALWAYS;
        this.serviceAccount = props.serviceAccount;
        this.securityContext = new PodSecurityContext(props.securityContext);
        this.dns = new PodDns(props.dns);
        this.dockerRegistryAuth = props.dockerRegistryAuth;
        this.automountServiceAccountToken = props.automountServiceAccountToken ?? false;
        if (props.containers) {
            props.containers.forEach(c => this.addContainer(c));
        }
        if (props.volumes) {
            props.volumes.forEach(v => this.addVolume(v));
        }
        if (props.initContainers) {
            props.initContainers.forEach(c => this.addInitContainer(c));
        }
        if (props.hostAliases) {
            props.hostAliases.forEach(c => this.addHostAlias(c));
        }
    }
    get containers() {
        return [...this._containers];
    }
    get initContainers() {
        return [...this._initContainers];
    }
    get volumes() {
        return Array.from(this._volumes.values());
    }
    get hostAliases() {
        return [...this._hostAliases];
    }
    /**
     * @see IPodSelector.toPodSelectorConfig()
     */
    toPodSelectorConfig() {
        const podAddress = this.podMetadata.getLabel(Pod.ADDRESS_LABEL);
        if (!podAddress) {
            // shouldn't happen because we add this label automatically in both pods and workloads.
            throw new Error(`Unable to create a label selector since ${Pod.ADDRESS_LABEL} label is missing`);
        }
        return {
            labelSelector: LabelSelector.of({ labels: { [Pod.ADDRESS_LABEL]: podAddress } }),
            namespaces: this.metadata.namespace ? {
                names: [this.metadata.namespace],
            } : undefined,
        };
    }
    /**
     * @see INetworkPolicyPeer.toNetworkPolicyPeerConfig()
     */
    toNetworkPolicyPeerConfig() {
        return { podSelector: this.toPodSelectorConfig() };
    }
    /**
     * @see INetworkPolicyPeer.toPodSelector()
     */
    toPodSelector() {
        return this;
    }
    addContainer(cont) {
        const impl = new container.Container(cont);
        this._containers.push(impl);
        return impl;
    }
    addInitContainer(cont) {
        // https://kubernetes.io/docs/concepts/workloads/pods/init-containers/#differences-from-regular-containers
        if (cont.readiness) {
            throw new Error('Init containers must not have a readiness probe');
        }
        if (cont.liveness) {
            throw new Error('Init containers must not have a liveness probe');
        }
        if (cont.startup) {
            throw new Error('Init containers must not have a startup probe');
        }
        const impl = new container.Container({
            ...cont,
            name: cont.name ?? `init-${this._initContainers.length}`,
        });
        this._initContainers.push(impl);
        return impl;
    }
    addHostAlias(hostAlias) {
        this._hostAliases.push(hostAlias);
    }
    addVolume(vol) {
        const existingVolume = this._volumes.get(vol.name);
        if (existingVolume) {
            throw new Error(`Volume with name ${vol.name} already exists`);
        }
        this._volumes.set(vol.name, vol);
    }
    /**
     * @see ISubect.toSubjectConfiguration()
     */
    toSubjectConfiguration() {
        if (!this.serviceAccount && !this.automountServiceAccountToken) {
            throw new Error(`${this.name} cannot be converted to a role binding subject:`
                + ' You must either assign a service account to it, or use \'automountServiceAccountToken: true\'');
        }
        // 'default' is assumed to be the name of the default service account
        // in the cluster.
        const serviceAccountName = this.serviceAccount?.name ?? 'default';
        return {
            kind: 'ServiceAccount',
            name: serviceAccountName,
            apiGroup: '',
        };
    }
    /**
     * @internal
     */
    _toPodSpec() {
        if (this.containers.length === 0) {
            throw new Error('PodSpec must have at least 1 container');
        }
        const volumes = new Map();
        const containers = [];
        const initContainers = [];
        for (const cont of this.containers) {
            // automatically add volume from the container mount
            // to this pod so thats its available to the container.
            for (const mount of cont.mounts) {
                addVolume(mount.volume);
            }
            containers.push(cont._toKube());
        }
        for (const cont of this.initContainers) {
            // automatically add volume from the container mount
            // to this pod so thats its available to the container.
            for (const mount of cont.mounts) {
                addVolume(mount.volume);
            }
            initContainers.push(cont._toKube());
        }
        for (const vol of this.volumes) {
            addVolume(vol);
        }
        function addVolume(vol) {
            const existingVolume = volumes.get(vol.name);
            // its ok to call this function twice on the same volume, but its not ok to
            // call it twice on a different volume with the same name.
            if (existingVolume && existingVolume !== vol) {
                throw new Error(`Invalid mount configuration. At least two different volumes have the same name: ${vol.name}`);
            }
            volumes.set(vol.name, vol);
        }
        const dns = this.dns._toKube();
        return {
            restartPolicy: this.restartPolicy,
            serviceAccountName: this.serviceAccount?.name,
            containers: containers,
            securityContext: utils_1.undefinedIfEmpty(this.securityContext._toKube()),
            initContainers: utils_1.undefinedIfEmpty(initContainers),
            hostAliases: utils_1.undefinedIfEmpty(this.hostAliases),
            volumes: utils_1.undefinedIfEmpty(Array.from(volumes.values()).map(v => v._toKube())),
            dnsPolicy: dns.policy,
            dnsConfig: utils_1.undefinedIfEmpty(dns.config),
            hostname: dns.hostname,
            subdomain: dns.subdomain,
            setHostnameAsFqdn: dns.hostnameAsFQDN,
            imagePullSecrets: this.dockerRegistryAuth ? [{ name: this.dockerRegistryAuth.name }] : undefined,
            automountServiceAccountToken: this.automountServiceAccountToken,
        };
    }
}
exports.AbstractPod = AbstractPod;
_a = JSII_RTTI_SYMBOL_1;
AbstractPod[_a] = { fqn: "cdk8s-plus-22.AbstractPod", version: "2.0.0-rc.131" };
/**
 * Match a resource by labels.
 */
class LabelSelector {
    constructor(expressions, labels) {
        this.expressions = expressions;
        this.labels = labels;
    }
    static of(options = {}) {
        return new LabelSelector(options.expressions ?? [], options.labels ?? {});
    }
    isEmpty() {
        return this.expressions.length === 0 && Object.keys(this.labels).length === 0;
    }
    /**
     * @internal
     */
    _toKube() {
        if (this.isEmpty()) {
            return {};
        }
        return {
            matchExpressions: utils_1.undefinedIfEmpty(this.expressions.map(q => ({ key: q.key, operator: q.operator, values: q.values }))),
            matchLabels: utils_1.undefinedIfEmpty(this.labels),
        };
    }
}
exports.LabelSelector = LabelSelector;
_b = JSII_RTTI_SYMBOL_1;
LabelSelector[_b] = { fqn: "cdk8s-plus-22.LabelSelector", version: "2.0.0-rc.131" };
/**
 * Pod is a collection of containers that can run on a host. This resource is
 * created by clients and scheduled onto hosts.
 */
class Pod extends AbstractPod {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.resourceType = 'pods';
        this.apiObject = new k8s.KubePod(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
        this.metadata.addLabel(Pod.ADDRESS_LABEL, cdk8s_1.Names.toLabelValue(this));
        this.scheduling = new PodScheduling(this);
        this.connections = new PodConnections(this);
    }
    get podMetadata() {
        return this.metadata;
    }
    /**
     * @internal
     */
    _toKube() {
        const scheduling = this.scheduling._toKube();
        return {
            ...this._toPodSpec(),
            affinity: scheduling.affinity,
            nodeName: scheduling.nodeName,
            tolerations: scheduling.tolerations,
        };
    }
}
exports.Pod = Pod;
_c = JSII_RTTI_SYMBOL_1;
Pod[_c] = { fqn: "cdk8s-plus-22.Pod", version: "2.0.0-rc.131" };
/**
 * This label is autoamtically added by cdk8s to any pod. It provides
 * a unique and stable identifier for the pod.
 */
Pod.ADDRESS_LABEL = 'cdk8s.io/metadata.addr';
/**
 * Holds dns settings of the pod.
 */
class PodDns {
    constructor(props = {}) {
        this.hostname = props.hostname;
        this.subdomain = props.subdomain;
        this.policy = props.policy ?? DnsPolicy.CLUSTER_FIRST;
        this.hostnameAsFQDN = props.hostnameAsFQDN ?? false;
        this._nameservers = props.nameservers ?? [];
        this._searches = props.searches ?? [];
        this._options = props.options ?? [];
    }
    /**
     * Nameservers defined for this pod.
     */
    get nameservers() {
        return [...this._nameservers];
    }
    /**
     * Search domains defined for this pod.
     */
    get searches() {
        return [...this._searches];
    }
    /**
     * Custom dns options defined for this pod.
     */
    get options() {
        return [...this._options];
    }
    /**
     * Add a nameserver.
     */
    addNameserver(...nameservers) {
        this._nameservers.push(...nameservers);
    }
    /**
     * Add a search domain.
     */
    addSearch(...searches) {
        this._searches.push(...searches);
    }
    /**
     * Add a custom option.
     */
    addOption(...options) {
        this._options.push(...options);
    }
    /**
     * @internal
     */
    _toKube() {
        if (this.policy === DnsPolicy.NONE && this.nameservers.length === 0) {
            throw new Error('When dns policy is set to NONE, at least one nameserver is required');
        }
        if (this.nameservers.length > 3) {
            throw new Error('There can be at most 3 nameservers specified');
        }
        if (this.searches.length > 6) {
            throw new Error('There can be at most 6 search domains specified');
        }
        return {
            hostname: this.hostname,
            subdomain: this.subdomain,
            hostnameAsFQDN: this.hostnameAsFQDN,
            policy: this.policy,
            config: {
                nameservers: utils_1.undefinedIfEmpty(this.nameservers),
                searches: utils_1.undefinedIfEmpty(this.searches),
                options: utils_1.undefinedIfEmpty(this.options),
            },
        };
    }
}
exports.PodDns = PodDns;
_d = JSII_RTTI_SYMBOL_1;
PodDns[_d] = { fqn: "cdk8s-plus-22.PodDns", version: "2.0.0-rc.131" };
/**
 * Holds pod-level security attributes and common container settings.
 */
class PodSecurityContext {
    constructor(props = {}) {
        this._sysctls = [];
        this.ensureNonRoot = props.ensureNonRoot ?? true;
        this.fsGroupChangePolicy = props.fsGroupChangePolicy ?? FsGroupChangePolicy.ALWAYS;
        this.user = props.user;
        this.group = props.group;
        this.fsGroup = props.fsGroup;
        for (const sysctl of props.sysctls ?? []) {
            this._sysctls.push(sysctl);
        }
    }
    get sysctls() {
        return [...this._sysctls];
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            runAsGroup: this.group,
            runAsUser: this.user,
            fsGroup: this.fsGroup,
            runAsNonRoot: this.ensureNonRoot,
            fsGroupChangePolicy: this.fsGroupChangePolicy,
            sysctls: utils_1.undefinedIfEmpty(this._sysctls),
        };
    }
}
exports.PodSecurityContext = PodSecurityContext;
_e = JSII_RTTI_SYMBOL_1;
PodSecurityContext[_e] = { fqn: "cdk8s-plus-22.PodSecurityContext", version: "2.0.0-rc.131" };
/**
 * Restart policy for all containers within the pod.
 */
var RestartPolicy;
(function (RestartPolicy) {
    /**
     * Always restart the pod after it exits.
     */
    RestartPolicy["ALWAYS"] = "Always";
    /**
     * Only restart if the pod exits with a non-zero exit code.
     */
    RestartPolicy["ON_FAILURE"] = "OnFailure";
    /**
     * Never restart the pod.
     */
    RestartPolicy["NEVER"] = "Never";
})(RestartPolicy = exports.RestartPolicy || (exports.RestartPolicy = {}));
var FsGroupChangePolicy;
(function (FsGroupChangePolicy) {
    /**
     * Only change permissions and ownership if permission and ownership of root directory does
     * not match with expected permissions of the volume.
     * This could help shorten the time it takes to change ownership and permission of a volume
     */
    FsGroupChangePolicy["ON_ROOT_MISMATCH"] = "OnRootMismatch";
    /**
     * Always change permission and ownership of the volume when volume is mounted.
     */
    FsGroupChangePolicy["ALWAYS"] = "Always";
})(FsGroupChangePolicy = exports.FsGroupChangePolicy || (exports.FsGroupChangePolicy = {}));
/**
 * Pod DNS policies.
 */
var DnsPolicy;
(function (DnsPolicy) {
    /**
     * Any DNS query that does not match the configured cluster domain suffix,
     * such as "www.kubernetes.io", is forwarded to the
     * upstream nameserver inherited from the node.
     * Cluster administrators may have extra stub-domain and upstream DNS servers configured.
     */
    DnsPolicy["CLUSTER_FIRST"] = "ClusterFirst";
    /**
     * For Pods running with hostNetwork, you should
     * explicitly set its DNS policy "ClusterFirstWithHostNet".
     */
    DnsPolicy["CLUSTER_FIRST_WITH_HOST_NET"] = "ClusterFirstWithHostNet";
    /**
     * The Pod inherits the name resolution configuration
     * from the node that the pods run on.
     */
    DnsPolicy["DEFAULT"] = "Default";
    /**
     * It allows a Pod to ignore DNS settings from the Kubernetes environment.
     * All DNS settings are supposed to be provided using the dnsConfig
     * field in the Pod Spec.
     */
    DnsPolicy["NONE"] = "None";
})(DnsPolicy = exports.DnsPolicy || (exports.DnsPolicy = {}));
/**
 * Represents a query that can be performed against nodes with labels.
 */
class NodeLabelQuery {
    constructor(key, operator, values) {
        this.key = key;
        this.operator = operator;
        this.values = values;
    }
    /**
     * Requires value of label `key` to equal `value`.
     */
    static is(key, value) {
        return NodeLabelQuery.in(key, [value]);
    }
    /**
     * Requires value of label `key` to be one of `values`.
     */
    static in(key, values) {
        return new NodeLabelQuery(key, 'In', values);
    }
    /**
     * Requires value of label `key` to be none of `values`.
     */
    static notIn(key, values) {
        return new NodeLabelQuery(key, 'NotIn', values);
    }
    /**
     * Requires label `key` to exist.
     */
    static exists(key) {
        return new NodeLabelQuery(key, 'Exists', undefined);
    }
    /**
     * Requires label `key` to not exist.
     */
    static doesNotExist(key) {
        return new NodeLabelQuery(key, 'DoesNotExist', undefined);
    }
    /**
     * Requires value of label `key` to greater than all elements in `values`.
     */
    static gt(key, values) {
        return new NodeLabelQuery(key, 'Gt', values);
    }
    /**
     * Requires value of label `key` to less than all elements in `values`.
     */
    static lt(key, values) {
        return new NodeLabelQuery(key, 'Lt', values);
    }
}
exports.NodeLabelQuery = NodeLabelQuery;
_f = JSII_RTTI_SYMBOL_1;
NodeLabelQuery[_f] = { fqn: "cdk8s-plus-22.NodeLabelQuery", version: "2.0.0-rc.131" };
/**
 * Represents a query that can be performed against resources with labels.
 */
class LabelExpression {
    constructor(key, operator, values) {
        this.key = key;
        this.operator = operator;
        this.values = values;
    }
    /**
     * Requires value of label `key` to be one of `values`.
     */
    static in(key, values) {
        return new LabelExpression(key, 'In', values);
    }
    /**
     * Requires value of label `key` to be none of `values`.
     */
    static notIn(key, values) {
        return new LabelExpression(key, 'NotIn', values);
    }
    /**
     * Requires label `key` to exist.
     */
    static exists(key) {
        return new LabelExpression(key, 'Exists', undefined);
    }
    /**
     * Requires label `key` to not exist.
     */
    static doesNotExist(key) {
        return new LabelExpression(key, 'DoesNotExist', undefined);
    }
}
exports.LabelExpression = LabelExpression;
_g = JSII_RTTI_SYMBOL_1;
LabelExpression[_g] = { fqn: "cdk8s-plus-22.LabelExpression", version: "2.0.0-rc.131" };
/**
 * Taint effects.
 */
var TaintEffect;
(function (TaintEffect) {
    /**
     * This means that no pod will be able to schedule
     * onto the node unless it has a matching toleration.
     */
    TaintEffect["NO_SCHEDULE"] = "NoSchedule";
    /**
     * This is a "preference" or "soft" version of `NO_SCHEDULE` -- the system
     * will try to avoid placing a pod that does not tolerate the taint on the node,
     * but it is not required
     */
    TaintEffect["PREFER_NO_SCHEDULE"] = "PreferNoSchedule";
    /**
     * This affects pods that are already running on the node as follows:
     *
     * - Pods that do not tolerate the taint are evicted immediately.
     * - Pods that tolerate the taint without specifying `duration` remain bound forever.
     * - Pods that tolerate the taint with a specified `duration` remain bound for
     *   the specified amount of time.
     */
    TaintEffect["NO_EXECUTE"] = "NoExecute";
})(TaintEffect = exports.TaintEffect || (exports.TaintEffect = {}));
/**
 * Taint queries that can be perfomed against nodes.
 */
class NodeTaintQuery {
    constructor(operator, key, value, effect, evictAfter) {
        this.operator = operator;
        this.key = key;
        this.value = value;
        this.effect = effect;
        this.evictAfter = evictAfter;
        if (evictAfter && effect !== TaintEffect.NO_EXECUTE) {
            throw new Error('Only \'NO_EXECUTE\' effects can specify \'evictAfter\'');
        }
    }
    /**
     * Matches a taint with a specific key and value.
     */
    static is(key, value, options = {}) {
        return new NodeTaintQuery('Equal', key, value, options.effect, options.evictAfter);
    }
    /**
     * Matches a tain with any value of a specific key.
     */
    static exists(key, options = {}) {
        return new NodeTaintQuery('Exists', key, undefined, options.effect, options.evictAfter);
    }
    /**
     * Matches any taint.
     */
    static any() {
        return new NodeTaintQuery('Exists');
    }
}
exports.NodeTaintQuery = NodeTaintQuery;
_h = JSII_RTTI_SYMBOL_1;
NodeTaintQuery[_h] = { fqn: "cdk8s-plus-22.NodeTaintQuery", version: "2.0.0-rc.131" };
/**
 * Represents a group of pods.
 */
class Pods extends constructs_1.Construct {
    constructor(scope, id, expressions, labels, namespaces) {
        super(scope, id);
        this.expressions = expressions;
        this.labels = labels;
        this.namespaces = namespaces;
    }
    /**
     * Select pods in the cluster with various selectors.
     */
    static select(scope, id, options) {
        return new Pods(scope, id, options.expressions, options.labels, options.namespaces);
    }
    /**
     * Select all pods.
     */
    static all(scope, id, options = {}) {
        return Pods.select(scope, id, { namespaces: options.namespaces });
    }
    /**
     * @see IPodSelector.toPodSelectorConfig()
     */
    toPodSelectorConfig() {
        return {
            labelSelector: LabelSelector.of({ expressions: this.expressions, labels: this.labels }),
            namespaces: this.namespaces?.toNamespaceSelectorConfig(),
        };
    }
    /**
     * @see INetworkPolicyPeer.toNetworkPolicyPeerConfig()
     */
    toNetworkPolicyPeerConfig() {
        return { podSelector: this.toPodSelectorConfig() };
    }
    /**
     * @see INetworkPolicyPeer.toPodSelector()
     */
    toPodSelector() {
        return this;
    }
}
exports.Pods = Pods;
_j = JSII_RTTI_SYMBOL_1;
Pods[_j] = { fqn: "cdk8s-plus-22.Pods", version: "2.0.0-rc.131" };
/**
 * A node that is matched by label selectors.
 */
class LabeledNode {
    constructor(labelSelector) {
        this.labelSelector = labelSelector;
    }
    ;
}
exports.LabeledNode = LabeledNode;
_k = JSII_RTTI_SYMBOL_1;
LabeledNode[_k] = { fqn: "cdk8s-plus-22.LabeledNode", version: "2.0.0-rc.131" };
/**
 * A node that is matched by taint selectors.
 */
class TaintedNode {
    constructor(taintSelector) {
        this.taintSelector = taintSelector;
    }
    ;
}
exports.TaintedNode = TaintedNode;
_l = JSII_RTTI_SYMBOL_1;
TaintedNode[_l] = { fqn: "cdk8s-plus-22.TaintedNode", version: "2.0.0-rc.131" };
/**
 * A node that is matched by its name.
 */
class NamedNode {
    constructor(name) {
        this.name = name;
    }
    ;
}
exports.NamedNode = NamedNode;
_m = JSII_RTTI_SYMBOL_1;
NamedNode[_m] = { fqn: "cdk8s-plus-22.NamedNode", version: "2.0.0-rc.131" };
/**
 * Represents a node in the cluster.
 */
class Node {
    /**
     * Match a node by its labels.
     */
    static labeled(...labelSelector) {
        return new LabeledNode(labelSelector);
    }
    /**
     * Match a node by its name.
     */
    static named(nodeName) {
        return new NamedNode(nodeName);
    }
    /**
     * Match a node by its taints.
     */
    static tainted(...taintSelector) {
        return new TaintedNode(taintSelector);
    }
}
exports.Node = Node;
_o = JSII_RTTI_SYMBOL_1;
Node[_o] = { fqn: "cdk8s-plus-22.Node", version: "2.0.0-rc.131" };
/**
 * Available topology domains.
 */
class Topology {
    constructor(key) {
        this.key = key;
    }
    /**
     * Custom key for the node label that the system uses to denote the topology domain.
     */
    static custom(key) {
        return new Topology(key);
    }
    ;
}
exports.Topology = Topology;
_p = JSII_RTTI_SYMBOL_1;
Topology[_p] = { fqn: "cdk8s-plus-22.Topology", version: "2.0.0-rc.131" };
/**
 * A hostname represents a single node in the cluster.
 *
 * @see https://kubernetes.io/docs/reference/labels-annotations-taints/#kubernetesiohostname
 */
Topology.HOSTNAME = new Topology('kubernetes.io/hostname');
/**
 * A zone represents a logical failure domain. It is common for Kubernetes clusters to
 * span multiple zones for increased availability. While the exact definition of a zone is
 * left to infrastructure implementations, common properties of a zone include very low
 * network latency within a zone, no-cost network traffic within a zone, and failure
 * independence from other zones. For example, nodes within a zone might share a network
 * switch, but nodes in different zones should not.
 *
 * @see https://kubernetes.io/docs/reference/labels-annotations-taints/#topologykubernetesiozone
 */
Topology.ZONE = new Topology('topology.kubernetes.io/zone');
/**
 * A region represents a larger domain, made up of one or more zones. It is uncommon
 * for Kubernetes clusters to span multiple regions. While the exact definition of a
 * zone or region is left to infrastructure implementations, common properties of a region
 * include higher network latency between them than within them, non-zero cost for network
 * traffic between them, and failure independence from other zones or regions.
 *
 * For example, nodes within a region might share power infrastructure (e.g. a UPS or generator), but
 * nodes in different regions typically would not.
 *
 * @see https://kubernetes.io/docs/reference/labels-annotations-taints/#topologykubernetesioregion
 */
Topology.REGION = new Topology('topology.kubernetes.io/region');
/**
 * Controls the pod scheduling strategy.
 */
class PodScheduling {
    constructor(instance) {
        this.instance = instance;
        this._nodeAffinityPreferred = [];
        this._nodeAffinityRequired = [];
        this._podAffinityPreferred = [];
        this._podAffinityRequired = [];
        this._podAntiAffinityPreferred = [];
        this._podAntiAffinityRequired = [];
        this._tolerations = [];
    }
    /**
     * Assign this pod a specific node by name.
     *
     * The scheduler ignores the Pod, and the kubelet on the named node
     * tries to place the Pod on that node. Overrules any affinity rules of the pod.
     *
     * Some limitations of static assignment are:
     *
     * - If the named node does not exist, the Pod will not run, and in some
     *   cases may be automatically deleted.
     * - If the named node does not have the resources to accommodate the Pod,
     *   the Pod will fail and its reason will indicate why, for example OutOfmemory or OutOfcpu.
     * - Node names in cloud environments are not always predictable or stable.
     *
     * Will throw is the pod is already assigned to named node.
     *
     * Under the hood, this method utilizes the `nodeName` property.
     */
    assign(node) {
        if (this._nodeName) {
            // disallow overriding an static node assignment
            throw new Error(`Cannot assign ${this.instance.podMetadata.name} to node ${node.name}. It is already assigned to node ${this._nodeName}`);
        }
        else {
            this._nodeName = node.name;
        }
    }
    /**
     * Allow this pod to tolerate taints matching these tolerations.
     *
     * You can put multiple taints on the same node and multiple tolerations on the same pod.
     * The way Kubernetes processes multiple taints and tolerations is like a filter: start with
     * all of a node's taints, then ignore the ones for which the pod has a matching toleration;
     * the remaining un-ignored taints have the indicated effects on the pod. In particular:
     *
     * - if there is at least one un-ignored taint with effect NoSchedule then Kubernetes will
     *   not schedule the pod onto that node
     * - if there is no un-ignored taint with effect NoSchedule but there is at least one un-ignored
     *   taint with effect PreferNoSchedule then Kubernetes will try to not schedule the pod onto the node
     * - if there is at least one un-ignored taint with effect NoExecute then the pod will be evicted from
     *   the node (if it is already running on the node), and will not be scheduled onto the node (if it is
     *   not yet running on the node).
     *
     * Under the hood, this method utilizes the `tolerations` property.
     *
     * @see https://kubernetes.io/docs/concepts/scheduling-eviction/taint-and-toleration/
     */
    tolerate(node) {
        for (const query of node.taintSelector) {
            this._tolerations.push({
                key: query.key,
                value: query.value,
                effect: query.effect,
                operator: query.operator,
                tolerationSeconds: query.evictAfter?.toSeconds(),
            });
        }
    }
    /**
     * Attract this pod to a node matched by selectors.
     * You can select a node by using `Node.labeled()`.
     *
     * Attracting to multiple nodes (i.e invoking this method multiple times) acts as
     * an OR condition, meaning the pod will be assigned to either one of the nodes.
     *
     * Under the hood, this method utilizes the `nodeAffinity` property.
     *
     * @see https://kubernetes.io/docs/concepts/scheduling-eviction/assign-pod-node/#node-affinity
     */
    attract(node, options = {}) {
        const term = this.createNodeAffinityTerm(node);
        if (options.weight) {
            this.validateWeight(options.weight);
            this._nodeAffinityPreferred.push({ weight: options.weight, preference: term });
        }
        else {
            this._nodeAffinityRequired.push(term);
        }
    }
    /**
     * Co-locate this pod with a scheduling selection.
     *
     * A selection can be one of:
     *
     * - An instance of a `Pod`.
     * - An instance of a `Workload` (e.g `Deployment`, `StatefulSet`).
     * - An un-managed pod that can be selected via `Pods.select()`.
     *
     * Co-locating with multiple selections ((i.e invoking this method multiple times)) acts as
     * an AND condition. meaning the pod will be assigned to a node that satisfies all
     * selections (i.e runs at least one pod that satisifies each selection).
     *
     * Under the hood, this method utilizes the `podAffinity` property.
     *
     * @see https://kubernetes.io/docs/concepts/scheduling-eviction/assign-pod-node/#inter-pod-affinity-and-anti-affinity
     */
    colocate(selector, options = {}) {
        const topology = options.topology ?? Topology.HOSTNAME;
        const term = this.createPodAffinityTerm(topology, selector);
        if (options.weight) {
            this.validateWeight(options.weight);
            this._podAffinityPreferred.push({ weight: options.weight, podAffinityTerm: term });
        }
        else {
            this._podAffinityRequired.push(term);
        }
    }
    /**
     * Seperate this pod from a scheduling selection.
     *
     * A selection can be one of:
     *
     * - An instance of a `Pod`.
     * - An instance of a `Workload` (e.g `Deployment`, `StatefulSet`).
     * - An un-managed pod that can be selected via `Pods.select()`.
     *
     * Seperating from multiple selections acts as an AND condition. meaning the pod
     * will not be assigned to a node that satisfies all selections (i.e runs at least one pod that satisifies each selection).
     *
     * Under the hood, this method utilizes the `podAntiAffinity` property.
     *
     * @see https://kubernetes.io/docs/concepts/scheduling-eviction/assign-pod-node/#inter-pod-affinity-and-anti-affinity
     */
    separate(selector, options = {}) {
        const topology = options.topology ?? Topology.HOSTNAME;
        const term = this.createPodAffinityTerm(topology, selector);
        if (options.weight) {
            this.validateWeight(options.weight);
            this._podAntiAffinityPreferred.push({ weight: options.weight, podAffinityTerm: term });
        }
        else {
            this._podAntiAffinityRequired.push(term);
        }
    }
    createPodAffinityTerm(topology, selector) {
        const config = selector.toPodSelectorConfig();
        return {
            topologyKey: topology.key,
            labelSelector: config.labelSelector._toKube(),
            namespaceSelector: config.namespaces?.labelSelector?._toKube(),
            namespaces: config.namespaces?.names,
        };
    }
    createNodeAffinityTerm(node) {
        return { matchExpressions: node.labelSelector.map(s => ({ key: s.key, operator: s.operator, values: s.values })) };
    }
    validateWeight(weight) {
        if (weight < 1 || weight > 100) {
            // https://kubernetes.io/docs/concepts/scheduling-eviction/assign-pod-node/#node-affinity-weight
            throw new Error(`Invalid affinity weight: ${weight}. Must be in range 1-100`);
        }
    }
    /**
     * @internal
     */
    _toKube() {
        const atLeastOne = (...arrays) => {
            return arrays.flat().length > 0;
        };
        const hasNodeAffinity = atLeastOne(this._nodeAffinityPreferred, this._nodeAffinityRequired);
        const hasPodAffinity = atLeastOne(this._podAffinityPreferred, this._podAffinityRequired);
        const hasPodAntiAffinty = atLeastOne(this._podAntiAffinityPreferred, this._podAntiAffinityRequired);
        const hasAffinity = hasNodeAffinity || hasPodAffinity || hasPodAntiAffinty;
        return {
            affinity: hasAffinity ? {
                nodeAffinity: hasNodeAffinity ? {
                    preferredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._nodeAffinityPreferred),
                    requiredDuringSchedulingIgnoredDuringExecution: this._nodeAffinityRequired.length > 0 ? {
                        nodeSelectorTerms: this._nodeAffinityRequired,
                    } : undefined,
                } : undefined,
                podAffinity: hasPodAffinity ? {
                    preferredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._podAffinityPreferred),
                    requiredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._podAffinityRequired),
                } : undefined,
                podAntiAffinity: hasPodAntiAffinty ? {
                    preferredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._podAntiAffinityPreferred),
                    requiredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._podAntiAffinityRequired),
                } : undefined,
            } : undefined,
            nodeName: this._nodeName,
            tolerations: utils_1.undefinedIfEmpty(this._tolerations),
        };
    }
}
exports.PodScheduling = PodScheduling;
_q = JSII_RTTI_SYMBOL_1;
PodScheduling[_q] = { fqn: "cdk8s-plus-22.PodScheduling", version: "2.0.0-rc.131" };
/**
 * Isolation determines which policies are created
 * when allowing connections from a a pod / workload to peers.
 */
var PodConnectionsIsolation;
(function (PodConnectionsIsolation) {
    /**
     * Only creates network policies that select the pod.
     */
    PodConnectionsIsolation["POD"] = "POD";
    /**
     * Only creates network policies that select the peer.
     */
    PodConnectionsIsolation["PEER"] = "PEER";
})(PodConnectionsIsolation = exports.PodConnectionsIsolation || (exports.PodConnectionsIsolation = {}));
/**
 * Controls network isolation rules for inter-pod communication.
 */
class PodConnections {
    constructor(instance) {
        this.instance = instance;
    }
    /**
     * Allow network traffic from this pod to the peer.
     *
     * By default, this will create an egress network policy for this pod, and an ingress
     * network policy for the peer. This is required if both sides are already isolated.
     * Use `options.isolation` to control this behavior.
     *
     * @example
     *
     * // create only an egress policy that selects the 'web' pod to allow outgoing traffic
     * // to the 'redis' pod. this requires the 'redis' pod to not be isolated for ingress.
     * web.connections.allowTo(redis, { isolation: Isolation.POD })
     *
     * // create only an ingress policy that selects the 'redis' peer to allow incoming traffic
     * // from the 'web' pod. this requires the 'web' pod to not be isolated for egress.
     * web.connections.allowTo(redis, { isolation: Isolation.PEER })
     *
     */
    allowTo(peer, options = {}) {
        return this.allow('Egress', peer, { ports: this.extractPorts(peer), ...options });
    }
    /**
     * Allow network traffic from the peer to this pod.
     *
     * By default, this will create an ingress network policy for this pod, and an egress
     * network policy for the peer. This is required if both sides are already isolated.
     * Use `options.isolation` to control this behavior.
     *
     * @example
     *
     * // create only an egress policy that selects the 'web' pod to allow outgoing traffic
     * // to the 'redis' pod. this requires the 'redis' pod to not be isolated for ingress.
     * redis.connections.allowFrom(web, { isolation: Isolation.PEER })
     *
     * // create only an ingress policy that selects the 'redis' peer to allow incoming traffic
     * // from the 'web' pod. this requires the 'web' pod to not be isolated for egress.
     * redis.connections.allowFrom(web, { isolation: Isolation.POD })
     *
     */
    allowFrom(peer, options = {}) {
        return this.allow('Ingress', peer, { ports: this.extractPorts(this.instance), ...options });
    }
    allow(direction, peer, options = {}) {
        const config = peer.toNetworkPolicyPeerConfig();
        networkpolicy.validatePeerConfig(config);
        const peerAddress = utils_1.address(peer);
        if (!options.isolation || options.isolation === PodConnectionsIsolation.POD) {
            const src = new networkpolicy.NetworkPolicy(this.instance, `Allow${direction}${peerAddress}`, {
                selector: this.instance,
                // the policy must be defined in the namespace of the pod
                // so it can select it.
                metadata: { namespace: this.instance.metadata.namespace },
            });
            switch (direction) {
                case 'Egress':
                    src.addEgressRule(peer, options.ports);
                    break;
                case 'Ingress':
                    src.addIngressRule(peer, options.ports);
            }
        }
        if (!options.isolation || options.isolation === PodConnectionsIsolation.PEER) {
            if (config.ipBlock) {
                // for an ip block we don't need to create the opposite policies
                return;
            }
            const podSelector = peer.toPodSelector();
            if (!podSelector) {
                throw new Error(`Unable to create policies for peer '${peer.node.addr}' since its not a pod selector`);
            }
            const oppositeDirection = direction === 'Egress' ? 'Ingress' : 'Egress';
            const podSelectorConfig = podSelector.toPodSelectorConfig();
            let namespaces;
            if (!podSelectorConfig.namespaces) {
                // if the peer doesn't specify namespaces, we assume the same namespace.
                namespaces = [this.instance.metadata.namespace];
            }
            else {
                // a peer cannot specify namespaces by labels because
                // we won't be able to extract the names of those namespaces.
                if (podSelectorConfig.namespaces.labelSelector && !podSelectorConfig.namespaces.labelSelector.isEmpty()) {
                    throw new Error(`Unable to create an ${oppositeDirection} policy for peer '${peer.node.path}' (pod=${this.instance.name}). Peer must specify namespaces only by name`);
                }
                // a peer must specify namespaces by name.
                if (!podSelectorConfig.namespaces.names) {
                    throw new Error(`Unable to create an ${oppositeDirection} policy for peer '${peer.node.path}' (pod=${this.instance.name}). Peer must specify namespace names`);
                }
                namespaces = podSelectorConfig.namespaces.names;
            }
            for (const name of namespaces) {
                switch (direction) {
                    case 'Egress':
                        new networkpolicy.NetworkPolicy(this.instance, `AllowIngress${name}${peerAddress}`, {
                            selector: podSelector,
                            metadata: { namespace: name },
                            ingress: { rules: [{ peer: this.instance, ports: options.ports }] },
                        });
                        break;
                    case 'Ingress':
                        new networkpolicy.NetworkPolicy(this.instance, `AllowEgress${name}${peerAddress}`, {
                            selector: podSelector,
                            metadata: { namespace: name },
                            egress: { rules: [{ peer: this.instance, ports: options.ports }] },
                        });
                        break;
                    default:
                        throw new Error(`Unsupported direction: ${direction}`);
                }
            }
        }
    }
    extractPorts(selector) {
        return container.extractContainerPorts(selector).map(n => networkpolicy.NetworkPolicyPort.tcp(n.number));
    }
}
exports.PodConnections = PodConnections;
_r = JSII_RTTI_SYMBOL_1;
PodConnections[_r] = { fqn: "cdk8s-plus-22.PodConnections", version: "2.0.0-rc.131" };
//# sourceMappingURL=data:application/json;base64,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