"use strict";
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
Object.defineProperty(exports, "__esModule", { value: true });
const javascript_1 = require("projen/lib/javascript");
const python_1 = require("projen/lib/python");
const typescript_1 = require("projen/lib/typescript");
const synth_1 = require("projen/lib/util/synth");
const src_1 = require("../src");
describe("NX Monorepo Unit Tests", () => {
    it("Empty Monorepo", () => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "mainline",
            name: "Defaults",
        });
        expect(synth_1.synthSnapshot(project)).toMatchSnapshot();
    });
    it("Ignore Patterns", () => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "mainline",
            name: "IgnorePatterns",
            nxConfig: {
                nxIgnore: ["pattern1.txt", "**/foo.ts"],
            },
        });
        expect(synth_1.synthSnapshot(project)).toMatchSnapshot();
    });
    it("Target Dependencies", () => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "mainline",
            name: "TargetDependencies",
            nxConfig: {
                targetDependencies: {
                    test: [
                        { target: "test", projects: src_1.TargetDependencyProject.DEPENDENCIES },
                    ],
                    eslint: [
                        { target: "eslint", projects: src_1.TargetDependencyProject.SELF },
                    ],
                },
            },
        });
        expect(synth_1.synthSnapshot(project)).toMatchSnapshot();
    });
    it("Affected Branch", () => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "main",
            name: "AffectedBranch",
            nxConfig: {
                affectedBranch: "main",
            },
        });
        expect(synth_1.synthSnapshot(project)).toMatchSnapshot();
    });
    it.each([javascript_1.NodePackageManager.PNPM, javascript_1.NodePackageManager.YARN])("Additional Workspace Packages", (packageManager) => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "mainline",
            packageManager,
            name: "AdditionalWorkspacePackages",
            workspaceConfig: {
                additionalPackages: ["my/custom/package"],
            },
        });
        new typescript_1.TypeScriptProject({
            name: "ts-subproject",
            outdir: "packages/ts-subproject",
            parent: project,
            packageManager,
            defaultReleaseBranch: "mainline",
        });
        project.addWorkspacePackages("another/custom/package", "yet/another/package");
        expect(synth_1.synthSnapshot(project)).toMatchSnapshot();
    });
    it("PNPM", () => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "mainline",
            name: "PNPM",
            packageManager: javascript_1.NodePackageManager.PNPM,
        });
        new typescript_1.TypeScriptProject({
            name: "ts-subproject",
            outdir: "packages/ts-subproject",
            parent: project,
            packageManager: javascript_1.NodePackageManager.PNPM,
            defaultReleaseBranch: "mainline",
        });
        expect(synth_1.synthSnapshot(project)).toMatchSnapshot();
    });
    it("Validate consistent Package Managers", () => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "mainline",
            name: "PNPM",
            packageManager: javascript_1.NodePackageManager.PNPM,
        });
        new typescript_1.TypeScriptProject({
            name: "ts-subproject",
            outdir: "packages/ts-subproject",
            parent: project,
            defaultReleaseBranch: "mainline",
        });
        expect(() => synth_1.synthSnapshot(project)).toThrow("ts-subproject packageManager does not match the monorepo packageManager: pnpm.");
    });
    it("Composite", () => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "mainline",
            name: "Composite",
        });
        const tsProject = new typescript_1.TypeScriptProject({
            name: "ts-subproject",
            outdir: "packages/ts-subproject",
            parent: project,
            defaultReleaseBranch: "mainline",
        });
        new typescript_1.TypeScriptProject({
            name: "ts-subproject2",
            outdir: "packages/ts-subproject2",
            parent: project,
            defaultReleaseBranch: "mainline",
            devDeps: ["ts-subproject"],
        });
        const pyProject = new python_1.PythonProject({
            parent: project,
            outdir: "packages/py-subproject",
            authorEmail: "test@test.com",
            authorName: "test",
            moduleName: "py_subproject",
            name: "py-subproject",
            version: "0.0.0",
        });
        project.addImplicitDependency(tsProject, pyProject);
        expect(synth_1.synthSnapshot(project)).toMatchSnapshot();
    });
});
//# sourceMappingURL=data:application/json;base64,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