"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MountPropagation = exports.Container = exports.ImagePullPolicy = exports.EnvValue = exports.ResourceFieldPaths = exports.EnvFieldPaths = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const k8s = require("./imports/k8s");
var EnvFieldPaths;
(function (EnvFieldPaths) {
    /**
     * The name of the pod.
     */
    EnvFieldPaths["POD_NAME"] = "metadata.name";
    /**
     * The namespace of the pod.
     */
    EnvFieldPaths["POD_NAMESPACE"] = "metadata.namespace";
    /**
     * The uid of the pod.
     */
    EnvFieldPaths["POD_UID"] = "metadata.uid";
    /**
     * The labels of the pod.
     */
    EnvFieldPaths["POD_LABEL"] = "metadata.labels";
    /**
     * The annotations of the pod.
     */
    EnvFieldPaths["POD_ANNOTATION"] = "metadata.annotations";
    /**
     * The ipAddress of the pod.
     */
    EnvFieldPaths["POD_IP"] = "status.podIP";
    /**
     * The service account name of the pod.
     */
    EnvFieldPaths["SERVICE_ACCOUNT_NAME"] = "spec.serviceAccountName";
    /**
     * The name of the node.
     */
    EnvFieldPaths["NODE_NAME"] = "spec.nodeName";
    /**
     * The ipAddress of the node.
     */
    EnvFieldPaths["NODE_IP"] = "status.hostIP";
    /**
     * The ipAddresess of the pod.
     */
    EnvFieldPaths["POD_IPS"] = "status.podIPs";
})(EnvFieldPaths = exports.EnvFieldPaths || (exports.EnvFieldPaths = {}));
var ResourceFieldPaths;
(function (ResourceFieldPaths) {
    /**
     * CPU limit of the container.
     */
    ResourceFieldPaths["CPU_LIMIT"] = "limits.cpu";
    /**
     * Memory limit of the container.
     */
    ResourceFieldPaths["MEMORY_LIMIT"] = "limits.memory";
    /**
     * CPU request of the container.
     */
    ResourceFieldPaths["CPU_REQUEST"] = "requests.cpu";
    /**
     * Memory request of the container.
     */
    ResourceFieldPaths["MEMORY_REQUEST"] = "requests.memory";
    /**
     * Ephemeral storage limit of the container.
     */
    ResourceFieldPaths["STORAGE_LIMIT"] = "limits.ephemeral-storage";
    /**
     * Ephemeral storage request of the container.
     */
    ResourceFieldPaths["STORAGE_REQUEST"] = "requests.ephemeral-storage";
})(ResourceFieldPaths = exports.ResourceFieldPaths || (exports.ResourceFieldPaths = {}));
/**
 * Utility class for creating reading env values from various sources.
 */
class EnvValue {
    constructor(value, valueFrom) {
        this.value = value;
        this.valueFrom = valueFrom;
    }
    /**
     * Create a value by reading a specific key inside a config map.
     *
     * @param configMap - The config map.
     * @param key - The key to extract the value from.
     * @param options - Additional options.
     */
    static fromConfigMap(configMap, key, options = {}) {
        const source = {
            configMapKeyRef: {
                name: configMap.name,
                key,
                optional: options.optional,
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Defines an environment value from a secret JSON value.
     *
     * @param secretValue The secret value (secrent + key)
     * @param options Additional options
     */
    static fromSecretValue(secretValue, options = {}) {
        const source = {
            secretKeyRef: {
                name: secretValue.secret.name,
                key: secretValue.key,
                optional: options.optional,
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a value from the given argument.
     *
     * @param value - The value.
     */
    static fromValue(value) {
        return new EnvValue(value);
    }
    /**
     *
     * Create a value from a field reference.
     *
     * @param fieldPath: The field reference.
     * @param options: Additional options.
     */
    static fromFieldRef(fieldPath, options = {}) {
        let needsKey = false;
        if (fieldPath === EnvFieldPaths.POD_LABEL || fieldPath === EnvFieldPaths.POD_ANNOTATION) {
            needsKey = true;
        }
        if (needsKey && options.key === undefined) {
            throw new Error(`${fieldPath} requires a key`);
        }
        const source = {
            fieldRef: {
                fieldPath: (`${fieldPath}` + (needsKey ? `['${options.key}']` : '')),
                ...(options.apiVersion ? { apiVersion: options.apiVersion } : {}),
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a value from a resource.
     *
     * @param resource: Resource to select the value from.
     * @param options: Additional options.
     */
    static fromResource(resource, options = {}) {
        const source = {
            resourceFieldRef: {
                resource: `${resource}`,
                ...(options.divisor ? { divisor: k8s.IntOrString.fromString(options.divisor) } : {}),
                ...(options.container ? { containerName: options.container.name } : {}),
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a value from a key in the current process environment.
     *
     * @param key - The key to read.
     * @param options - Additional options.
     */
    static fromProcess(key, options = {}) {
        const value = process.env[key];
        if (options.required && !value) {
            throw new Error(`Missing ${key} env variable`);
        }
        return EnvValue.fromValue(value);
    }
}
exports.EnvValue = EnvValue;
_a = JSII_RTTI_SYMBOL_1;
EnvValue[_a] = { fqn: "cdk8s-plus-22.EnvValue", version: "1.0.0-beta.136" };
var ImagePullPolicy;
(function (ImagePullPolicy) {
    /**
     * Every time the kubelet launches a container, the kubelet queries the container image registry
     * to resolve the name to an image digest. If the kubelet has a container image with that exact
     * digest cached locally, the kubelet uses its cached image; otherwise, the kubelet downloads
     * (pulls) the image with the resolved digest, and uses that image to launch the container.
     *
     * Default is Always if ImagePullPolicy is omitted and either the image tag is :latest or
     * the image tag is omitted.
     */
    ImagePullPolicy["ALWAYS"] = "Always";
    /**
     * The image is pulled only if it is not already present locally.
     *
     * Default is IfNotPresent if ImagePullPolicy is omitted and the image tag is present but
     * not :latest
     */
    ImagePullPolicy["IF_NOT_PRESENT"] = "IfNotPresent";
    /**
     * The image is assumed to exist locally. No attempt is made to pull the image.
     */
    ImagePullPolicy["NEVER"] = "Never";
})(ImagePullPolicy = exports.ImagePullPolicy || (exports.ImagePullPolicy = {}));
/**
 * A single application container that you want to run within a pod.
 */
class Container {
    constructor(props) {
        var _c, _d, _e, _f;
        if (props instanceof Container) {
            throw new Error('Attempted to construct a container from a Container object.');
        }
        this.name = (_c = props.name) !== null && _c !== void 0 ? _c : 'main';
        this.image = props.image;
        this.port = props.port;
        this._command = props.command;
        this._args = props.args;
        this._env = (_d = props.env) !== null && _d !== void 0 ? _d : {};
        this._readiness = props.readiness;
        this._liveness = props.liveness;
        this._startup = props.startup;
        this.workingDir = props.workingDir;
        this.mounts = (_e = props.volumeMounts) !== null && _e !== void 0 ? _e : [];
        this.imagePullPolicy = (_f = props.imagePullPolicy) !== null && _f !== void 0 ? _f : ImagePullPolicy.ALWAYS;
    }
    /**
     * Entrypoint array (the command to execute when the container starts).
     * @returns a copy of the entrypoint array, cannot be modified
     */
    get command() {
        return this._command ? [...this._command] : undefined;
    }
    /**
     * Arguments to the entrypoint.
     *
     * @returns a copy of the arguments array, cannot be modified.
     */
    get args() {
        return this._args ? [...this._args] : undefined;
    }
    /**
     * Add an environment value to the container. The variable value can come
     * from various dynamic sources such a secrets of config maps.
     *
     * @see EnvValue.fromXXX
     *
     * @param name - The variable name.
     * @param value - The variable value.
     */
    addEnv(name, value) {
        this._env[name] = value;
    }
    /**
     * The environment variables for this container.
     *
     * Returns a copy. To add environment variables use `addEnv()`.
     */
    get env() {
        return { ...this._env };
    }
    /**
     * Mount a volume to a specific path so that it is accessible by the container.
     * Every pod that is configured to use this container will autmoatically have access to the volume.
     *
     * @param path - The desired path in the container.
     * @param volume - The volume to mount.
     */
    mount(path, volume, options = {}) {
        this.mounts.push({ path, volume, ...options });
    }
    /**
     * @internal
     */
    _toKube() {
        var _c, _d, _e;
        const volumeMounts = [];
        for (const mount of this.mounts) {
            volumeMounts.push({
                name: mount.volume.name,
                mountPath: mount.path,
                readOnly: mount.readOnly,
                mountPropagation: mount.propagation,
                subPath: mount.subPath,
                subPathExpr: mount.subPathExpr,
            });
        }
        const ports = new Array();
        if (this.port) {
            ports.push({
                containerPort: this.port,
            });
        }
        return {
            name: this.name,
            image: this.image,
            imagePullPolicy: this.imagePullPolicy,
            ports,
            volumeMounts,
            command: this.command,
            args: this.args,
            workingDir: this.workingDir,
            env: renderEnv(this._env),
            readinessProbe: (_c = this._readiness) === null || _c === void 0 ? void 0 : _c._toKube(this),
            livenessProbe: (_d = this._liveness) === null || _d === void 0 ? void 0 : _d._toKube(this),
            startupProbe: (_e = this._startup) === null || _e === void 0 ? void 0 : _e._toKube(this),
        };
    }
}
exports.Container = Container;
_b = JSII_RTTI_SYMBOL_1;
Container[_b] = { fqn: "cdk8s-plus-22.Container", version: "1.0.0-beta.136" };
var MountPropagation;
(function (MountPropagation) {
    /**
     * This volume mount will not receive any subsequent mounts that are mounted
     * to this volume or any of its subdirectories by the host. In similar
     * fashion, no mounts created by the Container will be visible on the host.
     *
     * This is the default mode.
     *
     * This mode is equal to `private` mount propagation as described in the Linux
     * kernel documentation
     */
    MountPropagation["NONE"] = "None";
    /**
     * This volume mount will receive all subsequent mounts that are mounted to
     * this volume or any of its subdirectories.
     *
     * In other words, if the host mounts anything inside the volume mount, the
     * Container will see it mounted there.
     *
     * Similarly, if any Pod with Bidirectional mount propagation to the same
     * volume mounts anything there, the Container with HostToContainer mount
     * propagation will see it.
     *
     * This mode is equal to `rslave` mount propagation as described in the Linux
     * kernel documentation
     */
    MountPropagation["HOST_TO_CONTAINER"] = "HostToContainer";
    /**
     * This volume mount behaves the same the HostToContainer mount. In addition,
     * all volume mounts created by the Container will be propagated back to the
     * host and to all Containers of all Pods that use the same volume
     *
     * A typical use case for this mode is a Pod with a FlexVolume or CSI driver
     * or a Pod that needs to mount something on the host using a hostPath volume.
     *
     * This mode is equal to `rshared` mount propagation as described in the Linux
     * kernel documentation
     *
     * Caution: Bidirectional mount propagation can be dangerous. It can damage
     * the host operating system and therefore it is allowed only in privileged
     * Containers. Familiarity with Linux kernel behavior is strongly recommended.
     * In addition, any volume mounts created by Containers in Pods must be
     * destroyed (unmounted) by the Containers on termination.
     *
     */
    MountPropagation["BIDIRECTIONAL"] = "Bidirectional";
})(MountPropagation = exports.MountPropagation || (exports.MountPropagation = {}));
function renderEnv(env) {
    const result = new Array();
    for (const [name, v] of Object.entries(env)) {
        result.push({
            name,
            value: v.value,
            valueFrom: v.valueFrom,
        });
    }
    return result;
}
//# sourceMappingURL=data:application/json;base64,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