import { useCallback, useRef, useState } from 'react';
import { useHistory } from 'react-router';
import { Button } from '../../../components';
import { Modal } from '../../../components/Modal/Modal';
import { Space } from '../../../components/Space/Space';
import { useAPI } from '../../../providers/ApiProvider';
import { ProjectProvider, useProject } from '../../../providers/ProjectProvider';
import { useFixedLocation } from '../../../providers/RoutesProvider';
import { Elem } from '../../../utils/bem';
import { useRefresh } from '../../../utils/hooks';
import { ImportPage } from './Import';
import { useImportPage } from './useImportPage';

export const Inner = () => {
  const history = useHistory();
  const location = useFixedLocation();
  const modal = useRef();
  const refresh = useRefresh();
  const { project } = useProject();
  const [waiting, setWaitingStatus] = useState(false);
  const api = useAPI();

  const { uploading, uploadDisabled, finishUpload, fileIds, pageProps } = useImportPage(project);

  const backToDM = useCallback(() => {
    const path = location.pathname.replace(ImportModal.path, '');
    const search = location.search;
    const pathname = `${path}${search !== '?' ? search : ''}`;

    return refresh(pathname);
  }, [location, history]);

  const onCancel = useCallback(async () => {
    setWaitingStatus(true);
    await api.callApi('deleteFileUploads', {
      params: {
        pk: project.id,
      },
      body: {
        file_upload_ids: fileIds,
      },
    });
    setWaitingStatus(false);
    modal?.current?.hide();
    backToDM();
  }, [modal, project, fileIds, backToDM]);

  const onFinish = useCallback(async () => {
    const imported = await finishUpload();

    if (!imported) return;
    backToDM();
  }, [backToDM, finishUpload]);

  return (
    <Modal
      title="Import data"
      ref={modal}
      onHide={() => backToDM()}
      closeOnClickOutside={false}
      fullscreen
      visible
      bare
    >
      <Modal.Header divided>
        <Elem block="modal" name="title">Import Data</Elem>

        <Space>
          <Button waiting={waiting} onClick={onCancel}>Cancel</Button>
          <Button look="primary" onClick={onFinish} waiting={waiting || uploading} disabled={uploadDisabled}>
            Import
          </Button>
        </Space>
      </Modal.Header>
      <ImportPage project={project} {...pageProps} />
    </Modal>
  );
};
export const ImportModal = () => {
  return (
    <ProjectProvider>
      <Inner />
    </ProjectProvider>
  );
};

ImportModal.path = "/import";
ImportModal.modal = true;
