"use strict";
const assert_1 = require("@aws-cdk/assert");
const aws_dynamodb_1 = require("@aws-cdk/aws-dynamodb");
const aws_dynamodb_2 = require("@aws-cdk/aws-dynamodb");
const core_1 = require("@aws-cdk/core");
const assert = require("assert");
const lib_1 = require("../lib");
// tslint:disable:object-literal-key-quotes
// CDK parameters
const CONSTRUCT_NAME = 'aws-cdk-dynamodb-global';
// DynamoDB table parameters
const TABLE_NAME = 'GlobalTable';
const TABLE_PARTITION_KEY = { name: 'hashKey', type: aws_dynamodb_1.AttributeType.STRING };
const STACK_PROPS = {
    partitionKey: TABLE_PARTITION_KEY,
    tableName: TABLE_NAME,
    regions: ['us-east-1', 'us-east-2', 'us-west-2']
};
module.exports = {
    'Default Global DynamoDB stack': {
        'global dynamo'(test) {
            const stack = new core_1.Stack();
            new lib_1.GlobalTable(stack, CONSTRUCT_NAME, STACK_PROPS);
            const topStack = stack.node.findChild(CONSTRUCT_NAME);
            for (const reg of STACK_PROPS.regions) {
                const tableStack = topStack.node.findChild(CONSTRUCT_NAME + '-' + reg);
                assert_1.expect(tableStack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
                    "KeySchema": [
                        {
                            "AttributeName": "hashKey",
                            "KeyType": "HASH"
                        }
                    ],
                    "AttributeDefinitions": [
                        {
                            "AttributeName": "hashKey",
                            "AttributeType": "S"
                        }
                    ],
                    "StreamSpecification": {
                        "StreamViewType": "NEW_AND_OLD_IMAGES"
                    },
                    "TableName": "GlobalTable"
                }));
            }
            const customResourceStack = stack.node.findChild(CONSTRUCT_NAME + "-CustomResource");
            assert_1.expect(customResourceStack).to(assert_1.haveResource('AWS::Lambda::Function', {
                Description: "Lambda to make DynamoDB a global table",
                Handler: "index.handler",
                Timeout: 300
            }));
            assert_1.expect(customResourceStack).to(assert_1.haveResource('AWS::CloudFormation::CustomResource', {
                Regions: STACK_PROPS.regions,
                ResourceType: "Custom::DynamoGlobalTableCoordinator",
                TableName: TABLE_NAME,
            }));
            test.done();
        },
    },
    'Enforce StreamSpecification': {
        'global dynamo should only allow NEW_AND_OLD_IMAGES'(test) {
            const stack = new core_1.Stack();
            try {
                new lib_1.GlobalTable(stack, CONSTRUCT_NAME, {
                    tableName: TABLE_NAME,
                    stream: aws_dynamodb_1.StreamViewType.KEYS_ONLY,
                    partitionKey: TABLE_PARTITION_KEY,
                    regions: ['us-east-1', 'us-east-2', 'us-west-2']
                });
                // We are expecting the above line to throw a TypeError since
                // the streamSpecification is wrong.  Force a failure on this
                // line if we get there.
                assert_1.expect(stack).to(assert_1.haveResource('Fail::this::test::IfWeGetThisFar', {}));
            }
            catch (TypeError) {
                assert_1.expect(stack);
            }
            test.done();
        },
    },
    'Check getting tables': {
        'global dynamo should only allow NEW_AND_OLD_IMAGES'(test) {
            const stack = new core_1.Stack();
            const regTables = new lib_1.GlobalTable(stack, CONSTRUCT_NAME, {
                tableName: TABLE_NAME,
                partitionKey: TABLE_PARTITION_KEY,
                regions: ['us-east-1', 'us-east-2', 'us-west-2']
            });
            assert(regTables.regionalTables.length === 3);
            for (const table of regTables.regionalTables) {
                assert(table instanceof aws_dynamodb_2.Table);
            }
            test.done();
        },
    }
};
//# sourceMappingURL=data:application/json;base64,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