"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const dynamodb = require("@aws-cdk/aws-dynamodb");
const cdk = require("@aws-cdk/core");
const global_table_coordinator_1 = require("./global-table-coordinator");
/**
 * This class works by deploying an AWS DynamoDB table into each region specified in  GlobalTableProps.regions[],
 * then triggering a CloudFormation Custom Resource Lambda to link them all together to create linked AWS Global DynamoDB tables.
 */
class GlobalTable extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * Creates dynamoDB tables across regions that will be able to be globbed together into a global table
         */
        this._regionalTables = new Array();
        this._regionalTables = [];
        if (props.stream != null && props.stream !== dynamodb.StreamViewType.NEW_AND_OLD_IMAGES) {
            throw new Error("dynamoProps.stream MUST be set to dynamodb.StreamViewType.NEW_AND_OLD_IMAGES");
        }
        // need to set this stream specification, otherwise global tables don't work
        // And no way to set a default value in an interface
        const regionalTableProps = {
            ...props,
            removalPolicy: props.removalPolicy,
            stream: dynamodb.StreamViewType.NEW_AND_OLD_IMAGES,
        };
        // here we loop through the configured regions.
        // in each region we'll deploy a separate stack with a DynamoDB Table with identical properties in the individual stacks
        for (const reg of props.regions) {
            const regionalStack = new cdk.Stack(this, id + "-" + reg, { env: { region: reg } });
            const regionalTable = new dynamodb.Table(regionalStack, `${id}-GlobalTable-${reg}`, regionalTableProps);
            this._regionalTables.push(regionalTable);
        }
        this.lambdaGlobalTableCoordinator = new global_table_coordinator_1.GlobalTableCoordinator(scope, id + "-CustomResource", props);
        for (const table of this._regionalTables) {
            this.lambdaGlobalTableCoordinator.node.addDependency(table);
        }
    }
    /**
     * Obtain tables deployed in other each region
     */
    get regionalTables() {
        return this._regionalTables.map(x => x);
    }
}
exports.GlobalTable = GlobalTable;
//# sourceMappingURL=data:application/json;base64,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