import { ApiObjectMetadata, ApiObjectMetadataDefinition } from 'cdk8s';
import { Construct } from 'constructs';
import { AbstractPod, AbstractPodProps } from './pod';
/**
 * Properties for `Workload`.
 */
export interface WorkloadProps extends AbstractPodProps {
    /**
     * The pod metadata of this workload.
     */
    readonly podMetadata?: ApiObjectMetadata;
    /**
     * Automatically allocates a pod label selector for this workload and add
     * it to the pod metadata. This ensures this workload manages pods created by
     * its pod template.
     *
     * @default true
     */
    readonly select?: boolean;
}
/**
 * Possible operators.
 */
export declare enum LabelSelectorRequirementOperator {
    /**
     * In.
     */
    IN = "In",
    /**
     * NotIn.
     */
    NOT_IN = "NotIn",
    /**
     * Exists.
     */
    EXISTS = "Exists",
    /**
     * DoesNotExist.
     */
    DOES_NOT_EXIST = "DoesNotExist"
}
/**
 * A label selector requirement is a selector that contains values, a key, and an operator that
 * relates the key and values.
 */
export interface LabelSelectorRequirement {
    /**
     * The label key that the selector applies to.
     */
    readonly key: string;
    /**
     * Represents a key's relationship to a set of values.
     */
    readonly operator: LabelSelectorRequirementOperator;
    /**
     * An array of string values. If the operator is In or NotIn, the values array
     * must be non-empty. If the operator is Exists or DoesNotExist,
     * the values array must be empty. This array is replaced during a strategic merge patch.
     */
    readonly values?: string[];
}
/**
 * A workload is an application running on Kubernetes. Whether your workload is a single
 * component or several that work together, on Kubernetes you run it inside a set of pods.
 * In Kubernetes, a Pod represents a set of running containers on your cluster.
 */
export declare abstract class Workload extends AbstractPod {
    /**
     * The metadata of pods in this workload.
     */
    readonly podMetadata: ApiObjectMetadataDefinition;
    private readonly _matchLabels;
    private readonly _matchExpressions;
    constructor(scope: Construct, id: string, props?: WorkloadProps);
    /**
     * Configure selectors for this workload.
     */
    select(...selectors: LabelSelector[]): void;
    /**
     * The label matchers this workload will use in order to select pods.
     *
     * Returns a a copy. Use `select()` to add label matchers.
     */
    get matchLabels(): Record<string, string>;
    /**
     * The expression matchers this workload will use in order to select pods.
     *
     * Returns a a copy. Use `select()` to add expression matchers.
     */
    get matchExpressions(): LabelSelectorRequirement[];
}
/**
 * A label selector is a label query over a set of resources.
 *
 * @see https://kubernetes.io/docs/concepts/overview/working-with-objects/labels/#label-selectors
 */
export declare class LabelSelector {
    readonly applyToTemplate: boolean;
    readonly key: string;
    readonly values?: string[] | undefined;
    readonly operator?: LabelSelectorRequirementOperator | undefined;
    /**
     * Creates a `matchLabels` entry from the key and value.
     * Use `applyToTemplate` to also add this label to the pod metadata of the workload.
     */
    static is(key: string, value: string, applyToTemplate?: boolean): LabelSelector;
    /**
     * Creates a `matchExpressions` "In" entry.
     */
    static in(key: string, values: string[]): LabelSelector;
    /**
     * Creates a `matchExpressions` "NotIn" entry.
     */
    static notIn(key: string, values: string[]): LabelSelector;
    /**
     * Creates a `matchExpressions` "Exists" entry.
     */
    static exists(key: string): LabelSelector;
    /**
     * Creates a `matchExpressions` "DoesNotExist" entry.
     */
    static doesNotExist(key: string): LabelSelector;
    private constructor();
}
