"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Protocol = exports.Service = exports.ServiceType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const base = require("./base");
const k8s = require("./imports/k8s");
const ingress = require("./ingress");
/**
 * For some parts of your application (for example, frontends) you may want to expose a Service onto an
 * external IP address, that's outside of your cluster.
 * Kubernetes ServiceTypes allow you to specify what kind of Service you want.
 * The default is ClusterIP.
 */
var ServiceType;
(function (ServiceType) {
    /**
     * Exposes the Service on a cluster-internal IP.
     * Choosing this value makes the Service only reachable from within the cluster.
     * This is the default ServiceType
     */
    ServiceType["CLUSTER_IP"] = "ClusterIP";
    /**
     * Exposes the Service on each Node's IP at a static port (the NodePort).
     * A ClusterIP Service, to which the NodePort Service routes, is automatically created.
     * You'll be able to contact the NodePort Service, from outside the cluster,
     * by requesting <NodeIP>:<NodePort>.
     */
    ServiceType["NODE_PORT"] = "NodePort";
    /**
     * Exposes the Service externally using a cloud provider's load balancer.
     * NodePort and ClusterIP Services, to which the external load balancer routes,
     * are automatically created.
     */
    ServiceType["LOAD_BALANCER"] = "LoadBalancer";
    /**
     * Maps the Service to the contents of the externalName field (e.g. foo.bar.example.com), by returning a CNAME record with its value.
     * No proxying of any kind is set up.
     *
     * > Note: You need either kube-dns version 1.7 or CoreDNS version 0.0.8 or higher to use the ExternalName type.
     */
    ServiceType["EXTERNAL_NAME"] = "ExternalName";
})(ServiceType = exports.ServiceType || (exports.ServiceType = {}));
/**
 * An abstract way to expose an application running on a set of Pods as a network service.
 * With Kubernetes you don't need to modify your application to use an unfamiliar service discovery mechanism.
 * Kubernetes gives Pods their own IP addresses and a single DNS name for a set of Pods, and can load-balance across them.
 *
 * For example, consider a stateless image-processing backend which is running with 3 replicas. Those replicas are fungible—frontends do not care which backend they use.
 * While the actual Pods that compose the backend set may change, the frontend clients should not need to be aware of that,
 * nor should they need to keep track of the set of backends themselves.
 * The Service abstraction enables this decoupling.
 *
 * If you're able to use Kubernetes APIs for service discovery in your application, you can query the API server for Endpoints,
 * that get updated whenever the set of Pods in a Service changes. For non-native applications, Kubernetes offers ways to place a network port
 * or load balancer in between your application and the backend Pods.
 */
class Service extends base.Resource {
    constructor(scope, id, props = {}) {
        var _b, _c, _d;
        super(scope, id);
        this.apiObject = new k8s.KubeService(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
        this.clusterIP = props.clusterIP;
        this.externalName = props.externalName;
        if (props.externalName !== undefined) {
            this.type = ServiceType.EXTERNAL_NAME;
        }
        else {
            this.type = (_b = props.type) !== null && _b !== void 0 ? _b : ServiceType.CLUSTER_IP;
        }
        this._externalIPs = (_c = props.externalIPs) !== null && _c !== void 0 ? _c : [];
        this._ports = [];
        this._selector = {};
        this._loadBalancerSourceRanges = props.loadBalancerSourceRanges;
        for (const portAndOptions of (_d = props.ports) !== null && _d !== void 0 ? _d : []) {
            this.serve(portAndOptions.port, portAndOptions);
        }
    }
    /**
     * Expose a service via an ingress using the specified path.
     *
     * @param path The path to expose the service under.
     * @param options Additional options.
     *
     * @returns The `Ingress` resource that was used.
     */
    exposeViaIngress(path, options = {}) {
        var _b;
        const ingr = (_b = options.ingress) !== null && _b !== void 0 ? _b : new ingress.Ingress(this, 'Ingress');
        ingr.addRule(path, ingress.IngressBackend.fromService(this), options.pathType);
        return ingr;
    }
    /**
     * Returns the labels which are used to select pods for this service.
     */
    get selector() {
        return this._selector;
    }
    /**
     * Ports for this service.
     *
     * Use `serve()` to expose additional service ports.
     */
    get ports() {
        return [...this._ports];
    }
    /**
     * Associate a deployment to this service.
     *
     * If not targetPort is specific in the portOptions, then requests will be routed
     * to the port exposed by the first container in the deployment's pods.
     * The deployment's `labelSelector` will be used to select pods.
     *
     * @param depl The deployment to expose
     * @param options Optional settings for the port.
     */
    addDeployment(depl, options = {}) {
        var _b, _c;
        const containers = depl.containers;
        if (containers.length === 0) {
            throw new Error('Cannot expose a deployment without containers');
        }
        // just a PoC, we assume the first container is the main one.
        // TODO: figure out what the correct thing to do here.
        const container = containers[0];
        const port = (_b = options.port) !== null && _b !== void 0 ? _b : container.port;
        const targetPort = (_c = options.targetPort) !== null && _c !== void 0 ? _c : containers[0].port;
        if (!port) {
            throw new Error('Cannot determine port. Either pass `port` in options or configure a port on the first container of the deployment');
        }
        const selector = Object.entries(depl.matchLabels);
        if (selector.length === 0) {
            throw new Error('deployment does not have a label selector');
        }
        if (Object.keys(this.selector).length > 0) {
            throw new Error('a selector is already defined for this service. cannot add a deployment');
        }
        for (const [k, v] of selector) {
            this.addSelector(k, v);
        }
        this.serve(port, { ...options, targetPort });
    }
    /**
     * Services defined using this spec will select pods according the provided label.
     *
     * @param label The label key.
     * @param value The label value.
     */
    addSelector(label, value) {
        this._selector[label] = value;
    }
    /**
     * Configure a port the service will bind to.
     * This method can be called multiple times.
     *
     * @param port The port definition.
     */
    serve(port, options = {}) {
        this._ports.push({ ...options, port });
    }
    /**
     * @internal
     */
    _toKube() {
        if (this._ports.length === 0 && this.type !== ServiceType.EXTERNAL_NAME) {
            throw new Error('A service must be configured with a port');
        }
        if (this.type === ServiceType.EXTERNAL_NAME && this.externalName === undefined) {
            throw new Error('A service with type EXTERNAL_NAME requires an externalName prop');
        }
        const ports = [];
        for (const port of this._ports) {
            ports.push({
                name: port.name,
                port: port.port,
                targetPort: port.targetPort ? k8s.IntOrString.fromNumber(port.targetPort) : undefined,
                nodePort: port.nodePort,
                protocol: port.protocol,
            });
        }
        return this.type !== ServiceType.EXTERNAL_NAME ? {
            clusterIp: this.clusterIP,
            externalIPs: this._externalIPs,
            externalName: this.externalName,
            type: this.type,
            selector: this._selector,
            ports: ports,
            loadBalancerSourceRanges: this._loadBalancerSourceRanges,
        } : {
            type: this.type,
            externalName: this.externalName,
        };
    }
}
exports.Service = Service;
_a = JSII_RTTI_SYMBOL_1;
Service[_a] = { fqn: "cdk8s-plus-22.Service", version: "1.0.0-beta.216" };
var Protocol;
(function (Protocol) {
    Protocol["TCP"] = "TCP";
    Protocol["UDP"] = "UDP";
    Protocol["SCTP"] = "SCTP";
})(Protocol = exports.Protocol || (exports.Protocol = {}));
//# sourceMappingURL=data:application/json;base64,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