"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GCEPersistentDiskPersistentVolume = exports.AzureDiskPersistentVolume = exports.AwsElasticBlockStorePersistentVolume = exports.PersistentVolumeReclaimPolicy = exports.PersistentVolume = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const base = require("./base");
const k8s = require("./imports/k8s");
const pvc = require("./pvc");
const volume = require("./volume");
/**
 * A PersistentVolume (PV) is a piece of storage in the cluster that has been
 * provisioned by an administrator or dynamically provisioned using Storage Classes.
 * It is a resource in the cluster just like a node is a cluster resource.
 * PVs are volume plugins like Volumes, but have a lifecycle independent of any
 * individual Pod that uses the PV. This API object captures the details of the
 * implementation of the storage, be that NFS, iSCSI, or a
 * cloud-provider-specific storage system.
 */
class PersistentVolume extends base.Resource {
    constructor(scope, id, props = {}) {
        var _e, _f;
        super(scope, id);
        this.storage = props.storage;
        this.mode = (_e = props.volumeMode) !== null && _e !== void 0 ? _e : pvc.PersistentVolumeMode.FILE_SYSTEM;
        this.storageClassName = props.storageClassName;
        this._accessModes = props.accessModes;
        this.mountOptions = props.mountOptions;
        this.reclaimPolicy = (_f = props.reclaimPolicy) !== null && _f !== void 0 ? _f : PersistentVolumeReclaimPolicy.RETAIN;
        if (props.claim) {
            this.bind(props.claim);
        }
        this.apiObject = new k8s.KubePersistentVolume(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
    }
    /**
     * Imports a pv from the cluster as a reference.
     * @param volumeName The name of the pv to reference.
     */
    static fromPersistentVolumeName(volumeName) {
        return { name: volumeName };
    }
    /**
     * Access modes requirement of this claim.
     */
    get accessModes() {
        return this._accessModes ? [...this._accessModes] : undefined;
    }
    /**
     * PVC this volume is bound to. Undefined means this volume is not yet
     * claimed by any PVC.
     */
    get claim() {
        return this._claim;
    }
    /**
     * Reserve a `PersistentVolume` by creating a `PersistentVolumeClaim`
     * that is wired to claim this volume.
     *
     * Note that this method will throw in case the volume is already claimed.
     *
     * @see https://kubernetes.io/docs/concepts/storage/persistent-volumes/#reserving-a-persistentvolume
     */
    reserve() {
        const claim = new pvc.PersistentVolumeClaim(this, `${this.name}PVC`, {
            metadata: { name: `pvc-${this.name}`, namespace: this.metadata.namespace },
            // the storage classes must match, otherwise the claim
            // will use the default class (or no class at all), which may be different than the class
            // of this volume. note that other requirements are not needed since
            // when they are not defined, any volume satisifies them.
            storageClassName: this.storageClassName,
        });
        this.bind(claim);
        claim.bind(this);
        return claim;
    }
    /**
     * Bind a volume to a specific claim.
     * Note that you must also bind the claim to the volume.
     *
     * @see https://kubernetes.io/docs/concepts/storage/persistent-volumes/#binding
     *
     * @param claim The PVC to bind to.
     */
    bind(claim) {
        if (this._claim && this._claim.name !== claim.name) {
            throw new Error(`Cannot bind volume '${this.name}' to claim '${claim.name}' since it is already bound to claim '${this._claim.name}'`);
        }
        this._claim = claim;
    }
    asVolume() {
        const claim = this.reserve();
        return volume.Volume.fromPersistentVolumeClaim(claim);
    }
    /**
     * @internal
     */
    _toKube() {
        var _e, _f, _g;
        const storage = this.storage ? k8s.Quantity.fromString(this.storage.toGibibytes() + 'Gi') : undefined;
        return {
            claimRef: this._claim ? { name: (_e = this._claim) === null || _e === void 0 ? void 0 : _e.name } : undefined,
            accessModes: (_f = this.accessModes) === null || _f === void 0 ? void 0 : _f.map(a => a.toString()),
            capacity: storage ? { storage } : undefined,
            mountOptions: (_g = this.mountOptions) === null || _g === void 0 ? void 0 : _g.map(o => o),
            storageClassName: this.storageClassName,
            persistentVolumeReclaimPolicy: this.reclaimPolicy,
            volumeMode: this.mode,
        };
    }
}
exports.PersistentVolume = PersistentVolume;
_a = JSII_RTTI_SYMBOL_1;
PersistentVolume[_a] = { fqn: "cdk8s-plus-22.PersistentVolume", version: "1.0.0-beta.216" };
/**
 * Reclaim Policies.
 */
var PersistentVolumeReclaimPolicy;
(function (PersistentVolumeReclaimPolicy) {
    /**
     * The Retain reclaim policy allows for manual reclamation of the resource.
     * When the PersistentVolumeClaim is deleted, the PersistentVolume still exists and the
     * volume is considered "released". But it is not yet available for another claim
     * because the previous claimant's data remains on the volume.
     * An administrator can manually reclaim the volume with the following steps:
     *
     * 1. Delete the PersistentVolume. The associated storage asset in external
     *    infrastructure (such as an AWS EBS, GCE PD, Azure Disk, or Cinder volume) still exists after the PV is deleted.
     * 2. Manually clean up the data on the associated storage asset accordingly.
     * 3. Manually delete the associated storage asset.
     *
     * If you want to reuse the same storage asset, create a new PersistentVolume
     * with the same storage asset definition.
     */
    PersistentVolumeReclaimPolicy["RETAIN"] = "Retain";
    /**
     * For volume plugins that support the Delete reclaim policy, deletion removes both the
     * PersistentVolume object from Kubernetes, as well as the associated storage asset in
     * the external infrastructure, such as an AWS EBS, GCE PD, Azure Disk, or Cinder volume.
     * Volumes that were dynamically provisioned inherit the reclaim policy of their StorageClass, which defaults to Delete.
     * The administrator should configure the StorageClass according to users' expectations; otherwise,
     * the PV must be edited or patched after it is created
     */
    PersistentVolumeReclaimPolicy["DELETE"] = "Delete";
})(PersistentVolumeReclaimPolicy = exports.PersistentVolumeReclaimPolicy || (exports.PersistentVolumeReclaimPolicy = {}));
/**
 * Represents an AWS Disk resource that is attached to a kubelet's host machine and
 * then exposed to the pod.
 *
 * @see https://kubernetes.io/docs/concepts/storage/volumes#awselasticblockstore
 */
class AwsElasticBlockStorePersistentVolume extends PersistentVolume {
    constructor(scope, id, props) {
        var _e, _f;
        super(scope, id, props);
        this.volumeId = props.volumeId;
        this.readOnly = (_e = props.readOnly) !== null && _e !== void 0 ? _e : false;
        this.fsType = (_f = props.fsType) !== null && _f !== void 0 ? _f : 'ext4';
        this.partition = props.partition;
    }
    /**
     * @internal
     */
    _toKube() {
        const spec = super._toKube();
        return {
            ...spec,
            awsElasticBlockStore: {
                volumeId: this.volumeId,
                fsType: this.fsType,
                partition: this.partition,
                readOnly: this.readOnly,
            },
        };
    }
}
exports.AwsElasticBlockStorePersistentVolume = AwsElasticBlockStorePersistentVolume;
_b = JSII_RTTI_SYMBOL_1;
AwsElasticBlockStorePersistentVolume[_b] = { fqn: "cdk8s-plus-22.AwsElasticBlockStorePersistentVolume", version: "1.0.0-beta.216" };
/**
 * AzureDisk represents an Azure Data Disk mount on the host and bind mount to the pod.
 */
class AzureDiskPersistentVolume extends PersistentVolume {
    constructor(scope, id, props) {
        var _e, _f, _g, _h;
        super(scope, id, props);
        this.diskName = props.diskName;
        this.diskUri = props.diskUri;
        this.cachingMode = (_e = props.cachingMode) !== null && _e !== void 0 ? _e : volume.AzureDiskPersistentVolumeCachingMode.NONE;
        this.fsType = (_f = props.fsType) !== null && _f !== void 0 ? _f : 'ext4';
        this.kind = (_g = props.kind) !== null && _g !== void 0 ? _g : volume.AzureDiskPersistentVolumeKind.SHARED;
        this.readOnly = (_h = props.readOnly) !== null && _h !== void 0 ? _h : false;
    }
    /**
     * @internal
     *
     * @see https://github.com/kubernetes/examples/blob/master/staging/volumes/azure_disk/README.md
     */
    _toKube() {
        const spec = super._toKube();
        return {
            ...spec,
            azureDisk: {
                diskName: this.diskName,
                diskUri: this.diskUri,
                cachingMode: this.cachingMode,
                fsType: this.fsType,
                kind: this.kind,
                readOnly: this.readOnly,
            },
        };
    }
}
exports.AzureDiskPersistentVolume = AzureDiskPersistentVolume;
_c = JSII_RTTI_SYMBOL_1;
AzureDiskPersistentVolume[_c] = { fqn: "cdk8s-plus-22.AzureDiskPersistentVolume", version: "1.0.0-beta.216" };
/**
 * GCEPersistentDisk represents a GCE Disk resource that is attached to a kubelet's host machine
 * and then exposed to the pod. Provisioned by an admin.
 *
 * @see https://kubernetes.io/docs/concepts/storage/volumes#gcepersistentdisk
 */
class GCEPersistentDiskPersistentVolume extends PersistentVolume {
    constructor(scope, id, props) {
        var _e, _f;
        super(scope, id, props);
        this.pdName = props.pdName;
        this.readOnly = (_e = props.readOnly) !== null && _e !== void 0 ? _e : false;
        this.fsType = (_f = props.fsType) !== null && _f !== void 0 ? _f : 'ext4';
        this.partition = props.partition;
    }
    /**
     * @internal
     *
     * @see https://github.com/kubernetes/examples/blob/master/staging/volumes/azure_disk/README.md
     */
    _toKube() {
        const spec = super._toKube();
        return {
            ...spec,
            gcePersistentDisk: {
                pdName: this.pdName,
                fsType: this.fsType,
                partition: this.partition,
                readOnly: this.readOnly,
            },
        };
    }
}
exports.GCEPersistentDiskPersistentVolume = GCEPersistentDiskPersistentVolume;
_d = JSII_RTTI_SYMBOL_1;
GCEPersistentDiskPersistentVolume[_d] = { fqn: "cdk8s-plus-22.GCEPersistentDiskPersistentVolume", version: "1.0.0-beta.216" };
//# sourceMappingURL=data:application/json;base64,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