"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.IngressBackend = exports.Ingress = exports.HttpIngressPathType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const base = require("./base");
const k8s = require("./imports/k8s");
/**
 * Specify how the path is matched against request paths.
 *
 * @see https://kubernetes.io/docs/concepts/services-networking/ingress/#path-types
 */
var HttpIngressPathType;
(function (HttpIngressPathType) {
    /**
     * Matches the URL path exactly.
     */
    HttpIngressPathType["PREFIX"] = "Prefix";
    /**
     * Matches based on a URL path prefix split by '/'.
     */
    HttpIngressPathType["EXACT"] = "Exact";
    /**
     * Matching is specified by the underlying IngressClass.
     */
    HttpIngressPathType["IMPLEMENTATION_SPECIFIC"] = "ImplementationSpecific";
})(HttpIngressPathType = exports.HttpIngressPathType || (exports.HttpIngressPathType = {}));
/**
 * Ingress is a collection of rules that allow inbound connections to reach the
 * endpoints defined by a backend. An Ingress can be configured to give services
 * externally-reachable urls, load balance traffic, terminate SSL, offer name
 * based virtual hosting etc.
 */
class Ingress extends base.Resource {
    constructor(scope, id, props = {}) {
        var _c;
        super(scope, id);
        this._rulesPerHost = {};
        this._tlsConfig = [];
        this.apiObject = new k8s.KubeIngress(this, 'Resource', {
            metadata: props.metadata,
            spec: {
                defaultBackend: cdk8s_1.Lazy.any({ produce: () => { var _c; return (_c = this._defaultBackend) === null || _c === void 0 ? void 0 : _c._toKube(); } }),
                rules: cdk8s_1.Lazy.any({ produce: () => this.synthRules() }),
                tls: cdk8s_1.Lazy.any({ produce: () => this.tlsConfig() }),
            },
        });
        if (props.defaultBackend) {
            this.addDefaultBackend(props.defaultBackend);
        }
        this.addRules(...(_c = props.rules) !== null && _c !== void 0 ? _c : []);
        if (props.tls) {
            this.addTls(props.tls);
        }
    }
    onValidate() {
        if (!this._defaultBackend && Object.keys(this._rulesPerHost).length === 0) {
            return ['ingress with no rules or default backend'];
        }
        return [];
    }
    /**
     * Defines the default backend for this ingress. A default backend capable of
     * servicing requests that don't match any rule.
     *
     * @param backend The backend to use for requests that do not match any rule.
     */
    addDefaultBackend(backend) {
        this.addRules({ backend });
    }
    /**
     * Specify a default backend for a specific host name. This backend will be used as a catch-all for requests
     * targeted to this host name (the `Host` header matches this value).
     *
     * @param host The host name to match
     * @param backend The backend to route to
     */
    addHostDefaultBackend(host, backend) {
        if (!host) {
            throw new Error('host must not be an empty string');
        }
        this.addRules({ host, backend });
    }
    /**
     * Adds an ingress rule applied to requests to a specific host and a specific
     * HTTP path (the `Host` header matches this value).
     *
     * @param host The host name
     * @param path The HTTP path
     * @param backend The backend to route requests to
     * @param pathType How the path is matched against request paths
     */
    addHostRule(host, path, backend, pathType) {
        if (!host) {
            throw new Error('host must not be an empty string');
        }
        this.addRules({ host, backend, path, pathType });
    }
    /**
     * Adds an ingress rule applied to requests sent to a specific HTTP path.
     *
     * @param path The HTTP path
     * @param backend The backend to route requests to
     * @param pathType How the path is matched against request paths
     */
    addRule(path, backend, pathType) {
        this.addRules({ backend, path, pathType });
    }
    /**
     * Adds rules to this ingress.
     * @param rules The rules to add
     */
    addRules(...rules) {
        var _c, _d, _e, _f;
        for (const rule of rules) {
            // default backend is not really a rule
            if (!rule.host && !rule.path) {
                if (this._defaultBackend) {
                    throw new Error('a default backend is already defined for this ingress');
                }
                this._defaultBackend = rule.backend;
                continue;
            }
            const host = (_c = rule.host) !== null && _c !== void 0 ? _c : '';
            const backend = rule.backend;
            const path = (_d = rule.path) !== null && _d !== void 0 ? _d : '/';
            const pathType = (_e = rule.pathType) !== null && _e !== void 0 ? _e : HttpIngressPathType.PREFIX;
            if (path && !path.startsWith('/')) {
                throw new Error(`ingress paths must begin with a "/": ${path}`);
            }
            const routes = this._rulesPerHost[host] = (_f = this._rulesPerHost[host]) !== null && _f !== void 0 ? _f : [];
            // check if we already have a rule for this host/path
            if (routes.find(r => r.path === path)) {
                throw new Error(`there is already an ingress rule for ${host}${path}`);
            }
            routes.push({
                backend: backend._toKube(),
                path,
                pathType,
            });
        }
    }
    synthRules() {
        const rules = new Array();
        for (const [host, paths] of Object.entries(this._rulesPerHost)) {
            rules.push({
                host: host ? host : undefined,
                http: { paths: paths.sort(sortByPath) },
            });
        }
        return rules.length > 0 ? rules : undefined;
    }
    addTls(tls) {
        this._tlsConfig.push(...tls);
    }
    tlsConfig() {
        var _c;
        if (this._tlsConfig.length == 0) {
            return undefined;
        }
        const tls = new Array();
        for (const entry of this._tlsConfig) {
            tls.push({
                hosts: entry.hosts,
                secretName: (_c = entry.secret) === null || _c === void 0 ? void 0 : _c.name,
            });
        }
        return tls;
    }
}
exports.Ingress = Ingress;
_a = JSII_RTTI_SYMBOL_1;
Ingress[_a] = { fqn: "cdk8s-plus-22.Ingress", version: "1.0.0-beta.216" };
/**
 * The backend for an ingress path.
 */
class IngressBackend {
    constructor(backend) {
        this.backend = backend;
    }
    /**
     * A Kubernetes `Service` to use as the backend for this path.
     * @param serv The service object.
     */
    static fromService(serv, options = {}) {
        if (serv.ports.length === 0) {
            throw new Error('service does not expose any ports');
        }
        let servicePort;
        if (serv.ports.length === 1) {
            servicePort = serv.ports[0].port;
        }
        else {
            if (options.port !== undefined) {
                const found = serv.ports.find(p => p.port === options.port);
                if (found) {
                    servicePort = found.port;
                }
                else {
                    throw new Error(`service exposes ports ${serv.ports.map(p => p.port).join(',')} but backend is defined to use port ${options.port}`);
                }
            }
            else {
                throw new Error(`unable to determine service port since service exposes multiple ports: ${serv.ports.map(x => x.port).join(',')}`);
            }
        }
        if (options.port !== undefined && servicePort !== options.port) {
            throw new Error(`backend defines port ${options.port} but service exposes port ${servicePort}`);
        }
        return new IngressBackend({
            service: {
                name: serv.name,
                port: { number: servicePort },
            },
        });
    }
    /**
     * @internal
     */
    _toKube() { return this.backend; }
}
exports.IngressBackend = IngressBackend;
_b = JSII_RTTI_SYMBOL_1;
IngressBackend[_b] = { fqn: "cdk8s-plus-22.IngressBackend", version: "1.0.0-beta.216" };
function sortByPath(lhs, rhs) {
    var _c, _d;
    const p1 = (_c = lhs.path) !== null && _c !== void 0 ? _c : '';
    const p2 = (_d = rhs.path) !== null && _d !== void 0 ? _d : '';
    return p1.localeCompare(p2);
}
//# sourceMappingURL=data:application/json;base64,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