"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DeploymentStrategy = exports.PercentOrAbsolute = exports.Deployment = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const k8s = require("./imports/k8s");
const service = require("./service");
const workload = require("./workload");
/**
*
* A Deployment provides declarative updates for Pods and ReplicaSets.
*
* You describe a desired state in a Deployment, and the Deployment Controller changes the actual
* state to the desired state at a controlled rate. You can define Deployments to create new ReplicaSets, or to remove
* existing Deployments and adopt all their resources with new Deployments.
*
* > Note: Do not manage ReplicaSets owned by a Deployment. Consider opening an issue in the main Kubernetes repository if your use case is not covered below.
*
* Use Case
* ---------
*
* The following are typical use cases for Deployments:
*
* - Create a Deployment to rollout a ReplicaSet. The ReplicaSet creates Pods in the background.
*   Check the status of the rollout to see if it succeeds or not.
* - Declare the new state of the Pods by updating the PodTemplateSpec of the Deployment.
*   A new ReplicaSet is created and the Deployment manages moving the Pods from the old ReplicaSet to the new one at a controlled rate.
*   Each new ReplicaSet updates the revision of the Deployment.
* - Rollback to an earlier Deployment revision if the current state of the Deployment is not stable.
*   Each rollback updates the revision of the Deployment.
* - Scale up the Deployment to facilitate more load.
* - Pause the Deployment to apply multiple fixes to its PodTemplateSpec and then resume it to start a new rollout.
* - Use the status of the Deployment as an indicator that a rollout has stuck.
* - Clean up older ReplicaSets that you don't need anymore.
*
**/
class Deployment extends workload.Workload {
    constructor(scope, id, props = {}) {
        var _d, _e, _f, _g;
        super(scope, id, props);
        this.apiObject = new k8s.KubeDeployment(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
        this.minReady = (_d = props.minReady) !== null && _d !== void 0 ? _d : cdk8s_1.Duration.seconds(0);
        this.progressDeadline = (_e = props.progressDeadline) !== null && _e !== void 0 ? _e : cdk8s_1.Duration.seconds(600);
        if (this.progressDeadline.toSeconds() <= this.minReady.toSeconds()) {
            throw new Error(`'progressDeadline' (${this.progressDeadline.toSeconds()}s) must be greater than 'minReady' (${this.minReady.toSeconds()}s)`);
        }
        this.replicas = (_f = props.replicas) !== null && _f !== void 0 ? _f : 1;
        this.strategy = (_g = props.strategy) !== null && _g !== void 0 ? _g : DeploymentStrategy.rollingUpdate();
    }
    /**
     * Expose a deployment via a service.
     *
     * This is equivalent to running `kubectl expose deployment <deployment-name>`.
     *
     * @param options Options to determine details of the service and port exposed.
     */
    exposeViaService(options = {}) {
        var _d;
        const ser = new service.Service(this, 'Service', {
            metadata: options.name ? { name: options.name } : undefined,
            type: (_d = options.serviceType) !== null && _d !== void 0 ? _d : service.ServiceType.CLUSTER_IP,
        });
        ser.addDeployment(this, { protocol: options.protocol, targetPort: options.targetPort, port: options.port });
        return ser;
    }
    /**
     * Expose a deployment via an ingress.
     *
     * This will first expose the deployment with a service, and then expose the service via an ingress.
     *
     * @param path The ingress path to register under.
     * @param options Additional options.
     */
    exposeViaIngress(path, options = {}) {
        const ser = this.exposeViaService(options);
        return ser.exposeViaIngress(path, options);
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            replicas: this.replicas,
            minReadySeconds: this.minReady.toSeconds(),
            progressDeadlineSeconds: this.progressDeadline.toSeconds(),
            template: {
                metadata: this.podMetadata.toJson(),
                spec: this._toPodSpec(),
            },
            selector: {
                matchExpressions: this.matchExpressions,
                matchLabels: this.matchLabels,
            },
            strategy: this.strategy._toKube(),
        };
    }
}
exports.Deployment = Deployment;
_a = JSII_RTTI_SYMBOL_1;
Deployment[_a] = { fqn: "cdk8s-plus-22.Deployment", version: "1.0.0-beta.216" };
/**
 * Union like class repsenting either a ration in
 * percents or an absolute number.
 */
class PercentOrAbsolute {
    constructor(value) {
        this.value = value;
    }
    /**
     * Percent ratio.
     */
    static percent(percent) {
        return new PercentOrAbsolute(`${percent}%`);
    }
    /**
     * Absolute number.
     */
    static absolute(num) {
        return new PercentOrAbsolute(num);
    }
    isZero() {
        return this.value === PercentOrAbsolute.absolute(0).value || this.value === PercentOrAbsolute.percent(0).value;
    }
}
exports.PercentOrAbsolute = PercentOrAbsolute;
_b = JSII_RTTI_SYMBOL_1;
PercentOrAbsolute[_b] = { fqn: "cdk8s-plus-22.PercentOrAbsolute", version: "1.0.0-beta.216" };
/**
 * Deployment strategies.
 */
class DeploymentStrategy {
    constructor(strategy) {
        this.strategy = strategy;
    }
    /**
     * All existing Pods are killed before new ones are created.
     *
     * @see https://kubernetes.io/docs/concepts/workloads/controllers/deployment/#recreate-deployment
     */
    static recreate() {
        return new DeploymentStrategy({
            type: 'Recreate',
        });
    }
    static rollingUpdate(options = {}) {
        var _d, _e;
        const maxSurge = (_d = options.maxSurge) !== null && _d !== void 0 ? _d : PercentOrAbsolute.percent(25);
        const maxUnavailable = (_e = options.maxSurge) !== null && _e !== void 0 ? _e : PercentOrAbsolute.percent(25);
        if (maxSurge.isZero() && maxUnavailable.isZero()) {
            throw new Error('\'maxSurge\' and \'maxUnavailable\' cannot be both zero');
        }
        return new DeploymentStrategy({
            type: 'RollingUpdate',
            rollingUpdate: { maxSurge, maxUnavailable },
        });
    }
    /**
     * @internal
     */
    _toKube() {
        return this.strategy;
    }
}
exports.DeploymentStrategy = DeploymentStrategy;
_c = JSII_RTTI_SYMBOL_1;
DeploymentStrategy[_c] = { fqn: "cdk8s-plus-22.DeploymentStrategy", version: "1.0.0-beta.216" };
//# sourceMappingURL=data:application/json;base64,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