"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NxMonorepoProject = exports.TargetDependencyProject = exports.SampleLanguage = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
const console = require("console");
const fs = require("fs");
const path = require("path");
const projen_1 = require("projen");
const typescript_1 = require("projen/lib/typescript");
const NX_MONOREPO_PLUGIN_PATH = ".nx/plugins/nx-monorepo-plugin.js";
/**
 * Supported languages to generate sample code.
 */
var SampleLanguage;
(function (SampleLanguage) {
    SampleLanguage["TYPESCRIPT"] = "ts";
    SampleLanguage["PYTHON"] = "py";
    SampleLanguage["JAVA"] = "java";
})(SampleLanguage = exports.SampleLanguage || (exports.SampleLanguage = {}));
/**
 * Supported enums for a TargetDependency.
 */
var TargetDependencyProject;
(function (TargetDependencyProject) {
    /**
     * Only rely on the package where the target is called.
     *
     * This is usually done for test like targets where you only want to run unit
     * tests on the target packages without testing all dependent packages.
     */
    TargetDependencyProject["SELF"] = "self";
    /**
     * Target relies on executing the target against all dependencies first.
     *
     * This is usually done for build like targets where you want to build all
     * dependant projects first.
     */
    TargetDependencyProject["DEPENDENCIES"] = "dependencies";
})(TargetDependencyProject = exports.TargetDependencyProject || (exports.TargetDependencyProject = {}));
/**
 * This project type will bootstrap a NX based monorepo with support for polygot
 * builds, build caching, dependency graph visualization and much more.
 *
 * @pjid nx-monorepo
 */
class NxMonorepoProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        super({
            ...options,
            github: false,
            jest: false,
            package: false,
            prettier: true,
            projenrcTs: true,
            release: false,
            sampleCode: false,
            defaultReleaseBranch: "mainline",
        });
        this.implicitDependencies = {};
        // Never publish a monorepo root package.
        this.package.addField("private", true);
        // No need to compile or test a monorepo root package.
        this.compileTask.reset();
        this.testTask.reset();
        this.addDevDeps("@nrwl/cli", "@nrwl/workspace");
        new projen_1.IgnoreFile(this, ".nxignore").exclude("test-reports", "target", ".env", ".pytest_cache", ...(options.nxIgnorePatterns || []));
        new projen_1.TextFile(this, NX_MONOREPO_PLUGIN_PATH, {
            readonly: true,
            lines: fs
                .readFileSync("./node_modules/aws-prototyping-sdk/lib/pdk_projen/nx_monorepo/plugin/nx-monorepo-plugin.js")
                .toString("utf-8")
                .split("\n"),
        });
        new projen_1.JsonFile(this, "nx.json", {
            obj: {
                extends: "@nrwl/workspace/presets/npm.json",
                plugins: [`./${NX_MONOREPO_PLUGIN_PATH}`],
                npmScope: "monorepo",
                tasksRunnerOptions: {
                    default: {
                        runner: "@nrwl/workspace/tasks-runners/default",
                        options: {
                            useDaemonProcess: false,
                            cacheableOperations: ["build", "test"],
                        },
                    },
                },
                implicitDependencies: this.implicitDependencies,
                targetDependencies: {
                    build: [
                        {
                            target: "build",
                            projects: "dependencies",
                        },
                    ],
                    ...(options.targetDependencies || {}),
                },
                affected: {
                    defaultBase: "mainline",
                },
            },
        });
        if (options.sampleLanguage) {
            if (fs.existsSync(path.join(this.outdir, ".projenrc.ts.bk"))) {
                console.log("Ignoring generation of sample code as this is a destructive action and should only be set on initial synthesis.");
            }
            else {
                fs.copyFileSync(path.join(this.outdir, ".projenrc.ts"), ".projenrc.ts.bk"); // Make a backup of the existing .projenrc.ts just in case
                fs.copyFileSync(path.join(this.outdir, `node_modules/aws-prototyping-sdk/samples/sample-nx-monorepo/src/nx-monorepo-sample-${options.sampleLanguage}.ts`), ".projenrc.ts");
                if (options.sampleContextJson) {
                    new projen_1.JsonFile(this, ".cdk.context.json", {
                        obj: JSON.parse(options.sampleContextJson),
                        readonly: false,
                        marker: false,
                    });
                }
            }
        }
    }
    /**
     * Create an implicit dependency between two Project's. This is typically
     * used in polygot repos where a Typescript project wants a build dependency
     * on a Python project as an example.
     *
     * @param dependent project you want to have the dependency.
     * @param dependee project you wish to depend on.
     */
    addImplicitDependency(dependent, dependee) {
        if (this.implicitDependencies[dependent.name]) {
            this.implicitDependencies[dependent.name].push(dependee.name);
        }
        else {
            this.implicitDependencies[dependent.name] = [dependee.name];
        }
    }
    // Remove this hack once subProjects is made public in Projen
    get subProjects() {
        // @ts-ignore
        const subProjects = this.subprojects || [];
        return subProjects.sort((a, b) => a.name.localeCompare(b.name));
    }
    /**
     * @inheritDoc
     */
    preSynthesize() {
        super.preSynthesize();
        // Add workspaces for each subproject
        this.package.addField("workspaces", this.subProjects.map((subProject) => path.relative(this.outdir, subProject.outdir)));
    }
    /**
     * @inheritDoc
     */
    synth() {
        // Check to see if a new subProject was added
        const newSubProject = this.subProjects.find((subProject) => !fs.existsSync(subProject.outdir));
        // Need to synth before generating the package.json otherwise the subdirectory won't exist
        newSubProject && super.synth();
        const bootstrapCdkJsonPath = path.join(this.outdir, ".cdk.context.json");
        const infraSubproject = this.subProjects.find((s) => s.name === "infra");
        if (fs.existsSync(bootstrapCdkJsonPath) && infraSubproject) {
            fs.copyFileSync(bootstrapCdkJsonPath, path.join(infraSubproject.outdir, "cdk.context.json"));
            fs.unlinkSync(bootstrapCdkJsonPath);
        }
        this.subProjects
            .filter((subProject) => !fs.existsSync(`${subProject.outdir}/package.json`) ||
            JSON.parse(fs.readFileSync(`${subProject.outdir}/package.json`).toString()).__pdk__)
            .forEach((subProject) => {
            // generate a package.json if not found
            const manifest = {};
            manifest.name = subProject.name;
            manifest.private = true;
            manifest.__pdk__ = true;
            manifest.scripts = subProject.tasks.all.reduce((p, c) => ({
                [c.name]: `npx projen ${c.name}`,
                ...p,
            }), {});
            manifest.version = "0.0.0";
            new projen_1.JsonFile(subProject, "package.json", {
                obj: manifest,
                readonly: true,
            });
        });
        super.synth();
    }
}
exports.NxMonorepoProject = NxMonorepoProject;
_a = JSII_RTTI_SYMBOL_1;
NxMonorepoProject[_a] = { fqn: "aws-prototyping-sdk.pdk_projen.NxMonorepoProject", version: "0.0.44" };
//# sourceMappingURL=data:application/json;base64,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