"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SonarCodeScanner = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_codebuild_1 = require("aws-cdk-lib/aws-codebuild");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_secretsmanager_1 = require("aws-cdk-lib/aws-secretsmanager");
const constructs_1 = require("constructs");
const sonarqube_commands_1 = require("./sonarqube-commands");
const unpackSourceAndArtifacts = () => [
    'export SYNTH_ARTIFACT_URI=`echo $SYNTH_ARTIFACT_LOCATION | awk \'{sub("arn:aws:s3:::","s3://")}1\' $1`',
    'export SYNTH_SOURCE_URI=`echo $SYNTH_SOURCE_VERSION | awk \'{sub("arn:aws:s3:::","s3://")}1\' $1`',
    "aws s3 cp $SYNTH_SOURCE_URI source.zip",
    "aws s3 cp $SYNTH_ARTIFACT_URI dist.zip",
    "unzip source.zip",
    "unzip dist.zip -d cdk.out",
    "rm source.zip dist.zip",
];
const owaspScan = () => "npx owasp-dependency-check --format HTML --out reports --exclude '**/node_modules/**/*' --exclude '**/reports/**/*' --exclude '**/cdk.out/**/*' --exclude '**/.env/**/*' --exclude '**/dist/**/*' --exclude '**/.git/**/*' --scan . --enableExperimental --bin /tmp/dep-check --disableRetireJS";
const cfnNagScan = (cfnNagIgnorePath) => `cfn_nag ${cfnNagIgnorePath !== null && cfnNagIgnorePath !== void 0 ? cfnNagIgnorePath : ""} cdk.out/**/*.template.json --output-format=json > reports/cfn-nag-report.json`;
class SonarCodeScanner extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const sonarQubeToken = new aws_secretsmanager_1.Secret(this, "SonarQubeToken");
        const synthBuildProject = aws_codebuild_1.Project.fromProjectArn(this, "SynthBuildProject", props.synthBuildArn);
        const validationProject = new aws_codebuild_1.Project(this, "ValidationProject", {
            environment: {
                buildImage: aws_codebuild_1.LinuxBuildImage.STANDARD_5_0,
            },
            environmentVariables: {
                SONARQUBE_TOKEN: {
                    type: aws_codebuild_1.BuildEnvironmentVariableType.SECRETS_MANAGER,
                    value: sonarQubeToken.secretArn,
                },
                SONARQUBE_ENDPOINT: {
                    type: aws_codebuild_1.BuildEnvironmentVariableType.PLAINTEXT,
                    value: props.sonarqubeEndpoint,
                },
                PROJECT_NAME: {
                    type: aws_codebuild_1.BuildEnvironmentVariableType.PLAINTEXT,
                    value: props.sonarqubeProjectName,
                },
            },
            buildSpec: aws_codebuild_1.BuildSpec.fromObject({
                version: "0.2",
                env: {
                    shell: "bash",
                },
                phases: {
                    install: {
                        commands: ["gem install cfn-nag"],
                    },
                    build: {
                        commands: [
                            ...unpackSourceAndArtifacts(),
                            ...sonarqube_commands_1.createSonarqubeProject(props),
                            owaspScan(),
                            cfnNagScan(props.cfnNagIgnorePath),
                            sonarqube_commands_1.sonarqubeScanner(),
                            ...sonarqube_commands_1.generateSonarqubeReports(),
                            ...(props.preArchiveCommands || []),
                        ],
                    },
                },
            }),
        });
        validationProject.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ["codebuild:BatchGetBuilds"],
            effect: aws_iam_1.Effect.ALLOW,
            resources: [synthBuildProject.projectArn],
        }));
        validationProject.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ["s3:GetObject*"],
            effect: aws_iam_1.Effect.ALLOW,
            resources: [props.artifactBucketArn, `${props.artifactBucketArn}/**`],
        }));
        props.artifactBucketKeyArn &&
            validationProject.addToRolePolicy(new aws_iam_1.PolicyStatement({
                actions: ["kms:Decrypt", "kms:DescribeKey"],
                effect: aws_iam_1.Effect.ALLOW,
                resources: [props.artifactBucketKeyArn],
            }));
        synthBuildProject.onBuildSucceeded("OnSynthSuccess", {
            target: new aws_events_targets_1.CodeBuildProject(validationProject, {
                event: aws_events_1.RuleTargetInput.fromObject({
                    environmentVariablesOverride: [
                        {
                            name: "SYNTH_BUILD_ID",
                            type: "PLAINTEXT",
                            value: aws_events_1.EventField.fromPath("$.detail.build-id"),
                        },
                        {
                            name: "SYNTH_ARTIFACT_LOCATION",
                            type: "PLAINTEXT",
                            value: aws_events_1.EventField.fromPath("$.detail.additional-information.artifact.location"),
                        },
                        {
                            name: "SYNTH_SOURCE_VERSION",
                            type: "PLAINTEXT",
                            value: aws_events_1.EventField.fromPath("$.detail.additional-information.source-version"),
                        },
                    ],
                }),
            }),
        });
        new aws_cdk_lib_1.CfnOutput(this, "SonarqubeSecretArn", {
            exportName: "SonarqubeSecretArn",
            value: sonarQubeToken.secretArn,
        });
    }
}
exports.SonarCodeScanner = SonarCodeScanner;
_a = JSII_RTTI_SYMBOL_1;
SonarCodeScanner[_a] = { fqn: "aws-prototyping-sdk.pdk_pipeline.SonarCodeScanner", version: "0.0.44" };
//# sourceMappingURL=data:application/json;base64,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