"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cognito = require("@aws-cdk/aws-cognito");
const ec2 = require("@aws-cdk/aws-ec2");
const elbv2 = require("@aws-cdk/aws-elasticloadbalancingv2");
const core_1 = require("@aws-cdk/core");
const actions = require("../lib");
class CognitoStack extends core_1.Stack {
    /// !hide
    constructor(scope, id) {
        var _a;
        super(scope, id);
        const vpc = new ec2.Vpc(this, 'Stack', {
            maxAzs: 2,
        });
        const certificate = {
            certificateArn: (_a = process.env.SELF_SIGNED_CERT_ARN) !== null && _a !== void 0 ? _a : '',
        };
        /// !show
        const lb = new elbv2.ApplicationLoadBalancer(this, 'LB', {
            vpc,
            internetFacing: true,
        });
        const userPool = new cognito.UserPool(this, 'UserPool');
        const userPoolClient = new cognito.UserPoolClient(this, 'Client', {
            userPool,
            // Required minimal configuration for use with an ELB
            generateSecret: true,
            authFlows: {
                userPassword: true,
                refreshToken: true,
            },
            oAuth: {
                flows: {
                    authorizationCodeGrant: true,
                },
                scopes: [cognito.OAuthScope.EMAIL],
                callbackUrls: [
                    `https://${lb.loadBalancerDnsName}/oauth2/idpresponse`,
                ],
            },
        });
        const cfnClient = userPoolClient.node.defaultChild;
        cfnClient.addPropertyOverride('RefreshTokenValidity', 1);
        cfnClient.addPropertyOverride('SupportedIdentityProviders', ['COGNITO']);
        const userPoolDomain = new cognito.UserPoolDomain(this, 'Domain', {
            userPool,
            cognitoDomain: {
                domainPrefix: 'test-cdk-prefix',
            },
        });
        lb.addListener('Listener', {
            port: 443,
            certificates: [certificate],
            defaultAction: new actions.AuthenticateCognitoAction({
                userPool,
                userPoolClient,
                userPoolDomain,
                next: elbv2.ListenerAction.fixedResponse(200, {
                    contentType: 'text/plain',
                    messageBody: 'Authenticated',
                }),
            }),
        });
        new core_1.CfnOutput(this, 'DNS', {
            value: lb.loadBalancerDnsName,
        });
    }
}
const app = new core_1.App();
new CognitoStack(app, 'integ-cognito');
app.synth();
//# sourceMappingURL=data:application/json;base64,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