"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Protocol = exports.Service = exports.ServiceType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s = require("cdk8s");
const base_1 = require("./base");
const k8s = require("./imports/k8s");
const ingress_1 = require("./ingress");
/**
 * For some parts of your application (for example, frontends) you may want to expose a Service onto an
 * external IP address, that's outside of your cluster.
 * Kubernetes ServiceTypes allow you to specify what kind of Service you want.
 * The default is ClusterIP.
 */
var ServiceType;
(function (ServiceType) {
    /**
     * Exposes the Service on a cluster-internal IP.
     * Choosing this value makes the Service only reachable from within the cluster.
     * This is the default ServiceType
     */
    ServiceType["CLUSTER_IP"] = "ClusterIP";
    /**
     * Exposes the Service on each Node's IP at a static port (the NodePort).
     * A ClusterIP Service, to which the NodePort Service routes, is automatically created.
     * You'll be able to contact the NodePort Service, from outside the cluster,
     * by requesting <NodeIP>:<NodePort>.
     */
    ServiceType["NODE_PORT"] = "NodePort";
    /**
     * Exposes the Service externally using a cloud provider's load balancer.
     * NodePort and ClusterIP Services, to which the external load balancer routes,
     * are automatically created.
     */
    ServiceType["LOAD_BALANCER"] = "LoadBalancer";
    /**
     * Maps the Service to the contents of the externalName field (e.g. foo.bar.example.com), by returning a CNAME record with its value.
     * No proxying of any kind is set up.
     *
     * > Note: You need either kube-dns version 1.7 or CoreDNS version 0.0.8 or higher to use the ExternalName type.
     */
    ServiceType["EXTERNAL_NAME"] = "ExternalName";
})(ServiceType = exports.ServiceType || (exports.ServiceType = {}));
/**
 * An abstract way to expose an application running on a set of Pods as a network service.
 * With Kubernetes you don't need to modify your application to use an unfamiliar service discovery mechanism.
 * Kubernetes gives Pods their own IP addresses and a single DNS name for a set of Pods, and can load-balance across them.
 *
 * For example, consider a stateless image-processing backend which is running with 3 replicas. Those replicas are fungible—frontends do not care which backend they use.
 * While the actual Pods that compose the backend set may change, the frontend clients should not need to be aware of that,
 * nor should they need to keep track of the set of backends themselves.
 * The Service abstraction enables this decoupling.
 *
 * If you're able to use Kubernetes APIs for service discovery in your application, you can query the API server for Endpoints,
 * that get updated whenever the set of Pods in a Service changes. For non-native applications, Kubernetes offers ways to place a network port
 * or load balancer in between your application and the backend Pods.
 */
class Service extends base_1.Resource {
    constructor(scope, id, props = {}) {
        var _b, _c, _d;
        super(scope, id);
        this.apiObject = new k8s.KubeService(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s.Lazy.any({ produce: () => this._toKube() }),
        });
        this.clusterIP = props.clusterIP;
        this.externalName = props.externalName;
        if (props.externalName !== undefined) {
            this.type = ServiceType.EXTERNAL_NAME;
        }
        else {
            this.type = (_b = props.type) !== null && _b !== void 0 ? _b : ServiceType.CLUSTER_IP;
        }
        this._externalIPs = (_c = props.externalIPs) !== null && _c !== void 0 ? _c : [];
        this._ports = [];
        this._selector = {};
        this._loadBalancerSourceRanges = props.loadBalancerSourceRanges;
        for (const portAndOptions of (_d = props.ports) !== null && _d !== void 0 ? _d : []) {
            this.serve(portAndOptions.port, portAndOptions);
        }
    }
    /**
     * Expose a service via an ingress using the specified path.
     *
     * @param path The path to expose the service under.
     * @param options Additional options.
     *
     * @returns The `Ingress` resource that was used.
     */
    exposeViaIngress(path, options = {}) {
        var _b;
        const ingress = (_b = options.ingress) !== null && _b !== void 0 ? _b : new ingress_1.Ingress(this, 'Ingress');
        ingress.addRule(path, ingress_1.IngressBackend.fromService(this), options.pathType);
        return ingress;
    }
    /**
     * Returns the labels which are used to select pods for this service.
     */
    get selector() {
        return this._selector;
    }
    /**
     * Ports for this service.
     *
     * Use `serve()` to expose additional service ports.
     */
    get ports() {
        return [...this._ports];
    }
    /**
     * Associate a deployment to this service.
     *
     * If not targetPort is specific in the portOptions, then requests will be routed
     * to the port exposed by the first container in the deployment's pods.
     * The deployment's `labelSelector` will be used to select pods.
     *
     * @param deployment The deployment to expose
     * @param options Optional settings for the port.
     */
    addDeployment(deployment, options = {}) {
        var _b, _c;
        const containers = deployment.containers;
        if (containers.length === 0) {
            throw new Error('Cannot expose a deployment without containers');
        }
        // just a PoC, we assume the first container is the main one.
        // TODO: figure out what the correct thing to do here.
        const container = containers[0];
        const port = (_b = options.port) !== null && _b !== void 0 ? _b : container.port;
        const targetPort = (_c = options.targetPort) !== null && _c !== void 0 ? _c : containers[0].port;
        if (!port) {
            throw new Error('Cannot determine port. Either pass `port` in options or configure a port on the first container of the deployment');
        }
        const selector = Object.entries(deployment.labelSelector);
        if (selector.length === 0) {
            throw new Error('deployment does not have a label selector');
        }
        if (Object.keys(this.selector).length > 0) {
            throw new Error('a selector is already defined for this service. cannot add a deployment');
        }
        for (const [k, v] of selector) {
            this.addSelector(k, v);
        }
        this.serve(port, { ...options, targetPort });
    }
    /**
     * Services defined using this spec will select pods according the provided label.
     *
     * @param label The label key.
     * @param value The label value.
     */
    addSelector(label, value) {
        this._selector[label] = value;
    }
    /**
     * Configure a port the service will bind to.
     * This method can be called multiple times.
     *
     * @param port The port definition.
     */
    serve(port, options = {}) {
        this._ports.push({ ...options, port });
    }
    /**
     * @internal
     */
    _toKube() {
        if (this._ports.length === 0 && this.type !== ServiceType.EXTERNAL_NAME) {
            throw new Error('A service must be configured with a port');
        }
        if (this.type === ServiceType.EXTERNAL_NAME && this.externalName === undefined) {
            throw new Error('A service with type EXTERNAL_NAME requires an externalName prop');
        }
        const ports = [];
        for (const port of this._ports) {
            ports.push({
                name: port.name,
                port: port.port,
                targetPort: port.targetPort ? k8s.IntOrString.fromNumber(port.targetPort) : undefined,
                nodePort: port.nodePort,
                protocol: port.protocol,
            });
        }
        return this.type !== ServiceType.EXTERNAL_NAME ? {
            clusterIp: this.clusterIP,
            externalIPs: this._externalIPs,
            externalName: this.externalName,
            type: this.type,
            selector: this._selector,
            ports: ports,
            loadBalancerSourceRanges: this._loadBalancerSourceRanges,
        } : {
            type: this.type,
            externalName: this.externalName,
        };
    }
}
exports.Service = Service;
_a = JSII_RTTI_SYMBOL_1;
Service[_a] = { fqn: "cdk8s-plus-22.Service", version: "1.0.0-beta.198" };
var Protocol;
(function (Protocol) {
    Protocol["TCP"] = "TCP";
    Protocol["UDP"] = "UDP";
    Protocol["SCTP"] = "SCTP";
})(Protocol = exports.Protocol || (exports.Protocol = {}));
//# sourceMappingURL=data:application/json;base64,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