"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FsGroupChangePolicy = exports.RestartPolicy = exports.PodSecurityContext = exports.Pod = exports.PodTemplate = exports.PodSpec = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const base_1 = require("./base");
const container_1 = require("./container");
const k8s = require("./imports/k8s");
/**
 * Provides read/write capabilities ontop of a `PodSpecProps`.
 */
class PodSpec {
    constructor(props = {}) {
        this._containers = [];
        this._initContainers = [];
        this._hostAliases = [];
        this._volumes = new Map();
        this.restartPolicy = props.restartPolicy;
        this.serviceAccount = props.serviceAccount;
        this.securityContext = new PodSecurityContext(props.securityContext);
        this.dockerRegistryAuth = props.dockerRegistryAuth;
        if (props.containers) {
            props.containers.forEach(c => this.addContainer(c));
        }
        if (props.volumes) {
            props.volumes.forEach(v => this.addVolume(v));
        }
        if (props.initContainers) {
            props.initContainers.forEach(c => this.addInitContainer(c));
        }
        if (props.hostAliases) {
            props.hostAliases.forEach(c => this.addHostAlias(c));
        }
    }
    get containers() {
        return [...this._containers];
    }
    get initContainers() {
        return [...this._initContainers];
    }
    get volumes() {
        return Array.from(this._volumes.values());
    }
    get hostAliases() {
        return [...this._hostAliases];
    }
    addContainer(container) {
        const impl = new container_1.Container(container);
        this._containers.push(impl);
        return impl;
    }
    addInitContainer(container) {
        var _e;
        // https://kubernetes.io/docs/concepts/workloads/pods/init-containers/#differences-from-regular-containers
        if (container.readiness) {
            throw new Error('Init containers must not have a readiness probe');
        }
        if (container.liveness) {
            throw new Error('Init containers must not have a liveness probe');
        }
        if (container.startup) {
            throw new Error('Init containers must not have a startup probe');
        }
        const impl = new container_1.Container({
            ...container,
            name: (_e = container.name) !== null && _e !== void 0 ? _e : `init-${this._initContainers.length}`,
        });
        this._initContainers.push(impl);
        return impl;
    }
    addHostAlias(hostAlias) {
        this._hostAliases.push(hostAlias);
    }
    addVolume(volume) {
        const existingVolume = this._volumes.get(volume.name);
        if (existingVolume) {
            throw new Error(`Volume with name ${volume.name} already exists`);
        }
        this._volumes.set(volume.name, volume);
    }
    /**
     * @internal
     */
    _toPodSpec() {
        var _e;
        if (this.containers.length === 0) {
            throw new Error('PodSpec must have at least 1 container');
        }
        const volumes = new Map();
        const containers = [];
        const initContainers = [];
        for (const container of this.containers) {
            // automatically add volume from the container mount
            // to this pod so thats its available to the container.
            for (const mount of container.mounts) {
                addVolume(mount.volume);
            }
            containers.push(container._toKube());
        }
        for (const container of this.initContainers) {
            // automatically add volume from the container mount
            // to this pod so thats its available to the container.
            for (const mount of container.mounts) {
                addVolume(mount.volume);
            }
            initContainers.push(container._toKube());
        }
        for (const volume of this.volumes) {
            addVolume(volume);
        }
        function addVolume(volume) {
            const existingVolume = volumes.get(volume.name);
            // its ok to call this function twice on the same volume, but its not ok to
            // call it twice on a different volume with the same name.
            if (existingVolume && existingVolume !== volume) {
                throw new Error(`Invalid mount configuration. At least two different volumes have the same name: ${volume.name}`);
            }
            volumes.set(volume.name, volume);
        }
        return {
            restartPolicy: this.restartPolicy,
            serviceAccountName: (_e = this.serviceAccount) === null || _e === void 0 ? void 0 : _e.name,
            containers: containers,
            securityContext: this.securityContext._toKube(),
            initContainers: initContainers,
            hostAliases: this.hostAliases,
            volumes: Array.from(volumes.values()).map(v => v._toKube()),
            imagePullSecrets: this.dockerRegistryAuth ? [{ name: this.dockerRegistryAuth.name }] : undefined,
        };
    }
}
exports.PodSpec = PodSpec;
_a = JSII_RTTI_SYMBOL_1;
PodSpec[_a] = { fqn: "cdk8s-plus-22.PodSpec", version: "1.0.0-beta.198" };
/**
 * Provides read/write capabilities ontop of a `PodTemplateProps`.
 */
class PodTemplate extends PodSpec {
    constructor(props = {}) {
        super(props);
        this.podMetadata = new cdk8s_1.ApiObjectMetadataDefinition(props.podMetadata);
    }
    /**
     * @internal
     */
    _toPodTemplateSpec() {
        return {
            metadata: this.podMetadata.toJson(),
            spec: this._toPodSpec(),
        };
    }
}
exports.PodTemplate = PodTemplate;
_b = JSII_RTTI_SYMBOL_1;
PodTemplate[_b] = { fqn: "cdk8s-plus-22.PodTemplate", version: "1.0.0-beta.198" };
/**
 * Pod is a collection of containers that can run on a host. This resource is
 * created by clients and scheduled onto hosts.
 */
class Pod extends base_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.apiObject = new k8s.KubePod(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._spec._toPodSpec() }),
        });
        this._spec = new PodSpec(props);
    }
    get containers() {
        return this._spec.containers;
    }
    get initContainers() {
        return this._spec.initContainers;
    }
    get volumes() {
        return this._spec.volumes;
    }
    get restartPolicy() {
        return this._spec.restartPolicy;
    }
    get serviceAccount() {
        return this._spec.serviceAccount;
    }
    get securityContext() {
        return this._spec.securityContext;
    }
    get hostAliases() {
        return this._spec.hostAliases;
    }
    addContainer(container) {
        return this._spec.addContainer(container);
    }
    addInitContainer(container) {
        return this._spec.addInitContainer(container);
    }
    addVolume(volume) {
        return this._spec.addVolume(volume);
    }
    addHostAlias(hostAlias) {
        return this._spec.addHostAlias(hostAlias);
    }
}
exports.Pod = Pod;
_c = JSII_RTTI_SYMBOL_1;
Pod[_c] = { fqn: "cdk8s-plus-22.Pod", version: "1.0.0-beta.198" };
/**
 * Holds pod-level security attributes and common container settings.
 */
class PodSecurityContext {
    constructor(props = {}) {
        var _e, _f, _g;
        this._sysctls = [];
        this.ensureNonRoot = (_e = props.ensureNonRoot) !== null && _e !== void 0 ? _e : false;
        this.fsGroupChangePolicy = (_f = props.fsGroupChangePolicy) !== null && _f !== void 0 ? _f : FsGroupChangePolicy.ALWAYS;
        this.user = props.user;
        this.group = props.group;
        this.fsGroup = props.fsGroup;
        for (const sysctl of (_g = props.sysctls) !== null && _g !== void 0 ? _g : []) {
            this._sysctls.push(sysctl);
        }
    }
    get sysctls() {
        return [...this._sysctls];
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            runAsGroup: this.group,
            runAsUser: this.user,
            fsGroup: this.fsGroup,
            runAsNonRoot: this.ensureNonRoot,
            fsGroupChangePolicy: this.fsGroupChangePolicy,
            sysctls: this._sysctls,
        };
    }
}
exports.PodSecurityContext = PodSecurityContext;
_d = JSII_RTTI_SYMBOL_1;
PodSecurityContext[_d] = { fqn: "cdk8s-plus-22.PodSecurityContext", version: "1.0.0-beta.198" };
/**
 * Restart policy for all containers within the pod.
 */
var RestartPolicy;
(function (RestartPolicy) {
    /**
     * Always restart the pod after it exits.
     */
    RestartPolicy["ALWAYS"] = "Always";
    /**
     * Only restart if the pod exits with a non-zero exit code.
     */
    RestartPolicy["ON_FAILURE"] = "OnFailure";
    /**
     * Never restart the pod.
     */
    RestartPolicy["NEVER"] = "Never";
})(RestartPolicy = exports.RestartPolicy || (exports.RestartPolicy = {}));
var FsGroupChangePolicy;
(function (FsGroupChangePolicy) {
    /**
     * Only change permissions and ownership if permission and ownership of root directory does
     * not match with expected permissions of the volume.
     * This could help shorten the time it takes to change ownership and permission of a volume
     */
    FsGroupChangePolicy["ON_ROOT_MISMATCH"] = "OnRootMismatch";
    /**
     * Always change permission and ownership of the volume when volume is mounted.
     */
    FsGroupChangePolicy["ALWAYS"] = "Always";
})(FsGroupChangePolicy = exports.FsGroupChangePolicy || (exports.FsGroupChangePolicy = {}));
//# sourceMappingURL=data:application/json;base64,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