import { ApiObject, ApiObjectMetadataDefinition } from 'cdk8s';
import { Construct } from 'constructs';
import { Resource, ResourceProps } from './base';
import { Container, ContainerProps } from './container';
import * as k8s from './imports/k8s';
import { Ingress } from './ingress';
import { RestartPolicy, IPodTemplate, PodTemplateProps, PodSecurityContext, HostAlias } from './pod';
import { ExposeServiceViaIngressOptions, Protocol, Service, ServiceType } from './service';
import { IServiceAccount } from './service-account';
import { Volume } from './volume';
/**
 * Properties for initialization of `Deployment`.
 */
export interface DeploymentProps extends ResourceProps, PodTemplateProps {
    /**
     * Number of desired pods.
     *
     * @default 1
     */
    readonly replicas?: number;
    /**
     * Automatically allocates a pod selector for this deployment.
     *
     * If this is set to `false` you must define your selector through
     * `deployment.podMetadata.addLabel()` and `deployment.selectByLabel()`.
     *
     * @default true
     */
    readonly defaultSelector?: boolean;
    /**
     * Specifies the strategy used to replace old Pods by new ones.
     *
     * @default - RollingUpdate with maxSurge and maxUnavailable set to 25%.
     */
    readonly strategy?: DeploymentStrategy;
}
/**
 * Options for exposing a deployment via a service.
 */
export interface ExposeDeploymentViaServiceOptions {
    /**
     * The port that the service should serve on.
     *
     * @default - Copied from the container of the deployment. If a port could not be determined, throws an error.
     */
    readonly port?: number;
    /**
     * The type of the exposed service.
     *
     * @default - ClusterIP.
     */
    readonly serviceType?: ServiceType;
    /**
     * The name of the service to expose.
     * This will be set on the Service.metadata and must be a DNS_LABEL
     *
     * @default undefined Uses the system generated name.
     */
    readonly name?: string;
    /**
     * The IP protocol for this port. Supports "TCP", "UDP", and "SCTP". Default is TCP.
     *
     * @default Protocol.TCP
     */
    readonly protocol?: Protocol;
    /**
     * The port number the service will redirect to.
     *
     * @default - The port of the first container in the deployment (ie. containers[0].port)
     */
    readonly targetPort?: number;
}
/**
 * Options for exposing a deployment via an ingress.
 */
export interface ExposeDeploymentViaIngressOptions extends ExposeDeploymentViaServiceOptions, ExposeServiceViaIngressOptions {
}
/**
*
* A Deployment provides declarative updates for Pods and ReplicaSets.
*
* You describe a desired state in a Deployment, and the Deployment Controller changes the actual
* state to the desired state at a controlled rate. You can define Deployments to create new ReplicaSets, or to remove
* existing Deployments and adopt all their resources with new Deployments.
*
* > Note: Do not manage ReplicaSets owned by a Deployment. Consider opening an issue in the main Kubernetes repository if your use case is not covered below.
*
* Use Case
* ---------
*
* The following are typical use cases for Deployments:
*
* - Create a Deployment to rollout a ReplicaSet. The ReplicaSet creates Pods in the background.
*   Check the status of the rollout to see if it succeeds or not.
* - Declare the new state of the Pods by updating the PodTemplateSpec of the Deployment.
*   A new ReplicaSet is created and the Deployment manages moving the Pods from the old ReplicaSet to the new one at a controlled rate.
*   Each new ReplicaSet updates the revision of the Deployment.
* - Rollback to an earlier Deployment revision if the current state of the Deployment is not stable.
*   Each rollback updates the revision of the Deployment.
* - Scale up the Deployment to facilitate more load.
* - Pause the Deployment to apply multiple fixes to its PodTemplateSpec and then resume it to start a new rollout.
* - Use the status of the Deployment as an indicator that a rollout has stuck.
* - Clean up older ReplicaSets that you don't need anymore.
*
**/
export declare class Deployment extends Resource implements IPodTemplate {
    /**
     * Number of desired pods.
     */
    readonly replicas: number;
    /**
     * The upgrade strategy of this deployment.
     */
    readonly strategy: DeploymentStrategy;
    /**
     * @see base.Resource.apiObject
     */
    protected readonly apiObject: ApiObject;
    private readonly _podTemplate;
    private readonly _labelSelector;
    constructor(scope: Construct, id: string, props?: DeploymentProps);
    get podMetadata(): ApiObjectMetadataDefinition;
    /**
     * The labels this deployment will match against in order to select pods.
     *
     * Returns a a copy. Use `selectByLabel()` to add labels.
     */
    get labelSelector(): Record<string, string>;
    get containers(): Container[];
    get initContainers(): Container[];
    get hostAliases(): HostAlias[];
    get volumes(): Volume[];
    get restartPolicy(): RestartPolicy | undefined;
    get serviceAccount(): IServiceAccount | undefined;
    get securityContext(): PodSecurityContext;
    /**
     * Configure a label selector to this deployment.
     * Pods that have the label will be selected by deployments configured with this spec.
     *
     * @param key - The label key.
     * @param value - The label value.
     */
    selectByLabel(key: string, value: string): void;
    /**
     * Expose a deployment via a service.
     *
     * This is equivalent to running `kubectl expose deployment <deployment-name>`.
     *
     * @param options Options to determine details of the service and port exposed.
     */
    exposeViaService(options?: ExposeDeploymentViaServiceOptions): Service;
    /**
     * Expose a deployment via an ingress.
     *
     * This will first expose the deployment with a service, and then expose the service via an ingress.
     *
     * @param path The ingress path to register under.
     * @param options Additional options.
     */
    exposeViaIngress(path: string, options?: ExposeDeploymentViaIngressOptions): Ingress;
    addContainer(container: ContainerProps): Container;
    addInitContainer(container: ContainerProps): Container;
    addHostAlias(hostAlias: HostAlias): void;
    addVolume(volume: Volume): void;
    /**
     * @internal
     */
    _toKube(): k8s.DeploymentSpec;
}
/**
 * Options for `DeploymentStrategy.rollingUpdate`.
 */
export interface DeploymentStrategyRollingUpdateOptions {
    /**
     * The maximum number of pods that can be scheduled above the desired number of pods.
     * Value can be an absolute number (ex: 5) or a percentage of desired pods (ex: 10%).
     * Absolute number is calculated from percentage by rounding up.
     * This can not be 0 if `maxUnavailable` is 0.
     *
     * Example: when this is set to 30%, the new ReplicaSet can be scaled up immediately when the rolling update
     * starts, such that the total number of old and new pods do not exceed 130% of desired pods.
     * Once old pods have been killed, new ReplicaSet can be scaled up further, ensuring that
     * total number of pods running at any time during the update is at most 130% of desired pods.
     *
     * @default '25%'
     */
    readonly maxSurge?: PercentOrAbsolute;
    /**
     * The maximum number of pods that can be unavailable during the update.
     * Value can be an absolute number (ex: 5) or a percentage of desired pods (ex: 10%).
     * Absolute number is calculated from percentage by rounding down.
     * This can not be 0 if `maxSurge` is 0.
     *
     * Example: when this is set to 30%, the old ReplicaSet can be scaled down to 70% of desired
     * pods immediately when the rolling update starts. Once new pods are ready, old ReplicaSet can
     * be scaled down further, followed by scaling up the new ReplicaSet, ensuring that the total
     * number of pods available at all times during the update is at least 70% of desired pods.
     *
     * @default '25%'
     */
    readonly maxUnavailable?: PercentOrAbsolute;
}
/**
 * Union like class repsenting either a ration in
 * percents or an absolute number.
 */
export declare class PercentOrAbsolute {
    readonly value: any;
    /**
     * Percent ratio.
     */
    static percent(percent: number): PercentOrAbsolute;
    /**
     * Absolute number.
     */
    static absolute(num: number): PercentOrAbsolute;
    private constructor();
    isZero(): boolean;
}
/**
 * Deployment strategies.
 */
export declare class DeploymentStrategy {
    private readonly strategy;
    /**
     * All existing Pods are killed before new ones are created.
     *
     * @see https://kubernetes.io/docs/concepts/workloads/controllers/deployment/#recreate-deployment
     */
    static recreate(): DeploymentStrategy;
    static rollingUpdate(options?: DeploymentStrategyRollingUpdateOptions): DeploymentStrategy;
    private constructor();
    /**
     * @internal
     */
    _toKube(): k8s.DeploymentStrategy;
}
