import { ApiObject, ApiObjectMetadataDefinition } from 'cdk8s';
import { Construct } from 'constructs';
import { Resource, ResourceProps } from './base';
import { Container, ContainerProps } from './container';
import * as k8s from './imports/k8s';
import { HostAlias, IPodTemplate, PodSecurityContext, PodTemplateProps, RestartPolicy } from './pod';
import { IServiceAccount } from './service-account';
import { Volume } from './volume';
/**
 * Properties for `DaemonSet`.
 */
export interface DaemonSetProps extends ResourceProps, PodTemplateProps {
    /**
     * Minimum number of seconds for which a newly created pod should
     * be ready without any of its container crashing, for it to be considered available.
     *
     * @default 0
     */
    readonly minReadySeconds?: number;
    /**
     * Automatically allocates a pod selector for this daemon set.
     *
     * If this is set to `false` you must define your selector through
     * `dset.podMetadata.addLabel()` and `dset.selectByLabel()`.
     *
     * @default true
     */
    readonly defaultSelector?: boolean;
}
/**
 * A DaemonSet ensures that all (or some) Nodes run a copy of a Pod.
 * As nodes are added to the cluster, Pods are added to them.
 * As nodes are removed from the cluster, those Pods are garbage collected.
 * Deleting a DaemonSet will clean up the Pods it created.
 *
 * Some typical uses of a DaemonSet are:
 *
 * - running a cluster storage daemon on every node
 * - running a logs collection daemon on every node
 * - running a node monitoring daemon on every node
 *
 * In a simple case, one DaemonSet, covering all nodes, would be used for each type of daemon.
 * A more complex setup might use multiple DaemonSets for a single type of daemon,
 * but with different flags and/or different memory and cpu requests for different hardware types.
 */
export declare class DaemonSet extends Resource implements IPodTemplate {
    private readonly _podTemplate;
    private readonly _labelSelector;
    /**
     * @see base.Resource.apiObject
     */
    protected readonly apiObject: ApiObject;
    readonly minReadySeconds: number;
    constructor(scope: Construct, id: string, props?: DaemonSetProps);
    /**
     * The labels this daemon set will match against in order to select pods.
     *
     * Returns a a copy. Use `selectByLabel()` to add labels.
     */
    get labelSelector(): Record<string, string>;
    get podMetadata(): ApiObjectMetadataDefinition;
    get containers(): Container[];
    get initContainers(): Container[];
    get hostAliases(): HostAlias[];
    get volumes(): Volume[];
    get restartPolicy(): RestartPolicy | undefined;
    get serviceAccount(): IServiceAccount | undefined;
    get securityContext(): PodSecurityContext;
    addContainer(container: ContainerProps): Container;
    addInitContainer(container: ContainerProps): Container;
    addHostAlias(hostAlias: HostAlias): void;
    addVolume(volume: Volume): void;
    /**
     * Configure a label selector to this daemon set.
     */
    selectByLabel(key: string, value: string): void;
    /**
     * @internal
     */
    _toKube(): k8s.DaemonSetSpec;
}
