"""*The HTTP application server for mocking APIs*

This module provides objects for mocking APIs as a web application with different Python framework.
"""

from typing import Any, Optional

from fake_api_server.model import APIConfig, MockAPIs, load_config

from .rest.application import BaseAppServer, FlaskServer


class MockHTTPServer:
    """*Mocking APIs as web application with HTTP*

    It provides the web application which mocks all APIs from configuration with one specific Python web framework. In
    default, it would use *Flask* to set up the web server to provide the APIs.
    """

    def __init__(
        self,
        config_path: Optional[str] = None,
        app_server: Optional[BaseAppServer] = None,
        auto_setup: Optional[bool] = False,
    ):
        """

        Args:
            config_path (str): The file path of configuration about mocked APIs. In default, it would search file
                *api.yaml* in the current directory.
            app_server (BaseAppServer): Which web application to use to set up the web application to mock APIs. In
                generally, it must be the *fake_api_server.application.BaseAppServer* type object. In default, it would use
                *Flask* to set up the web application.
            auto_setup (auto_setup): Initial and create mocked APIs when instantiate this object. In default, it's
                ``False``.
        """
        if not config_path:
            config_path = "api.yaml"
        self._config_path = config_path
        self._api_config: Optional[APIConfig] = load_config(path=self._config_path)

        if app_server and not isinstance(app_server, BaseAppServer):
            raise TypeError(
                f"The instance {app_server} must be *fake_api_server.application.BaseAppServer* type object."
            )
        if not app_server:
            app_server = FlaskServer()
        self._app_server = app_server
        self._web_application = None

        if auto_setup and (self._api_config and self._api_config.apis):
            self.create_apis(mocked_apis=self._api_config.apis)

    @property
    def web_app(self) -> Any:
        """:obj:`Any`: Property with only getter for the instance of web application, e.g., *Flask*, *FastAPI*, etc."""
        return self._app_server.web_application

    def create_apis(self, mocked_apis: MockAPIs) -> None:
        """Initial and create all mocked APIs from the data objects which be generated by configuration.

        Args:
            mocked_apis (MockAPIs): The data object of mocked APIs configuration which be generated by utility function
                *fake_api_server._utils.load_config*.

        Returns:
            None

        """
        self._app_server.create_api(mocked_apis)
