"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DeploymentStrategy = exports.PercentOrAbsolute = exports.Deployment = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const k8s = require("./imports/k8s");
const service = require("./service");
const utils_1 = require("./utils");
const workload = require("./workload");
/**
*
* A Deployment provides declarative updates for Pods and ReplicaSets.
*
* You describe a desired state in a Deployment, and the Deployment Controller changes the actual
* state to the desired state at a controlled rate. You can define Deployments to create new ReplicaSets, or to remove
* existing Deployments and adopt all their resources with new Deployments.
*
* > Note: Do not manage ReplicaSets owned by a Deployment. Consider opening an issue in the main Kubernetes repository if your use case is not covered below.
*
* Use Case
* ---------
*
* The following are typical use cases for Deployments:
*
* - Create a Deployment to rollout a ReplicaSet. The ReplicaSet creates Pods in the background.
*   Check the status of the rollout to see if it succeeds or not.
* - Declare the new state of the Pods by updating the PodTemplateSpec of the Deployment.
*   A new ReplicaSet is created and the Deployment manages moving the Pods from the old ReplicaSet to the new one at a controlled rate.
*   Each new ReplicaSet updates the revision of the Deployment.
* - Rollback to an earlier Deployment revision if the current state of the Deployment is not stable.
*   Each rollback updates the revision of the Deployment.
* - Scale up the Deployment to facilitate more load.
* - Pause the Deployment to apply multiple fixes to its PodTemplateSpec and then resume it to start a new rollout.
* - Use the status of the Deployment as an indicator that a rollout has stuck.
* - Clean up older ReplicaSets that you don't need anymore.
*
**/
class Deployment extends workload.Workload {
    constructor(scope, id, props = {}) {
        var _d, _e, _f, _g;
        super(scope, id, props);
        this.resourceType = 'deployments';
        this.apiObject = new k8s.KubeDeployment(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
        this.minReady = (_d = props.minReady) !== null && _d !== void 0 ? _d : cdk8s_1.Duration.seconds(0);
        this.progressDeadline = (_e = props.progressDeadline) !== null && _e !== void 0 ? _e : cdk8s_1.Duration.seconds(600);
        if (this.progressDeadline.toSeconds() <= this.minReady.toSeconds()) {
            throw new Error(`'progressDeadline' (${this.progressDeadline.toSeconds()}s) must be greater than 'minReady' (${this.minReady.toSeconds()}s)`);
        }
        this.replicas = (_f = props.replicas) !== null && _f !== void 0 ? _f : 1;
        this.strategy = (_g = props.strategy) !== null && _g !== void 0 ? _g : DeploymentStrategy.rollingUpdate();
    }
    /**
     * Expose a deployment via a service.
     *
     * This is equivalent to running `kubectl expose deployment <deployment-name>`.
     *
     * @param options Options to determine details of the service and port exposed.
     */
    exposeViaService(options = {}) {
        var _d;
        const ser = new service.Service(this, 'Service', {
            metadata: options.name ? { name: options.name } : undefined,
            type: (_d = options.serviceType) !== null && _d !== void 0 ? _d : service.ServiceType.CLUSTER_IP,
        });
        ser.addDeployment(this, { protocol: options.protocol, targetPort: options.targetPort, port: options.port });
        return ser;
    }
    /**
     * Expose a deployment via an ingress.
     *
     * This will first expose the deployment with a service, and then expose the service via an ingress.
     *
     * @param path The ingress path to register under.
     * @param options Additional options.
     */
    exposeViaIngress(path, options = {}) {
        const ser = this.exposeViaService(options);
        return ser.exposeViaIngress(path, options);
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            replicas: this.replicas,
            minReadySeconds: this.minReady.toSeconds(),
            progressDeadlineSeconds: this.progressDeadline.toSeconds(),
            template: {
                metadata: this.podMetadata.toJson(),
                spec: this._toPodSpec(),
            },
            selector: {
                matchExpressions: utils_1.undefinedIfEmpty(this.matchExpressions),
                matchLabels: utils_1.undefinedIfEmpty(this.matchLabels),
            },
            strategy: this.strategy._toKube(),
        };
    }
}
exports.Deployment = Deployment;
_a = JSII_RTTI_SYMBOL_1;
Deployment[_a] = { fqn: "cdk8s-plus-22.Deployment", version: "2.0.0-beta.10" };
/**
 * Union like class repsenting either a ration in
 * percents or an absolute number.
 */
class PercentOrAbsolute {
    constructor(value) {
        this.value = value;
    }
    /**
     * Percent ratio.
     */
    static percent(percent) {
        return new PercentOrAbsolute(`${percent}%`);
    }
    /**
     * Absolute number.
     */
    static absolute(num) {
        return new PercentOrAbsolute(num);
    }
    isZero() {
        return this.value === PercentOrAbsolute.absolute(0).value || this.value === PercentOrAbsolute.percent(0).value;
    }
}
exports.PercentOrAbsolute = PercentOrAbsolute;
_b = JSII_RTTI_SYMBOL_1;
PercentOrAbsolute[_b] = { fqn: "cdk8s-plus-22.PercentOrAbsolute", version: "2.0.0-beta.10" };
/**
 * Deployment strategies.
 */
class DeploymentStrategy {
    constructor(strategy) {
        this.strategy = strategy;
    }
    /**
     * All existing Pods are killed before new ones are created.
     *
     * @see https://kubernetes.io/docs/concepts/workloads/controllers/deployment/#recreate-deployment
     */
    static recreate() {
        return new DeploymentStrategy({
            type: 'Recreate',
        });
    }
    static rollingUpdate(options = {}) {
        var _d, _e;
        const maxSurge = (_d = options.maxSurge) !== null && _d !== void 0 ? _d : PercentOrAbsolute.percent(25);
        const maxUnavailable = (_e = options.maxUnavailable) !== null && _e !== void 0 ? _e : PercentOrAbsolute.percent(25);
        if (maxSurge.isZero() && maxUnavailable.isZero()) {
            throw new Error('\'maxSurge\' and \'maxUnavailable\' cannot be both zero');
        }
        return new DeploymentStrategy({
            type: 'RollingUpdate',
            rollingUpdate: { maxSurge, maxUnavailable },
        });
    }
    /**
     * @internal
     */
    _toKube() {
        return this.strategy;
    }
}
exports.DeploymentStrategy = DeploymentStrategy;
_c = JSII_RTTI_SYMBOL_1;
DeploymentStrategy[_c] = { fqn: "cdk8s-plus-22.DeploymentStrategy", version: "2.0.0-beta.10" };
//# sourceMappingURL=data:application/json;base64,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