"use strict";
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
Object.defineProperty(exports, "__esModule", { value: true });
exports.sonarqubeScanner = exports.createSonarqubeProject = exports.generateSonarqubeReports = void 0;
const cube = (path, action = "GET", args) => `curl -X ${action} -u $SONARQUBE_TOKEN: "$SONARQUBE_ENDPOINT${path}" ${args || ""}`;
const setupSonarqubeQualityGates = (defaultProfileOrGateName, specificProfileOrGateName) => [
    `export DEFAULT_GATE=\`${cube(`/api/qualitygates/search?gateName=${defaultProfileOrGateName}`)}\``,
    `export SPECIFIC_GATE=\`${cube(`/api/qualitygates/search?gateName=${specificProfileOrGateName}`)}\``,
    `if [[ "$(echo $SPECIFIC_GATE | jq .errors)" == "null" && "$(echo $SPECIFIC_GATE | jq '.results | length')" -gt 0 ]]; then export GATE_NAME=${specificProfileOrGateName}; else export GATE_NAME=${defaultProfileOrGateName}; fi`,
    `${cube("/api/qualitygates/select?projectKey=$PROJECT_NAME&gateName=$GATE_NAME", "POST")}`,
];
const setupSonarqubeQualityProfiles = (defaultProfileOrGateName, specificProfileOrGateName) => [
    `export DEFAULT_PROFILE=\`${cube(`/api/qualityprofiles/search?qualityProfile=${defaultProfileOrGateName}`)} | jq .profiles\``,
    `export SPECIFIC_PROFILE=\`${cube(`/api/qualityprofiles/search?qualityProfile=${specificProfileOrGateName}`)} | jq .profiles\``,
    `export MERGED_PROFILES=\`jq --argjson arr1 "$DEFAULT_PROFILE" --argjson arr2 "$SPECIFIC_PROFILE" -n '$arr1 + $arr2 | group_by(.language) | map(.[-1])'\``,
    `echo $MERGED_PROFILES | jq -c '.[]' | while read i; do ${cube("/api/qualityprofiles/add_project?project=$PROJECT_NAME&language=`echo $i | jq -r .language`&qualityProfile=`echo $i | jq -r .name`", "POST")}; done`,
];
const setupSonarqubePermissions = (authorizedGroup) => !authorizedGroup
    ? []
    : [
        "admin",
        "codeviewer",
        "issueadmin",
        "securityhotspotadmin",
        "scan",
        "user",
    ].map((p) => `${cube(`/api/permissions/add_group?projectKey=$PROJECT_NAME&groupName=${authorizedGroup}&permission=${p}`, "POST")}`);
const setupSonarqubeProject = ({ sonarqubeTags, sonarqubeAuthorizedGroup, sonarqubeDefaultProfileOrGateName, sonarqubeSpecificProfileOrGateName, }) => {
    return [
        ...setupSonarqubePermissions(sonarqubeAuthorizedGroup),
        `${cube("/api/project_branches/rename?project=$PROJECT_NAME&name=mainline", "POST")}`,
        `${cube(`/api/project_tags/set?project=$PROJECT_NAME&tags=${[
            sonarqubeAuthorizedGroup,
            ...(sonarqubeTags || []),
        ].join(",")}`, "POST")}`,
        ...setupSonarqubeQualityProfiles(sonarqubeDefaultProfileOrGateName, sonarqubeSpecificProfileOrGateName),
        ...setupSonarqubeQualityGates(sonarqubeDefaultProfileOrGateName, sonarqubeSpecificProfileOrGateName),
    ].join(";");
};
exports.generateSonarqubeReports = () => [
    cube("/api/bitegarden/report/pdf_issues_breakdown?resource=$PROJECT_NAME&branch=mainline", "GET", "--output reports/prototype-issues-report.pdf"),
    cube("/api/bitegarden/report/pdf?resource=$PROJECT_NAME&branch=mainline", "GET", "--output reports/prototype-executive-report.pdf"),
    cube("/api/security_reports/download?project=$PROJECT_NAME", "GET", "--output reports/prototype-security-report.pdf"),
];
exports.createSonarqubeProject = (props) => [
    `CREATE_PROJECT_OUTPUT=\`${cube("/api/projects/create?name=$PROJECT_NAME&project=$PROJECT_NAME&visibility=private", "POST")}\``,
    `if [[ "$(echo $CREATE_PROJECT_OUTPUT | jq .errors)" == "null" ]]; then ${setupSonarqubeProject(props)}; fi;`,
];
exports.sonarqubeScanner = (excludeGlobsForScan) => [
    "npx sonarqube-scanner -Dsonar.login=$SONARQUBE_TOKEN",
    "-Dsonar.projectKey=$PROJECT_NAME",
    "-Dsonar.projectName=$PROJECT_NAME",
    "-Dsonar.projectVersion=`echo $RESOLVED_SOURCE_VERSION | cut -c1-7`",
    "-Dsonar.branch.name=mainline",
    "-Dsonar.host.url=$SONARQUBE_ENDPOINT",
    "-Dsonar.cfn.nag.reportFiles=reports/cfn-nag-report.json",
    "-Dsonar.dependencyCheck.htmlReportPath=reports/dependency-check-report.html",
    "-Dsonar.javascript.lcov.reportPaths=**/coverage/lcov.info",
    "-Dsonar.clover.reportPath=**/coverage/clover.xml",
    `-Dsonar.exclusions="**/reports/**,**/coverage/**${excludeGlobsForScan ? `,${excludeGlobsForScan.join(",")}` : ""}"`,
    "-Dsonar.sources=.",
].join(" ");
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic29uYXJxdWJlLWNvbW1hbmRzLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vc3JjL2NvZGVfc2Nhbm5lci9zb25hcnF1YmUtY29tbWFuZHMudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBOzs7Ozs7Ozs7Ozs7Ozt3SEFjd0g7OztBQUl4SCxNQUFNLElBQUksR0FBRyxDQUFDLElBQVksRUFBRSxTQUFpQixLQUFLLEVBQUUsSUFBYSxFQUFFLEVBQUUsQ0FDbkUsV0FBVyxNQUFNLDZDQUE2QyxJQUFJLEtBQ2hFLElBQUksSUFBSSxFQUNWLEVBQUUsQ0FBQztBQUVMLE1BQU0sMEJBQTBCLEdBQUcsQ0FDakMsd0JBQWdDLEVBQ2hDLHlCQUFrQyxFQUNsQyxFQUFFLENBQUM7SUFDSCx5QkFBeUIsSUFBSSxDQUMzQixxQ0FBcUMsd0JBQXdCLEVBQUUsQ0FDaEUsSUFBSTtJQUNMLDBCQUEwQixJQUFJLENBQzVCLHFDQUFxQyx5QkFBeUIsRUFBRSxDQUNqRSxJQUFJO0lBQ0wsOElBQThJLHlCQUF5QiwyQkFBMkIsd0JBQXdCLE1BQU07SUFDaE8sR0FBRyxJQUFJLENBQ0wsdUVBQXVFLEVBQ3ZFLE1BQU0sQ0FDUCxFQUFFO0NBQ0osQ0FBQztBQUVGLE1BQU0sNkJBQTZCLEdBQUcsQ0FDcEMsd0JBQWdDLEVBQ2hDLHlCQUFrQyxFQUNsQyxFQUFFLENBQUM7SUFDSCw0QkFBNEIsSUFBSSxDQUM5Qiw4Q0FBOEMsd0JBQXdCLEVBQUUsQ0FDekUsbUJBQW1CO0lBQ3BCLDZCQUE2QixJQUFJLENBQy9CLDhDQUE4Qyx5QkFBeUIsRUFBRSxDQUMxRSxtQkFBbUI7SUFDcEIsMEpBQTBKO0lBQzFKLDBEQUEwRCxJQUFJLENBQzVELG9JQUFvSSxFQUNwSSxNQUFNLENBQ1AsUUFBUTtDQUNWLENBQUM7QUFFRixNQUFNLHlCQUF5QixHQUFHLENBQUMsZUFBd0IsRUFBRSxFQUFFLENBQzdELENBQUMsZUFBZTtJQUNkLENBQUMsQ0FBQyxFQUFFO0lBQ0osQ0FBQyxDQUFDO1FBQ0UsT0FBTztRQUNQLFlBQVk7UUFDWixZQUFZO1FBQ1osc0JBQXNCO1FBQ3RCLE1BQU07UUFDTixNQUFNO0tBQ1AsQ0FBQyxHQUFHLENBQ0gsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUNKLEdBQUcsSUFBSSxDQUNMLGlFQUFpRSxlQUFlLGVBQWUsQ0FBQyxFQUFFLEVBQ2xHLE1BQU0sQ0FDUCxFQUFFLENBQ04sQ0FBQztBQUVSLE1BQU0scUJBQXFCLEdBQUcsQ0FBQyxFQUM3QixhQUFhLEVBQ2Isd0JBQXdCLEVBQ3hCLGlDQUFpQyxFQUNqQyxrQ0FBa0MsR0FDWixFQUFFLEVBQUU7SUFDMUIsT0FBTztRQUNMLEdBQUcseUJBQXlCLENBQUMsd0JBQXdCLENBQUM7UUFDdEQsR0FBRyxJQUFJLENBQ0wsa0VBQWtFLEVBQ2xFLE1BQU0sQ0FDUCxFQUFFO1FBQ0gsR0FBRyxJQUFJLENBQ0wsb0RBQW9EO1lBQ2xELHdCQUF3QjtZQUN4QixHQUFHLENBQUMsYUFBYSxJQUFJLEVBQUUsQ0FBQztTQUN6QixDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxFQUNiLE1BQU0sQ0FDUCxFQUFFO1FBQ0gsR0FBRyw2QkFBNkIsQ0FDOUIsaUNBQWlDLEVBQ2pDLGtDQUFrQyxDQUNuQztRQUNELEdBQUcsMEJBQTBCLENBQzNCLGlDQUFpQyxFQUNqQyxrQ0FBa0MsQ0FDbkM7S0FDRixDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztBQUNkLENBQUMsQ0FBQztBQUVXLFFBQUEsd0JBQXdCLEdBQUcsR0FBRyxFQUFFLENBQUM7SUFDNUMsSUFBSSxDQUNGLG9GQUFvRixFQUNwRixLQUFLLEVBQ0wsOENBQThDLENBQy9DO0lBQ0QsSUFBSSxDQUNGLG1FQUFtRSxFQUNuRSxLQUFLLEVBQ0wsaURBQWlELENBQ2xEO0lBQ0QsSUFBSSxDQUNGLHNEQUFzRCxFQUN0RCxLQUFLLEVBQ0wsZ0RBQWdELENBQ2pEO0NBQ0YsQ0FBQztBQUVXLFFBQUEsc0JBQXNCLEdBQUcsQ0FBQyxLQUE0QixFQUFFLEVBQUUsQ0FBQztJQUN0RSwyQkFBMkIsSUFBSSxDQUM3QixrRkFBa0YsRUFDbEYsTUFBTSxDQUNQLElBQUk7SUFDTCwwRUFBMEUscUJBQXFCLENBQzdGLEtBQUssQ0FDTixPQUFPO0NBQ1QsQ0FBQztBQUVXLFFBQUEsZ0JBQWdCLEdBQUcsQ0FBQyxtQkFBOEIsRUFBRSxFQUFFLENBQ2pFO0lBQ0Usc0RBQXNEO0lBQ3RELGtDQUFrQztJQUNsQyxtQ0FBbUM7SUFDbkMsb0VBQW9FO0lBQ3BFLDhCQUE4QjtJQUM5QixzQ0FBc0M7SUFDdEMseURBQXlEO0lBQ3pELDZFQUE2RTtJQUM3RSwyREFBMkQ7SUFDM0Qsa0RBQWtEO0lBQ2xELG1EQUNFLG1CQUFtQixDQUFDLENBQUMsQ0FBQyxJQUFJLG1CQUFtQixDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxFQUM5RCxHQUFHO0lBQ0gsbUJBQW1CO0NBQ3BCLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKlxuIENvcHlyaWdodCBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuXG4gTGljZW5zZWQgdW5kZXIgdGhlIEFwYWNoZSBMaWNlbnNlLCBWZXJzaW9uIDIuMCAodGhlIFwiTGljZW5zZVwiKS5cbiBZb3UgbWF5IG5vdCB1c2UgdGhpcyBmaWxlIGV4Y2VwdCBpbiBjb21wbGlhbmNlIHdpdGggdGhlIExpY2Vuc2UuXG4gWW91IG1heSBvYnRhaW4gYSBjb3B5IG9mIHRoZSBMaWNlbnNlIGF0XG5cbiBodHRwOi8vd3d3LmFwYWNoZS5vcmcvbGljZW5zZXMvTElDRU5TRS0yLjBcblxuIFVubGVzcyByZXF1aXJlZCBieSBhcHBsaWNhYmxlIGxhdyBvciBhZ3JlZWQgdG8gaW4gd3JpdGluZywgc29mdHdhcmVcbiBkaXN0cmlidXRlZCB1bmRlciB0aGUgTGljZW5zZSBpcyBkaXN0cmlidXRlZCBvbiBhbiBcIkFTIElTXCIgQkFTSVMsXG4gV0lUSE9VVCBXQVJSQU5USUVTIE9SIENPTkRJVElPTlMgT0YgQU5ZIEtJTkQsIGVpdGhlciBleHByZXNzIG9yIGltcGxpZWQuXG4gU2VlIHRoZSBMaWNlbnNlIGZvciB0aGUgc3BlY2lmaWMgbGFuZ3VhZ2UgZ292ZXJuaW5nIHBlcm1pc3Npb25zIGFuZFxuIGxpbWl0YXRpb25zIHVuZGVyIHRoZSBMaWNlbnNlLlxuICoqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqICovXG5cbmltcG9ydCB7IFNvbmFyQ29kZVNjYW5uZXJQcm9wcyB9IGZyb20gXCIuL3NvbmFyLWNvZGUtc2Nhbm5lclwiO1xuXG5jb25zdCBjdWJlID0gKHBhdGg6IHN0cmluZywgYWN0aW9uOiBzdHJpbmcgPSBcIkdFVFwiLCBhcmdzPzogc3RyaW5nKSA9PlxuICBgY3VybCAtWCAke2FjdGlvbn0gLXUgJFNPTkFSUVVCRV9UT0tFTjogXCIkU09OQVJRVUJFX0VORFBPSU5UJHtwYXRofVwiICR7XG4gICAgYXJncyB8fCBcIlwiXG4gIH1gO1xuXG5jb25zdCBzZXR1cFNvbmFycXViZVF1YWxpdHlHYXRlcyA9IChcbiAgZGVmYXVsdFByb2ZpbGVPckdhdGVOYW1lOiBzdHJpbmcsXG4gIHNwZWNpZmljUHJvZmlsZU9yR2F0ZU5hbWU/OiBzdHJpbmdcbikgPT4gW1xuICBgZXhwb3J0IERFRkFVTFRfR0FURT1cXGAke2N1YmUoXG4gICAgYC9hcGkvcXVhbGl0eWdhdGVzL3NlYXJjaD9nYXRlTmFtZT0ke2RlZmF1bHRQcm9maWxlT3JHYXRlTmFtZX1gXG4gICl9XFxgYCxcbiAgYGV4cG9ydCBTUEVDSUZJQ19HQVRFPVxcYCR7Y3ViZShcbiAgICBgL2FwaS9xdWFsaXR5Z2F0ZXMvc2VhcmNoP2dhdGVOYW1lPSR7c3BlY2lmaWNQcm9maWxlT3JHYXRlTmFtZX1gXG4gICl9XFxgYCxcbiAgYGlmIFtbIFwiJChlY2hvICRTUEVDSUZJQ19HQVRFIHwganEgLmVycm9ycylcIiA9PSBcIm51bGxcIiAmJiBcIiQoZWNobyAkU1BFQ0lGSUNfR0FURSB8IGpxICcucmVzdWx0cyB8IGxlbmd0aCcpXCIgLWd0IDAgXV07IHRoZW4gZXhwb3J0IEdBVEVfTkFNRT0ke3NwZWNpZmljUHJvZmlsZU9yR2F0ZU5hbWV9OyBlbHNlIGV4cG9ydCBHQVRFX05BTUU9JHtkZWZhdWx0UHJvZmlsZU9yR2F0ZU5hbWV9OyBmaWAsXG4gIGAke2N1YmUoXG4gICAgXCIvYXBpL3F1YWxpdHlnYXRlcy9zZWxlY3Q/cHJvamVjdEtleT0kUFJPSkVDVF9OQU1FJmdhdGVOYW1lPSRHQVRFX05BTUVcIixcbiAgICBcIlBPU1RcIlxuICApfWAsXG5dO1xuXG5jb25zdCBzZXR1cFNvbmFycXViZVF1YWxpdHlQcm9maWxlcyA9IChcbiAgZGVmYXVsdFByb2ZpbGVPckdhdGVOYW1lOiBzdHJpbmcsXG4gIHNwZWNpZmljUHJvZmlsZU9yR2F0ZU5hbWU/OiBzdHJpbmdcbikgPT4gW1xuICBgZXhwb3J0IERFRkFVTFRfUFJPRklMRT1cXGAke2N1YmUoXG4gICAgYC9hcGkvcXVhbGl0eXByb2ZpbGVzL3NlYXJjaD9xdWFsaXR5UHJvZmlsZT0ke2RlZmF1bHRQcm9maWxlT3JHYXRlTmFtZX1gXG4gICl9IHwganEgLnByb2ZpbGVzXFxgYCxcbiAgYGV4cG9ydCBTUEVDSUZJQ19QUk9GSUxFPVxcYCR7Y3ViZShcbiAgICBgL2FwaS9xdWFsaXR5cHJvZmlsZXMvc2VhcmNoP3F1YWxpdHlQcm9maWxlPSR7c3BlY2lmaWNQcm9maWxlT3JHYXRlTmFtZX1gXG4gICl9IHwganEgLnByb2ZpbGVzXFxgYCxcbiAgYGV4cG9ydCBNRVJHRURfUFJPRklMRVM9XFxganEgLS1hcmdqc29uIGFycjEgXCIkREVGQVVMVF9QUk9GSUxFXCIgLS1hcmdqc29uIGFycjIgXCIkU1BFQ0lGSUNfUFJPRklMRVwiIC1uICckYXJyMSArICRhcnIyIHwgZ3JvdXBfYnkoLmxhbmd1YWdlKSB8IG1hcCguWy0xXSknXFxgYCxcbiAgYGVjaG8gJE1FUkdFRF9QUk9GSUxFUyB8IGpxIC1jICcuW10nIHwgd2hpbGUgcmVhZCBpOyBkbyAke2N1YmUoXG4gICAgXCIvYXBpL3F1YWxpdHlwcm9maWxlcy9hZGRfcHJvamVjdD9wcm9qZWN0PSRQUk9KRUNUX05BTUUmbGFuZ3VhZ2U9YGVjaG8gJGkgfCBqcSAtciAubGFuZ3VhZ2VgJnF1YWxpdHlQcm9maWxlPWBlY2hvICRpIHwganEgLXIgLm5hbWVgXCIsXG4gICAgXCJQT1NUXCJcbiAgKX07IGRvbmVgLFxuXTtcblxuY29uc3Qgc2V0dXBTb25hcnF1YmVQZXJtaXNzaW9ucyA9IChhdXRob3JpemVkR3JvdXA/OiBzdHJpbmcpID0+XG4gICFhdXRob3JpemVkR3JvdXBcbiAgICA/IFtdXG4gICAgOiBbXG4gICAgICAgIFwiYWRtaW5cIixcbiAgICAgICAgXCJjb2Rldmlld2VyXCIsXG4gICAgICAgIFwiaXNzdWVhZG1pblwiLFxuICAgICAgICBcInNlY3VyaXR5aG90c3BvdGFkbWluXCIsXG4gICAgICAgIFwic2NhblwiLFxuICAgICAgICBcInVzZXJcIixcbiAgICAgIF0ubWFwKFxuICAgICAgICAocCkgPT5cbiAgICAgICAgICBgJHtjdWJlKFxuICAgICAgICAgICAgYC9hcGkvcGVybWlzc2lvbnMvYWRkX2dyb3VwP3Byb2plY3RLZXk9JFBST0pFQ1RfTkFNRSZncm91cE5hbWU9JHthdXRob3JpemVkR3JvdXB9JnBlcm1pc3Npb249JHtwfWAsXG4gICAgICAgICAgICBcIlBPU1RcIlxuICAgICAgICAgICl9YFxuICAgICAgKTtcblxuY29uc3Qgc2V0dXBTb25hcnF1YmVQcm9qZWN0ID0gKHtcbiAgc29uYXJxdWJlVGFncyxcbiAgc29uYXJxdWJlQXV0aG9yaXplZEdyb3VwLFxuICBzb25hcnF1YmVEZWZhdWx0UHJvZmlsZU9yR2F0ZU5hbWUsXG4gIHNvbmFycXViZVNwZWNpZmljUHJvZmlsZU9yR2F0ZU5hbWUsXG59OiBTb25hckNvZGVTY2FubmVyUHJvcHMpID0+IHtcbiAgcmV0dXJuIFtcbiAgICAuLi5zZXR1cFNvbmFycXViZVBlcm1pc3Npb25zKHNvbmFycXViZUF1dGhvcml6ZWRHcm91cCksXG4gICAgYCR7Y3ViZShcbiAgICAgIFwiL2FwaS9wcm9qZWN0X2JyYW5jaGVzL3JlbmFtZT9wcm9qZWN0PSRQUk9KRUNUX05BTUUmbmFtZT1tYWlubGluZVwiLFxuICAgICAgXCJQT1NUXCJcbiAgICApfWAsXG4gICAgYCR7Y3ViZShcbiAgICAgIGAvYXBpL3Byb2plY3RfdGFncy9zZXQ/cHJvamVjdD0kUFJPSkVDVF9OQU1FJnRhZ3M9JHtbXG4gICAgICAgIHNvbmFycXViZUF1dGhvcml6ZWRHcm91cCxcbiAgICAgICAgLi4uKHNvbmFycXViZVRhZ3MgfHwgW10pLFxuICAgICAgXS5qb2luKFwiLFwiKX1gLFxuICAgICAgXCJQT1NUXCJcbiAgICApfWAsXG4gICAgLi4uc2V0dXBTb25hcnF1YmVRdWFsaXR5UHJvZmlsZXMoXG4gICAgICBzb25hcnF1YmVEZWZhdWx0UHJvZmlsZU9yR2F0ZU5hbWUsXG4gICAgICBzb25hcnF1YmVTcGVjaWZpY1Byb2ZpbGVPckdhdGVOYW1lXG4gICAgKSxcbiAgICAuLi5zZXR1cFNvbmFycXViZVF1YWxpdHlHYXRlcyhcbiAgICAgIHNvbmFycXViZURlZmF1bHRQcm9maWxlT3JHYXRlTmFtZSxcbiAgICAgIHNvbmFycXViZVNwZWNpZmljUHJvZmlsZU9yR2F0ZU5hbWVcbiAgICApLFxuICBdLmpvaW4oXCI7XCIpO1xufTtcblxuZXhwb3J0IGNvbnN0IGdlbmVyYXRlU29uYXJxdWJlUmVwb3J0cyA9ICgpID0+IFtcbiAgY3ViZShcbiAgICBcIi9hcGkvYml0ZWdhcmRlbi9yZXBvcnQvcGRmX2lzc3Vlc19icmVha2Rvd24/cmVzb3VyY2U9JFBST0pFQ1RfTkFNRSZicmFuY2g9bWFpbmxpbmVcIixcbiAgICBcIkdFVFwiLFxuICAgIFwiLS1vdXRwdXQgcmVwb3J0cy9wcm90b3R5cGUtaXNzdWVzLXJlcG9ydC5wZGZcIlxuICApLFxuICBjdWJlKFxuICAgIFwiL2FwaS9iaXRlZ2FyZGVuL3JlcG9ydC9wZGY/cmVzb3VyY2U9JFBST0pFQ1RfTkFNRSZicmFuY2g9bWFpbmxpbmVcIixcbiAgICBcIkdFVFwiLFxuICAgIFwiLS1vdXRwdXQgcmVwb3J0cy9wcm90b3R5cGUtZXhlY3V0aXZlLXJlcG9ydC5wZGZcIlxuICApLFxuICBjdWJlKFxuICAgIFwiL2FwaS9zZWN1cml0eV9yZXBvcnRzL2Rvd25sb2FkP3Byb2plY3Q9JFBST0pFQ1RfTkFNRVwiLFxuICAgIFwiR0VUXCIsXG4gICAgXCItLW91dHB1dCByZXBvcnRzL3Byb3RvdHlwZS1zZWN1cml0eS1yZXBvcnQucGRmXCJcbiAgKSxcbl07XG5cbmV4cG9ydCBjb25zdCBjcmVhdGVTb25hcnF1YmVQcm9qZWN0ID0gKHByb3BzOiBTb25hckNvZGVTY2FubmVyUHJvcHMpID0+IFtcbiAgYENSRUFURV9QUk9KRUNUX09VVFBVVD1cXGAke2N1YmUoXG4gICAgXCIvYXBpL3Byb2plY3RzL2NyZWF0ZT9uYW1lPSRQUk9KRUNUX05BTUUmcHJvamVjdD0kUFJPSkVDVF9OQU1FJnZpc2liaWxpdHk9cHJpdmF0ZVwiLFxuICAgIFwiUE9TVFwiXG4gICl9XFxgYCxcbiAgYGlmIFtbIFwiJChlY2hvICRDUkVBVEVfUFJPSkVDVF9PVVRQVVQgfCBqcSAuZXJyb3JzKVwiID09IFwibnVsbFwiIF1dOyB0aGVuICR7c2V0dXBTb25hcnF1YmVQcm9qZWN0KFxuICAgIHByb3BzXG4gICl9OyBmaTtgLFxuXTtcblxuZXhwb3J0IGNvbnN0IHNvbmFycXViZVNjYW5uZXIgPSAoZXhjbHVkZUdsb2JzRm9yU2Nhbj86IHN0cmluZ1tdKSA9PlxuICBbXG4gICAgXCJucHggc29uYXJxdWJlLXNjYW5uZXIgLURzb25hci5sb2dpbj0kU09OQVJRVUJFX1RPS0VOXCIsXG4gICAgXCItRHNvbmFyLnByb2plY3RLZXk9JFBST0pFQ1RfTkFNRVwiLFxuICAgIFwiLURzb25hci5wcm9qZWN0TmFtZT0kUFJPSkVDVF9OQU1FXCIsXG4gICAgXCItRHNvbmFyLnByb2plY3RWZXJzaW9uPWBlY2hvICRSRVNPTFZFRF9TT1VSQ0VfVkVSU0lPTiB8IGN1dCAtYzEtN2BcIixcbiAgICBcIi1Ec29uYXIuYnJhbmNoLm5hbWU9bWFpbmxpbmVcIixcbiAgICBcIi1Ec29uYXIuaG9zdC51cmw9JFNPTkFSUVVCRV9FTkRQT0lOVFwiLFxuICAgIFwiLURzb25hci5jZm4ubmFnLnJlcG9ydEZpbGVzPXJlcG9ydHMvY2ZuLW5hZy1yZXBvcnQuanNvblwiLFxuICAgIFwiLURzb25hci5kZXBlbmRlbmN5Q2hlY2suaHRtbFJlcG9ydFBhdGg9cmVwb3J0cy9kZXBlbmRlbmN5LWNoZWNrLXJlcG9ydC5odG1sXCIsXG4gICAgXCItRHNvbmFyLmphdmFzY3JpcHQubGNvdi5yZXBvcnRQYXRocz0qKi9jb3ZlcmFnZS9sY292LmluZm9cIixcbiAgICBcIi1Ec29uYXIuY2xvdmVyLnJlcG9ydFBhdGg9KiovY292ZXJhZ2UvY2xvdmVyLnhtbFwiLFxuICAgIGAtRHNvbmFyLmV4Y2x1c2lvbnM9XCIqKi9yZXBvcnRzLyoqLCoqL2NvdmVyYWdlLyoqJHtcbiAgICAgIGV4Y2x1ZGVHbG9ic0ZvclNjYW4gPyBgLCR7ZXhjbHVkZUdsb2JzRm9yU2Nhbi5qb2luKFwiLFwiKX1gIDogXCJcIlxuICAgIH1cImAsXG4gICAgXCItRHNvbmFyLnNvdXJjZXM9LlwiLFxuICBdLmpvaW4oXCIgXCIpO1xuIl19