"use strict";
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cdk_nag_1 = require("cdk-nag");
const aws_prototyping_1 = require("../src/packs/aws-prototyping");
const expectedErrors = [
    "AwsPrototyping-AutoScalingLaunchConfigPublicIpDisabled",
    "AwsPrototyping-Cloud9InstanceNoIngressSystemsManager",
    "AwsPrototyping-CloudFrontDistributionGeoRestrictions",
    "AwsPrototyping-CloudFrontDistributionS3OriginAccessIdentity",
    "AwsPrototyping-CodeBuildProjectEnvVarAwsCred",
    "AwsPrototyping-CognitoUserPoolNoUnauthenticatedLogins",
    "AwsPrototyping-CognitoUserPoolStrongPasswordPolicy",
    "AwsPrototyping-DMSReplicationNotPublic",
    "AwsPrototyping-DocumentDBCredentialsInSecretsManager",
    "AwsPrototyping-EC2InstancesInVPC",
    "AwsPrototyping-EC2RestrictedCommonPorts",
    "AwsPrototyping-EC2RestrictedInbound",
    "AwsPrototyping-EC2RestrictedSSH",
    "AwsPrototyping-ECROpenAccess",
    "AwsPrototyping-EKSClusterNoEndpointPublicAccess",
    "AwsPrototyping-ElastiCacheClusterInVPC",
    "AwsPrototyping-ElasticBeanstalkManagedUpdatesEnabled",
    "AwsPrototyping-EventBusOpenAccess",
    "AwsPrototyping-IAMPolicyNoStatementsWithAdminAccess",
    "AwsPrototyping-LambdaFunctionPublicAccessProhibited",
    "AwsPrototyping-LambdaFunctionUrlAuth",
    "AwsPrototyping-NeptuneClusterAutomaticMinorVersionUpgrade",
    "AwsPrototyping-OpenSearchInVPCOnly",
    "AwsPrototyping-OpenSearchNoUnsignedOrAnonymousAccess",
    "AwsPrototyping-RDSAutomaticMinorVersionUpgradeEnabled",
    "AwsPrototyping-RDSInstancePublicAccess",
    "AwsPrototyping-RDSRestrictedInbound",
    "AwsPrototyping-RedshiftClusterInVPC",
    "AwsPrototyping-RedshiftClusterPublicAccess",
    "AwsPrototyping-RedshiftClusterVersionUpgrade",
    "AwsPrototyping-S3BucketLevelPublicAccessProhibited",
    "AwsPrototyping-S3BucketPublicReadProhibited",
    "AwsPrototyping-S3BucketPublicWriteProhibited",
    "AwsPrototyping-S3WebBucketOAIAccess",
    "AwsPrototyping-VPCSubnetAutoAssignPublicIpDisabled",
];
const expectedWarnings = [
    "AwsPrototyping-APIGWAssociatedWithWAF",
    "AwsPrototyping-APIGWAuthorization",
    "AwsPrototyping-APIGWRequestValidation",
    "AwsPrototyping-AthenaWorkgroupEncryptedQueryResults",
    "AwsPrototyping-CloudFrontDistributionHttpsViewerNoOutdatedSSL",
    "AwsPrototyping-CloudFrontDistributionNoOutdatedSSL",
    "AwsPrototyping-CloudFrontDistributionWAFIntegration",
    "AwsPrototyping-CodeBuildProjectKMSEncryptedArtifacts",
    "AwsPrototyping-CodeBuildProjectManagedImages",
    "AwsPrototyping-CodeBuildProjectPrivilegedModeDisabled",
    "AwsPrototyping-CognitoUserPoolAdvancedSecurityModeEnforced",
    "AwsPrototyping-CognitoUserPoolAPIGWAuthorizer",
    "AwsPrototyping-CognitoUserPoolMFA",
    "AwsPrototyping-DocumentDBClusterEncryptionAtRest",
    "AwsPrototyping-DocumentDBClusterNonDefaultPort",
    "AwsPrototyping-DAXEncrypted",
    "AwsPrototyping-EC2EBSVolumeEncrypted",
    "AwsPrototyping-EC2InstanceNoPublicIp",
    "AwsPrototyping-EC2InstanceProfileAttached",
    "AwsPrototyping-EC2SecurityGroupDescription",
    "AwsPrototyping-ECSTaskDefinitionContainerLogging",
    "AwsPrototyping-ECSTaskDefinitionNoEnvironmentVariables",
    "AwsPrototyping-EFSEncrypted",
    "AwsPrototyping-EKSClusterControlPlaneLogs",
    "AwsPrototyping-ElastiCacheClusterNonDefaultPort",
    "AwsPrototyping-ElastiCacheRedisClusterEncryption",
    "AwsPrototyping-ElastiCacheRedisClusterRedisAuth",
    "AwsPrototyping-ElasticBeanstalkEC2InstanceLogsToS3",
    "AwsPrototyping-ElasticBeanstalkVPCSpecified",
    "AwsPrototyping-ALBHttpToHttpsRedirection",
    "AwsPrototyping-ELBTlsHttpsListenersOnly",
    "AwsPrototyping-EMRAuthEC2KeyPairOrKerberos",
    "AwsPrototyping-EMREncryptionInTransit",
    "AwsPrototyping-EMRLocalDiskEncryption",
    "AwsPrototyping-GlueEncryptedCloudWatchLogs",
    "AwsPrototyping-GlueJobBookmarkEncrypted",
    "AwsPrototyping-IAMNoManagedPolicies",
    "AwsPrototyping-IAMNoWildcardPermissions",
    "AwsPrototyping-IAMPolicyNoStatementsWithFullAccess",
    "AwsPrototyping-KinesisDataFirehoseSSE",
    "AwsPrototyping-KinesisDataStreamDefaultKeyWhenSSE",
    "AwsPrototyping-KinesisDataStreamSSE",
    "AwsPrototyping-KMSBackingKeyRotationEnabled",
    "AwsPrototyping-LambdaLatestVersion",
    "AwsPrototyping-MediaStoreContainerCORSPolicy",
    "AwsPrototyping-MediaStoreContainerHasContainerPolicy",
    "AwsPrototyping-MediaStoreContainerSSLRequestsOnly",
    "AwsPrototyping-MSKBrokerToBrokerTLS",
    "AwsPrototyping-MSKClientToBrokerTLS",
    "AwsPrototyping-NeptuneClusterEncryptionAtRest",
    "AwsPrototyping-NeptuneClusterIAMAuth",
    "AwsPrototyping-OpenSearchAllowlistedIPs",
    "AwsPrototyping-OpenSearchEncryptedAtRest",
    "AwsPrototyping-OpenSearchNodeToNodeEncryption",
    "AwsPrototyping-QuicksightSSLConnections",
    "AwsPrototyping-AuroraMySQLPostgresIAMAuth",
    "AwsPrototyping-RDSNonDefaultPort",
    "AwsPrototyping-RDSStorageEncrypted",
    "AwsPrototyping-RedshiftClusterEncryptionAtRest",
    "AwsPrototyping-RedshiftClusterNonDefaultPort",
    "AwsPrototyping-RedshiftClusterNonDefaultUsername",
    "AwsPrototyping-RedshiftRequireTlsSSL",
    "AwsPrototyping-S3BucketLoggingEnabled",
    "AwsPrototyping-S3BucketSSLRequestsOnly",
    "AwsPrototyping-SageMakerNotebookInstanceKMSKeyConfigured",
    "AwsPrototyping-SageMakerNotebookInVPC",
    "AwsPrototyping-SageMakerNotebookNoDirectInternetAccess",
    "AwsPrototyping-SecretsManagerRotationEnabled",
    "AwsPrototyping-SNSEncryptedKMS",
    "AwsPrototyping-SNSTopicSSLPublishOnly",
    "AwsPrototyping-SQSQueueSSE",
    "AwsPrototyping-SQSQueueSSLRequestsOnly",
    "AwsPrototyping-TimestreamDatabaseCustomerManagedKey",
    "AwsPrototyping-VPCNoNACLs",
];
describe("Check NagPack Details", () => {
    describe("AwsPrototyping-CDK", () => {
        class AwsPrototypingChecksExtended extends aws_prototyping_1.AwsPrototypingChecks {
            constructor() {
                super(...arguments);
                this.actualWarnings = new Array();
                this.actualErrors = new Array();
            }
            applyRule(params) {
                const ruleSuffix = params.ruleSuffixOverride
                    ? params.ruleSuffixOverride
                    : params.rule.name;
                const ruleId = `${pack.readPackName}-${ruleSuffix}`;
                if (params.level === cdk_nag_1.NagMessageLevel.WARN) {
                    this.actualWarnings.push(ruleId);
                }
                else {
                    this.actualErrors.push(ruleId);
                }
            }
        }
        const pack = new AwsPrototypingChecksExtended();
        test("Pack Name is correct", () => {
            expect(pack.readPackName).toStrictEqual("AwsPrototyping");
        });
        test("Pack contains expected warning and error rules", () => {
            jest.spyOn(pack, "applyRule");
            const stack = new aws_cdk_lib_1.Stack();
            aws_cdk_lib_1.Aspects.of(stack).add(pack);
            new aws_cdk_lib_1.CfnResource(stack, "rTestResource", { type: "foo" });
            assert_1.SynthUtils.synthesize(stack).messages;
            expect(pack.actualWarnings.sort()).toEqual(expectedWarnings.sort());
            expect(pack.actualErrors.sort()).toEqual(expectedErrors.sort());
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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