"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const javascript_1 = require("projen/lib/javascript");
const python_1 = require("projen/lib/python");
const typescript_1 = require("projen/lib/typescript");
const synth_1 = require("projen/lib/util/synth");
const src_1 = require("../src");
describe("NX Monorepo Unit Tests", () => {
    it("Empty Monorepo", () => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "mainline",
            name: "Defaults",
        });
        expect(synth_1.synthSnapshot(project)).toMatchSnapshot();
    });
    it("Ignore Patterns", () => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "mainline",
            name: "IgnorePatterns",
            nxConfig: {
                nxIgnore: ["pattern1.txt", "**/foo.ts"],
            },
        });
        expect(synth_1.synthSnapshot(project)).toMatchSnapshot();
    });
    it("Target Dependencies", () => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "mainline",
            name: "TargetDependencies",
            nxConfig: {
                targetDependencies: {
                    test: [
                        {
                            target: "test",
                            projects: src_1.Nx.TargetDependencyProject.DEPENDENCIES,
                        },
                    ],
                    eslint: [
                        { target: "eslint", projects: src_1.Nx.TargetDependencyProject.SELF },
                    ],
                },
            },
        });
        expect(synth_1.synthSnapshot(project)).toMatchSnapshot();
    });
    it("Affected Branch", () => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "main",
            name: "AffectedBranch",
            nxConfig: {
                affectedBranch: "main",
            },
        });
        expect(synth_1.synthSnapshot(project)).toMatchSnapshot();
    });
    it.each([
        javascript_1.NodePackageManager.PNPM,
        javascript_1.NodePackageManager.YARN,
        javascript_1.NodePackageManager.YARN2,
    ])("Additional Workspace Packages", (packageManager) => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "mainline",
            packageManager,
            name: "AdditionalWorkspacePackages",
            workspaceConfig: {
                additionalPackages: ["my/custom/package"],
            },
        });
        new typescript_1.TypeScriptProject({
            name: "ts-subproject",
            outdir: "packages/ts-subproject",
            parent: project,
            packageManager,
            defaultReleaseBranch: "mainline",
        });
        project.addWorkspacePackages("another/custom/package", "yet/another/package");
        expect(synth_1.synthSnapshot(project)).toMatchSnapshot();
    });
    it("Workspace Package Order", () => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "mainline",
            name: "WorkspacePackageOrder",
            workspaceConfig: {
                additionalPackages: ["packages/one"],
            },
        });
        new typescript_1.TypeScriptProject({
            name: "two",
            outdir: "packages/two",
            parent: project,
            defaultReleaseBranch: "mainline",
        });
        project.addWorkspacePackages("packages/three", "packages/four");
        new typescript_1.TypeScriptProject({
            name: "five",
            outdir: "packages/five",
            parent: project,
            defaultReleaseBranch: "mainline",
        });
        project.addWorkspacePackages("packages/six");
        new typescript_1.TypeScriptProject({
            name: "seven",
            outdir: "packages/seven",
            parent: project,
            defaultReleaseBranch: "mainline",
        });
        expect(synth_1.synthSnapshot(project)).toMatchSnapshot();
    });
    it("PNPM", () => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "mainline",
            name: "PNPM",
            packageManager: javascript_1.NodePackageManager.PNPM,
        });
        new typescript_1.TypeScriptProject({
            name: "ts-subproject",
            outdir: "packages/ts-subproject",
            parent: project,
            packageManager: javascript_1.NodePackageManager.PNPM,
            defaultReleaseBranch: "mainline",
        });
        expect(synth_1.synthSnapshot(project)).toMatchSnapshot();
    });
    it("Validate consistent Package Managers", () => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "mainline",
            name: "PNPM",
            packageManager: javascript_1.NodePackageManager.PNPM,
        });
        new typescript_1.TypeScriptProject({
            name: "ts-subproject",
            outdir: "packages/ts-subproject",
            parent: project,
            defaultReleaseBranch: "mainline",
        });
        expect(() => synth_1.synthSnapshot(project)).toThrow("ts-subproject packageManager does not match the monorepo packageManager: pnpm.");
    });
    it("Composite", () => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "mainline",
            name: "Composite",
        });
        const tsProject = new typescript_1.TypeScriptProject({
            name: "ts-subproject",
            outdir: "packages/ts-subproject",
            parent: project,
            defaultReleaseBranch: "mainline",
        });
        new typescript_1.TypeScriptProject({
            name: "ts-subproject2",
            outdir: "packages/ts-subproject2",
            parent: project,
            defaultReleaseBranch: "mainline",
            devDeps: ["ts-subproject"],
        });
        const pyProject = new python_1.PythonProject({
            parent: project,
            outdir: "packages/py-subproject",
            authorEmail: "test@test.com",
            authorName: "test",
            moduleName: "py_subproject",
            name: "py-subproject",
            version: "0.0.0",
        });
        project.addImplicitDependency(tsProject, pyProject);
        expect(synth_1.synthSnapshot(project)).toMatchSnapshot();
    });
});
//# sourceMappingURL=data:application/json;base64,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