"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FsGroupChangePolicy = exports.RestartPolicy = exports.PodSecurityContext = exports.Pod = exports.PodTemplate = exports.PodSpec = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const base_1 = require("./base");
const container_1 = require("./container");
const k8s = require("./imports/k8s");
/**
 * Provides read/write capabilities ontop of a `PodSpecProps`.
 */
class PodSpec {
    constructor(props = {}) {
        this._containers = [];
        this._initContainers = [];
        this._volumes = new Map();
        this.restartPolicy = props.restartPolicy;
        this.serviceAccount = props.serviceAccount;
        this.securityContext = new PodSecurityContext(props.securityContext);
        if (props.containers) {
            props.containers.forEach(c => this.addContainer(c));
        }
        if (props.volumes) {
            props.volumes.forEach(v => this.addVolume(v));
        }
        if (props.initContainers) {
            props.initContainers.forEach(c => this.addInitContainer(c));
        }
    }
    get containers() {
        return [...this._containers];
    }
    get initContainers() {
        return [...this._initContainers];
    }
    get volumes() {
        return Array.from(this._volumes.values());
    }
    addContainer(container) {
        const impl = new container_1.Container(container);
        this._containers.push(impl);
        return impl;
    }
    addInitContainer(container) {
        var _e;
        // https://kubernetes.io/docs/concepts/workloads/pods/init-containers/#differences-from-regular-containers
        if (container.readiness) {
            throw new Error('Init containers must not have a readiness probe');
        }
        if (container.liveness) {
            throw new Error('Init containers must not have a liveness probe');
        }
        if (container.startup) {
            throw new Error('Init containers must not have a startup probe');
        }
        const impl = new container_1.Container({
            ...container,
            name: (_e = container.name) !== null && _e !== void 0 ? _e : `init-${this._initContainers.length}`,
        });
        this._initContainers.push(impl);
        return impl;
    }
    addVolume(volume) {
        const existingVolume = this._volumes.get(volume.name);
        if (existingVolume) {
            throw new Error(`Volume with name ${volume.name} already exists`);
        }
        this._volumes.set(volume.name, volume);
    }
    /**
     * @internal
     */
    _toPodSpec() {
        var _e;
        if (this.containers.length === 0) {
            throw new Error('PodSpec must have at least 1 container');
        }
        const volumes = new Map();
        const containers = [];
        const initContainers = [];
        for (const container of this.containers) {
            // automatically add volume from the container mount
            // to this pod so thats its available to the container.
            for (const mount of container.mounts) {
                addVolume(mount.volume);
            }
            containers.push(container._toKube());
        }
        for (const container of this.initContainers) {
            // automatically add volume from the container mount
            // to this pod so thats its available to the container.
            for (const mount of container.mounts) {
                addVolume(mount.volume);
            }
            initContainers.push(container._toKube());
        }
        for (const volume of this.volumes) {
            addVolume(volume);
        }
        function addVolume(volume) {
            const existingVolume = volumes.get(volume.name);
            // its ok to call this function twice on the same volume, but its not ok to
            // call it twice on a different volume with the same name.
            if (existingVolume && existingVolume !== volume) {
                throw new Error(`Invalid mount configuration. At least two different volumes have the same name: ${volume.name}`);
            }
            volumes.set(volume.name, volume);
        }
        return {
            restartPolicy: this.restartPolicy,
            serviceAccountName: (_e = this.serviceAccount) === null || _e === void 0 ? void 0 : _e.name,
            containers: containers,
            securityContext: this.securityContext._toKube(),
            initContainers: initContainers,
            volumes: Array.from(volumes.values()).map(v => v._toKube()),
        };
    }
}
exports.PodSpec = PodSpec;
_a = JSII_RTTI_SYMBOL_1;
PodSpec[_a] = { fqn: "cdk8s-plus-22.PodSpec", version: "1.0.0-beta.155" };
/**
 * Provides read/write capabilities ontop of a `PodTemplateProps`.
 */
class PodTemplate extends PodSpec {
    constructor(props = {}) {
        super(props);
        this.podMetadata = new cdk8s_1.ApiObjectMetadataDefinition(props.podMetadata);
    }
    /**
     * @internal
     */
    _toPodTemplateSpec() {
        return {
            metadata: this.podMetadata.toJson(),
            spec: this._toPodSpec(),
        };
    }
}
exports.PodTemplate = PodTemplate;
_b = JSII_RTTI_SYMBOL_1;
PodTemplate[_b] = { fqn: "cdk8s-plus-22.PodTemplate", version: "1.0.0-beta.155" };
/**
 * Pod is a collection of containers that can run on a host. This resource is
 * created by clients and scheduled onto hosts.
 */
class Pod extends base_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.apiObject = new k8s.KubePod(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._spec._toPodSpec() }),
        });
        this._spec = new PodSpec(props);
    }
    get containers() {
        return this._spec.containers;
    }
    get initContainers() {
        return this._spec.initContainers;
    }
    get volumes() {
        return this._spec.volumes;
    }
    get restartPolicy() {
        return this._spec.restartPolicy;
    }
    get serviceAccount() {
        return this._spec.serviceAccount;
    }
    get securityContext() {
        return this._spec.securityContext;
    }
    addContainer(container) {
        return this._spec.addContainer(container);
    }
    addInitContainer(container) {
        return this._spec.addInitContainer(container);
    }
    addVolume(volume) {
        return this._spec.addVolume(volume);
    }
}
exports.Pod = Pod;
_c = JSII_RTTI_SYMBOL_1;
Pod[_c] = { fqn: "cdk8s-plus-22.Pod", version: "1.0.0-beta.155" };
/**
 * Holds pod-level security attributes and common container settings.
 */
class PodSecurityContext {
    constructor(props = {}) {
        var _e, _f, _g;
        this._sysctls = [];
        this.ensureNonRoot = (_e = props.ensureNonRoot) !== null && _e !== void 0 ? _e : false;
        this.fsGroupChangePolicy = (_f = props.fsGroupChangePolicy) !== null && _f !== void 0 ? _f : FsGroupChangePolicy.ALWAYS;
        this.user = props.user;
        this.group = props.group;
        this.fsGroup = props.fsGroup;
        for (const sysctl of (_g = props.sysctls) !== null && _g !== void 0 ? _g : []) {
            this._sysctls.push(sysctl);
        }
    }
    get sysctls() {
        return [...this._sysctls];
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            runAsGroup: this.group,
            runAsUser: this.user,
            fsGroup: this.fsGroup,
            runAsNonRoot: this.ensureNonRoot,
            fsGroupChangePolicy: this.fsGroupChangePolicy,
            sysctls: this._sysctls,
        };
    }
}
exports.PodSecurityContext = PodSecurityContext;
_d = JSII_RTTI_SYMBOL_1;
PodSecurityContext[_d] = { fqn: "cdk8s-plus-22.PodSecurityContext", version: "1.0.0-beta.155" };
/**
 * Restart policy for all containers within the pod.
 */
var RestartPolicy;
(function (RestartPolicy) {
    /**
     * Always restart the pod after it exits.
     */
    RestartPolicy["ALWAYS"] = "Always";
    /**
     * Only restart if the pod exits with a non-zero exit code.
     */
    RestartPolicy["ON_FAILURE"] = "OnFailure";
    /**
     * Never restart the pod.
     */
    RestartPolicy["NEVER"] = "Never";
})(RestartPolicy = exports.RestartPolicy || (exports.RestartPolicy = {}));
var FsGroupChangePolicy;
(function (FsGroupChangePolicy) {
    /**
     * Only change permissions and ownership if permission and ownership of root directory does
     * not match with expected permissions of the volume.
     * This could help shorten the time it takes to change ownership and permission of a volume
     */
    FsGroupChangePolicy["ON_ROOT_MISMATCH"] = "OnRootMismatch";
    /**
     * Always change permission and ownership of the volume when volume is mounted.
     */
    FsGroupChangePolicy["ALWAYS"] = "Always";
})(FsGroupChangePolicy = exports.FsGroupChangePolicy || (exports.FsGroupChangePolicy = {}));
//# sourceMappingURL=data:application/json;base64,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