# -*- coding: utf-8 -*-
# This file is part of Viper - https://github.com/viper-framework/viper
# See the file 'LICENSE' for copying permission.

import os
import hashlib
import binascii
import sys
from viper.common.exceptions import Python2UnsupportedUnicode
import six

if sys.version_info < (3, 0):
    # Make sure the read method returns a byte stream
    from io import open

try:
    import pydeep

    HAVE_SSDEEP = True
except ImportError:
    HAVE_SSDEEP = False

try:
    import magic
except ImportError:
    pass

try:
    from pymisp import MISPEvent
except ImportError:
    pass


class Singleton(type):
    _instances = {}

    def __call__(cls, *args, **kwargs):
        if cls not in cls._instances:
            cls._instances[cls] = super(Singleton, cls).__call__(*args, **kwargs)
        return cls._instances[cls]


class MispEvent(object):
    def __init__(self, event, offline=False):
        if isinstance(event, MISPEvent):
            self.event = event
        else:
            self.event = MISPEvent()
            if isinstance(event, six.string_types) and os.path.exists(event):
                self.event.load_file(event)
            else:
                self.event.load(event)
        self.off = offline
        if self.event.id:
            self.current_dump_file = '{}.json'.format(self.event.id)
        else:
            self.current_dump_file = None

    def online(self):
        self.off = False

    def offline(self):
        self.off = True

    def get_all_ips(self):
        return [a.value for a in self.event.attributes if a.type in ['ip-dst', 'ip-src']]

    def get_all_domains(self):
        return [a.value for a in self.event.attributes if a.type in ['domain', 'hostname']]

    def get_all_urls(self):
        return [a.value for a in self.event.attributes if a.type == 'url']

    def get_all_hashes(self):
        event_hashes = []
        sample_hashes = []
        for a in self.event.attributes:
            h = None
            if a.type in ('md5', 'sha1', 'sha256'):
                h = a.value
                event_hashes.append(h)
            elif a.type in ('filename|md5', 'filename|sha1', 'filename|sha256'):
                h = a.value.split('|')[1]
                event_hashes.append(h)
            elif a.type == 'malware-sample':
                h = a.value.split('|')[1]
                sample_hashes.append(h)
        return event_hashes, sample_hashes


class IOBytes(object):
    def __init__(self, byte_array):
        """
        Use to emulate some of the File object features so that a
        io.ByteIO instance can be treated like a file when it comes
        to calling the data property.

        @byte_array: a io.BytesIO instance
        """
        self._data = byte_array

    @property
    def data(self):
        return self._data


class File(object):
    def __init__(self, path):
        self.id = None
        self.path = path
        self.name = ''
        self.size = 0
        self.type = ''
        self.mime = ''
        self.md5 = ''
        self.sha1 = ''
        self.sha256 = ''
        self.sha512 = ''
        self.crc32 = ''
        self.ssdeep = ''
        self.tags = ''
        self.parent = ''
        self.children = ''

        if self.is_valid():
            self.name = os.path.basename(self.path)
            self.size = os.path.getsize(self.path)
            self.type = self.get_type()
            self.mime = self.get_mime()
            self.get_hashes()
            self.ssdeep = self.get_ssdeep()

    @property
    def data(self):
        with open(self.path, 'rb') as f:
            return f.read()

    def is_valid(self):
        if not os.path.exists(self.path):
            return False
        if not os.path.isfile(self.path):
            return False

        if sys.version_info < (3, 0):
            # on Python2 make sure to only handle ASCII filenames
            try:
                self.path.decode('ascii')
            except UnicodeEncodeError as err:
                raise Python2UnsupportedUnicode("Non ASCII character(s) in file name not supported on Python2.\n"
                                                "EncodeError: {}\n"
                                                "File: {}\n"
                                                "Please use Python >= 3.4".format(self.path.encode("utf-8"), err), "error")
            except UnicodeDecodeError as err:
                raise Python2UnsupportedUnicode("Non ASCII character(s) in file name not supported on Python2.\n"
                                                "DecodeError: {}\n"
                                                "File: {}\n"
                                                "Please use Python >= 3.4".format(self.path, err), "error")

        return True

    def get_chunks(self):
        try:
            with open(self.path, 'rb') as fd:
                while True:
                    chunk = fd.read(16 * 1024)
                    if not chunk:
                        break
                    yield chunk
        except Exception:
            return

    def get_hashes(self):
        crc = 0
        md5 = hashlib.md5()
        sha1 = hashlib.sha1()
        sha256 = hashlib.sha256()
        sha512 = hashlib.sha512()

        for chunk in self.get_chunks():
            crc = binascii.crc32(chunk, crc)
            md5.update(chunk)
            sha1.update(chunk)
            sha256.update(chunk)
            sha512.update(chunk)

        self.crc32 = ''.join('%02X' % ((crc >> i) & 0xff) for i in [24, 16, 8, 0])
        self.md5 = md5.hexdigest()
        self.sha1 = sha1.hexdigest()
        self.sha256 = sha256.hexdigest()
        self.sha512 = sha512.hexdigest()

    def get_ssdeep(self):
        if not HAVE_SSDEEP:
            return ''

        try:
            return pydeep.hash_file(self.path).decode()
        except Exception:
            return ''

    def get_type(self):
        try:
            ms = magic.open(magic.MAGIC_NONE)
            ms.load()
            file_type = ms.file(self.path)
        except Exception:
            try:
                file_type = magic.from_file(self.path)
            except Exception:
                try:
                    import subprocess
                    file_process = subprocess.Popen(['file', '-b', self.path], stdout=subprocess.PIPE)
                    file_type = file_process.stdout.read().strip().decode("utf-8")
                except Exception:
                    return ''
        finally:
            try:
                ms.close()
            except Exception:
                pass

        return file_type

    def get_mime(self):
        try:
            ms = magic.open(magic.MIME)
            ms.load()
            mime_type = ms.file(self.path)
        except Exception:
            try:
                mime = magic.Magic(mime=True)
                mime_type = mime.from_file(self.path)
            except Exception:
                return ''

        return mime_type


class Dictionary(dict):
    """Viper custom dict."""

    def __getattr__(self, key):
        return self.get(key, None)

    __setattr__ = dict.__setitem__
    __delattr__ = dict.__delitem__
