"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Mapping = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const aws_cdk_lib_1 = require("aws-cdk-lib");
const operation_1 = require("./operation");
const tasks_1 = require("./tasks");
const TT_MAPALL = 'Map_all';
const TT_MAP = 'Map';
const TT_MERGE = 'Merge';
const TT_ARITHMETIC = 'Arithmetic';
const OP_NOOP = 'NO_OP';
const OP_ADDITION = 'ADDITION';
const OP_SUBTRACTION = 'SUBTRACTION';
const OP_MULTIPLICATION = 'MULTIPLICATION';
const OP_DIVISION = 'DIVISION';
const TP_SOURCE_DATA_TYPE = 'SOURCE_DATA_TYPE';
const TP_DESTINATION_DATA_TYPE = 'DESTINATION_DATA_TYPE';
/**
 * A representation of an instance of a mapping operation, that is an operation translating source to destination fields
 */
class Mapping extends operation_1.OperationBase {
    static mapAll(config) {
        return new Mapping([
            new tasks_1.Task(TT_MAPALL, [], { operation: OP_NOOP }, { EXCLUDE_SOURCE_FIELDS_LIST: config ? `["${aws_cdk_lib_1.Fn.join('","', config.exclude)}"]` : '[]' }),
        ]);
    }
    static map(from, to) {
        const props = {};
        if (from.dataType) {
            props[TP_SOURCE_DATA_TYPE] = from.dataType;
        }
        if (to.dataType) {
            props[TP_DESTINATION_DATA_TYPE] = to.dataType;
        }
        return new Mapping([
            new tasks_1.Task(TT_MAP, [from.name], { operation: OP_NOOP }, props, to.name),
        ]);
    }
    /**
     * A mapping definition building concatenation of source fields into a destination field
     * @param from an array of source fields
     * @param to a desintation field
     * @param format a format
     * @returns a mapping instance with concatenation definition
     */
    static concat(from, to, format) {
        if (!to.dataType) {
            throw new Error('dataType for \'to\' required');
        }
        const tmpField = from.map(f => f.name).join(',');
        return new Mapping([
            new tasks_1.Task(TT_MERGE, from.map(f => f.name), { operation: OP_NOOP }, { CONCAT_FORMAT: format }, tmpField),
            new tasks_1.Task(TT_MAP, [tmpField], { operation: OP_NOOP }, {
                DESTINATION_DATA_TYPE: to.dataType,
                SOURCE_DATA_TYPE: 'string',
            }),
        ]);
    }
    /**
     * Specifies an addition mapping of two numeric values from asource to a destination
     * @param sourceField1 a numeric value
     * @param sourceField2 a numeric value
     * @param to a numeric value
     * @returns an IMapping instance
     */
    static add(sourceField1, sourceField2, to) {
        return Mapping.arithmetic(sourceField1, sourceField2, to, OP_ADDITION);
    }
    /**
     * Specifies a multiplication mapping of two numeric values from a source to a destination
     * @param sourceField1 a numeric value
     * @param sourceField2 a numeric value
     * @param to a numeric value
     * @returns an IMapping instance
     */
    static multiply(sourceField1, sourceField2, to) {
        return Mapping.arithmetic(sourceField1, sourceField2, to, OP_MULTIPLICATION);
    }
    /**
     * Specifies a subtraction mapping of two numeric values from a source to a destination
     * @param sourceField1 a numeric value
     * @param sourceField2 a numeric value
     * @param to a numeric value
     * @returns an IMapping instance
     */
    static subtract(sourceField1, sourceField2, to) {
        return Mapping.arithmetic(sourceField1, sourceField2, to, OP_SUBTRACTION);
    }
    /**
     * Specifies a division mapping of two numeric values from a source to a destination
     * @param sourceField1 a numeric value
     * @param sourceField2 a numeric value
     * @param to a numeric value
     * @returns an IMapping instance
     */
    static divide(sourceField1, sourceField2, to) {
        return Mapping.arithmetic(sourceField1, sourceField2, to, OP_DIVISION);
    }
    /**
     * Specifies an arithmetic mapping of two numeric values from a source to a destination
     * @param sourceField1 a numeric value
     * @param sourceField2 a numeric value
     * @param to a numeric value
     * @returns an IMapping instance
     */
    static arithmetic(sourceField1, sourceField2, to, operation) {
        if (!to.dataType) {
            throw new Error('dataType for \'to\' required');
        }
        const tmpField = `${sourceField1.name},${sourceField2.name}`;
        return new Mapping([
            new tasks_1.Task(TT_ARITHMETIC, [sourceField1.name, sourceField2.name], { operation: operation }, {
                MATH_OPERATION_FIELDS_ORDER: tmpField,
            }, tmpField),
            new tasks_1.Task(TT_MAP, [tmpField], { operation: OP_NOOP }, {
                DESTINATION_DATA_TYPE: to.dataType,
                SOURCE_DATA_TYPE: 'string',
            }, to.name),
        ]);
    }
}
exports.Mapping = Mapping;
_a = JSII_RTTI_SYMBOL_1;
Mapping[_a] = { fqn: "@cdklabs/cdk-appflow.Mapping", version: "0.0.12" };
//# sourceMappingURL=data:application/json;base64,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