"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FlowBase = exports.DataPullMode = exports.FlowStatus = exports.FlowType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_appflow_1 = require("aws-cdk-lib/aws-appflow");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const flow_time_updater_1 = require("./flow-time-updater");
var FlowType;
(function (FlowType) {
    FlowType["EVENT"] = "Event";
    FlowType["ON_DEMAND"] = "OnDemand";
    FlowType["SCHEDULED"] = "Scheduled";
})(FlowType = exports.FlowType || (exports.FlowType = {}));
var FlowStatus;
(function (FlowStatus) {
    FlowStatus["ACTIVE"] = "Active";
    FlowStatus["SUSPENDED"] = "Suspended";
})(FlowStatus = exports.FlowStatus || (exports.FlowStatus = {}));
var DataPullMode;
(function (DataPullMode) {
    DataPullMode["COMPLETE"] = "Complete";
    DataPullMode["INCREMENTAL"] = "Incremental";
})(DataPullMode = exports.DataPullMode || (exports.DataPullMode = {}));
class FlowBase extends aws_cdk_lib_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.mappings = [];
        this.validations = [];
        this.transforms = [];
        this.filters = [];
        this.type = props.type;
        this._projectionFilter = this.initProjectionFilter(props.source.connectorType);
        this.name = props.name || id;
        const resource = new aws_appflow_1.CfnFlow(this, id, {
            flowName: this.name,
            flowStatus: props.status,
            triggerConfig: {
                triggerType: props.type,
                triggerProperties: props.triggerConfig
                    && props.triggerConfig.properties
                    && this.buildTriggerProperties(scope, id, props.triggerConfig.properties),
            },
            kmsArn: props.key?.keyArn,
            metadataCatalogConfig: aws_cdk_lib_1.Lazy.any({ produce: () => this._catalogMetadata }),
            description: props.description,
            sourceFlowConfig: {
                ...props.source.bind(this),
                incrementalPullConfig: this.buildIncrementalPullConfig(props.triggerConfig),
            },
            // NOTE: currently only a single destination is allowed with AppFlow
            //       it might require a change of approach in the future.
            destinationFlowConfigList: [props.destination.bind(this)],
            tasks: aws_cdk_lib_1.Lazy.any({
                produce: () => {
                    const tasks = [
                        ...this.mappings,
                        ...this.transforms,
                        ...this.filters,
                        ...this.validations,
                    ];
                    // TODO: make sure that this.filters doesn't already have a projection definition
                    if (this._projectionFilter.sourceFields.length > 0) {
                        tasks.unshift(this._projectionFilter);
                    }
                    return tasks;
                },
            }),
        });
        this.arn = resource.attrFlowArn;
        this.source = props.source;
        props.mappings.forEach(m => this._addMapping(m));
        if (props.validations) {
            props.validations.forEach(v => this._addValidation(v));
        }
        if (props.transforms) {
            props.transforms.forEach(t => this._addTransform(t));
        }
        if (props.filters) {
            props.filters.forEach(f => this._addFilter(f));
        }
        this.node.addValidation({
            validate: () => this.mappings.length === 0 ? ['A Flow must have at least one mapping'] : [],
        });
    }
    buildTriggerProperties(scope, id, props) {
        const updater = new flow_time_updater_1.FlowTimeUpdater(scope, `${id}Updater`, {
            schedule: props.schedule,
            startTime: props.properties?.startTime,
            endTime: props.properties?.endTime,
        });
        this.node.addDependency(updater);
        return {
            dataPullMode: props.dataPullConfig.mode,
            flowErrorDeactivationThreshold: props.flowErrorDeactivationThreshold,
            scheduleExpression: updater.scheduleExpression,
            firstExecutionFrom: props.properties?.firstExecutionFrom &&
                Math.floor(props.properties.firstExecutionFrom.getTime() / 1000),
            scheduleStartTime: props.properties?.startTime && updater.startTime,
            scheduleEndTime: props.properties?.endTime && updater.endTime,
        };
    }
    initProjectionFilter(sourceType) {
        const filterConnectorOperator = {};
        filterConnectorOperator[sourceType.asTaskConnectorOperatorOrigin] = 'PROJECTION';
        return {
            taskType: 'Filter',
            connectorOperator: filterConnectorOperator,
            sourceFields: [],
        };
    }
    /**
       * Set the catalog definitionfor the flow
       *
       * @internal
       */
    _addCatalog(metadata) {
        this._catalogMetadata = metadata;
    }
    /**
     *
     * @param validation
     * @returns the flow to which the validation was added
     *
     * @internal
     */
    _addValidation(validation) {
        this.validations.push(...validation.bind(this, this.source));
        return this;
    }
    /**
     *
     * @param mapping
     * @returns the flow to which the validation was added
     *
     * @internal
     */
    _addMapping(mapping) {
        const boundMappingTasks = mapping.bind(this, this.source);
        this.addProjectionField(boundMappingTasks);
        this.mappings.push(...boundMappingTasks);
        return this;
    }
    /**
     *
     * @param filter
     * @returns the flow to which the validation was added
     *
     * @internal
     */
    _addFilter(filter) {
        const boundFilterTasks = filter.bind(this, this.source);
        this.addProjectionField(boundFilterTasks);
        this.filters.push(...boundFilterTasks);
        return this;
    }
    /**
     *
     * @param transform
     * @returns the flow to which the validation was added
     *
     * @internal
     */
    _addTransform(transform) {
        this.transforms.push(...transform.bind(this, this.source));
        return this;
    }
    addProjectionField(boundMappingTasks) {
        // TODO: test if this satisfies all the requirements.
        boundMappingTasks.forEach(boundMapping => {
            if (['Map', 'Filter'].includes(boundMapping.taskType)) {
                boundMapping.sourceFields.forEach(field => {
                    if (!this._projectionFilter.sourceFields.includes(field)) {
                        this._projectionFilter.sourceFields.push(field);
                    }
                });
            }
        });
    }
    // see: https://docs.aws.amazon.com/appflow/latest/userguide/flow-notifications.html
    onEvent(id, options = {}) {
        const rule = new aws_events_1.Rule(this, id, options);
        rule.addEventPattern({
            source: ['aws.appflow'],
            resources: [this.arn],
        });
        rule.addTarget(options.target);
        return rule;
    }
    onRunStarted(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['AppFlow Start Flow Run Report'],
        });
        return rule;
    }
    onRunCompleted(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['AppFlow End Flow Run Report'],
        });
        return rule;
    }
    buildIncrementalPullConfig(triggerConfig) {
        return triggerConfig && triggerConfig.properties && triggerConfig.properties.dataPullConfig
            && triggerConfig.properties.dataPullConfig.timestampField
            ? {
                datetimeTypeFieldName: triggerConfig.properties.dataPullConfig.timestampField,
            }
            : undefined;
    }
}
exports.FlowBase = FlowBase;
_a = JSII_RTTI_SYMBOL_1;
FlowBase[_a] = { fqn: "@cdklabs/cdk-appflow.FlowBase", version: "0.0.12" };
//# sourceMappingURL=data:application/json;base64,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