"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SonarCodeScanner = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_codebuild_1 = require("aws-cdk-lib/aws-codebuild");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_secretsmanager_1 = require("aws-cdk-lib/aws-secretsmanager");
const constructs_1 = require("constructs");
const sonarqube_commands_1 = require("./sonarqube-commands");
const unpackSourceAndArtifacts = (includeGlobsForScan) => [
    'export BUILT_ARTIFACT_URI=`aws codebuild batch-get-builds --ids $SYNTH_BUILD_ID | jq -r \'.builds[0].secondaryArtifacts[] | select(.artifactIdentifier == "Synth__") | .location\' | awk \'{sub("arn:aws:s3:::","s3://")}1\' $1`',
    "export SYNTH_SOURCE_URI=`aws codebuild batch-get-builds --ids $SYNTH_BUILD_ID | jq -r '.builds[0].sourceVersion' | awk '{sub(\"arn:aws:s3:::\",\"s3://\")}1' $1`",
    "aws s3 cp $SYNTH_SOURCE_URI source.zip",
    "aws s3 cp $BUILT_ARTIFACT_URI built.zip",
    "unzip source.zip -d src",
    "unzip built.zip -d built",
    "rm source.zip built.zip",
    `rsync -a built/* src --include="*/" ${includeGlobsForScan
        ? includeGlobsForScan.map((g) => `--include ${g}`).join(" ")
        : ""} --include="**/coverage/**" --include="**/cdk.out/**" --exclude="**/node_modules/**/*" --exclude="**/.env/**" --exclude="*" --prune-empty-dirs`,
];
const owaspScan = () => `npx owasp-dependency-check --format HTML --out src/reports --exclude '**/.git/**/*' --scan src --enableExperimental --bin /tmp/dep-check --disableRetireJS`;
const cfnNagScan = (cdkOutDir, cfnNagIgnorePath) => cdkOutDir
    ? `cfn_nag ${cfnNagIgnorePath ? `--deny-list-path=${cfnNagIgnorePath}` : ""} built/${cdkOutDir}/**/*.template.json --output-format=json > src/reports/cfn-nag-report.json`
    : 'echo "skipping cfn_nag as no cdkOutDir was specified.';
class SonarCodeScanner extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const sonarQubeToken = new aws_secretsmanager_1.Secret(this, "SonarQubeToken");
        const synthBuildProject = aws_codebuild_1.Project.fromProjectArn(this, "SynthBuildProject", props.synthBuildArn);
        const validationProject = new aws_codebuild_1.Project(this, "ValidationProject", {
            environment: {
                buildImage: aws_codebuild_1.LinuxBuildImage.STANDARD_5_0,
            },
            environmentVariables: {
                SONARQUBE_TOKEN: {
                    type: aws_codebuild_1.BuildEnvironmentVariableType.SECRETS_MANAGER,
                    value: sonarQubeToken.secretArn,
                },
                SONARQUBE_ENDPOINT: {
                    type: aws_codebuild_1.BuildEnvironmentVariableType.PLAINTEXT,
                    value: props.sonarqubeEndpoint,
                },
                PROJECT_NAME: {
                    type: aws_codebuild_1.BuildEnvironmentVariableType.PLAINTEXT,
                    value: props.sonarqubeProjectName,
                },
            },
            buildSpec: aws_codebuild_1.BuildSpec.fromObject({
                version: "0.2",
                env: {
                    shell: "bash",
                },
                phases: {
                    install: {
                        commands: ["npm install -g aws-cdk", "gem install cfn-nag"],
                    },
                    build: {
                        commands: [
                            ...unpackSourceAndArtifacts(props.includeGlobsForScan),
                            ...sonarqube_commands_1.createSonarqubeProject(props),
                            "mkdir -p src/reports",
                            owaspScan(),
                            cfnNagScan(props.cdkOutDir, props.cfnNagIgnorePath),
                            "cd src",
                            sonarqube_commands_1.sonarqubeScanner(props.excludeGlobsForScan),
                            ...sonarqube_commands_1.generateSonarqubeReports(),
                            ...(props.preArchiveCommands || []),
                        ],
                    },
                },
            }),
        });
        validationProject.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ["codebuild:BatchGetBuilds"],
            effect: aws_iam_1.Effect.ALLOW,
            resources: [synthBuildProject.projectArn],
        }));
        validationProject.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ["s3:GetObject*"],
            effect: aws_iam_1.Effect.ALLOW,
            resources: [props.artifactBucketArn, `${props.artifactBucketArn}/**`],
        }));
        props.artifactBucketKeyArn &&
            validationProject.addToRolePolicy(new aws_iam_1.PolicyStatement({
                actions: ["kms:Decrypt", "kms:DescribeKey"],
                effect: aws_iam_1.Effect.ALLOW,
                resources: [props.artifactBucketKeyArn],
            }));
        synthBuildProject.onBuildSucceeded("OnSynthSuccess", {
            target: new aws_events_targets_1.CodeBuildProject(validationProject, {
                event: aws_events_1.RuleTargetInput.fromObject({
                    environmentVariablesOverride: [
                        {
                            name: "SYNTH_BUILD_ID",
                            type: "PLAINTEXT",
                            value: aws_events_1.EventField.fromPath("$.detail.build-id"),
                        },
                    ],
                }),
            }),
        });
        new aws_cdk_lib_1.CfnOutput(this, "SonarqubeSecretArn", {
            exportName: "SonarqubeSecretArn",
            value: sonarQubeToken.secretArn,
        });
    }
}
exports.SonarCodeScanner = SonarCodeScanner;
_a = JSII_RTTI_SYMBOL_1;
SonarCodeScanner[_a] = { fqn: "aws-prototyping-sdk.pdk_pipeline.SonarCodeScanner", version: "0.0.51" };
//# sourceMappingURL=data:application/json;base64,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