"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SonarCodeScanner = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_codebuild_1 = require("aws-cdk-lib/aws-codebuild");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_secretsmanager_1 = require("aws-cdk-lib/aws-secretsmanager");
const constructs_1 = require("constructs");
const sonarqube_commands_1 = require("./sonarqube-commands");
const unpackSourceAndArtifacts = (includeGlobsForScan) => [
    'export BUILT_ARTIFACT_URI=`aws codebuild batch-get-builds --ids $SYNTH_BUILD_ID | jq -r \'.builds[0].secondaryArtifacts[] | select(.artifactIdentifier == "Synth__") | .location\' | awk \'{sub("arn:aws:s3:::","s3://")}1\' $1`',
    "export SYNTH_SOURCE_URI=`aws codebuild batch-get-builds --ids $SYNTH_BUILD_ID | jq -r '.builds[0].sourceVersion' | awk '{sub(\"arn:aws:s3:::\",\"s3://\")}1' $1`",
    "aws s3 cp $SYNTH_SOURCE_URI source.zip",
    "aws s3 cp $BUILT_ARTIFACT_URI built.zip",
    "unzip source.zip -d src",
    "unzip built.zip -d built",
    "rm source.zip built.zip",
    `rsync -a built/* src --include="*/" ${includeGlobsForScan
        ? includeGlobsForScan.map((g) => `--include ${g}`).join(" ")
        : ""} --include="**/coverage/**" --include="**/cdk.out/**" --exclude="**/node_modules/**/*" --exclude="**/.env/**" --exclude="*" --prune-empty-dirs`,
];
const owaspScan = () => `npx owasp-dependency-check --format HTML --out src/reports --exclude '**/.git/**/*' --scan src --enableExperimental --bin /tmp/dep-check --disableRetireJS`;
const cfnNagScan = (cdkOutDir, cfnNagIgnorePath) => cdkOutDir
    ? `cfn_nag ${cfnNagIgnorePath ? `--deny-list-path=${cfnNagIgnorePath}` : ""} built/${cdkOutDir}/**/*.template.json --output-format=json > src/reports/cfn-nag-report.json`
    : 'echo "skipping cfn_nag as no cdkOutDir was specified.';
class SonarCodeScanner extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const sonarQubeToken = new aws_secretsmanager_1.Secret(this, "SonarQubeToken");
        const synthBuildProject = aws_codebuild_1.Project.fromProjectArn(this, "SynthBuildProject", props.synthBuildArn);
        const validationProject = new aws_codebuild_1.Project(this, "ValidationProject", {
            environment: {
                buildImage: aws_codebuild_1.LinuxBuildImage.STANDARD_5_0,
            },
            environmentVariables: {
                SONARQUBE_TOKEN: {
                    type: aws_codebuild_1.BuildEnvironmentVariableType.SECRETS_MANAGER,
                    value: sonarQubeToken.secretArn,
                },
                SONARQUBE_ENDPOINT: {
                    type: aws_codebuild_1.BuildEnvironmentVariableType.PLAINTEXT,
                    value: props.sonarqubeEndpoint,
                },
                PROJECT_NAME: {
                    type: aws_codebuild_1.BuildEnvironmentVariableType.PLAINTEXT,
                    value: props.sonarqubeProjectName,
                },
            },
            buildSpec: aws_codebuild_1.BuildSpec.fromObject({
                version: "0.2",
                env: {
                    shell: "bash",
                },
                phases: {
                    install: {
                        commands: ["npm install -g aws-cdk", "gem install cfn-nag"],
                    },
                    build: {
                        commands: [
                            "export RESOLVED_SOURCE_VERSION=`aws codebuild batch-get-builds --ids $SYNTH_BUILD_ID | jq -r '.builds[0].resolvedSourceVersion'`",
                            ...unpackSourceAndArtifacts(props.includeGlobsForScan),
                            ...sonarqube_commands_1.createSonarqubeProject(props),
                            "mkdir -p src/reports",
                            owaspScan(),
                            cfnNagScan(props.cdkOutDir, props.cfnNagIgnorePath),
                            "cd src",
                            sonarqube_commands_1.sonarqubeScanner(props.excludeGlobsForScan),
                            ...sonarqube_commands_1.generateSonarqubeReports(),
                            ...(props.preArchiveCommands || []),
                        ],
                    },
                },
            }),
        });
        validationProject.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ["codebuild:BatchGetBuilds"],
            effect: aws_iam_1.Effect.ALLOW,
            resources: [synthBuildProject.projectArn],
        }));
        validationProject.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ["s3:GetObject*"],
            effect: aws_iam_1.Effect.ALLOW,
            resources: [props.artifactBucketArn, `${props.artifactBucketArn}/**`],
        }));
        props.artifactBucketKeyArn &&
            validationProject.addToRolePolicy(new aws_iam_1.PolicyStatement({
                actions: ["kms:Decrypt", "kms:DescribeKey"],
                effect: aws_iam_1.Effect.ALLOW,
                resources: [props.artifactBucketKeyArn],
            }));
        synthBuildProject.onBuildSucceeded("OnSynthSuccess", {
            target: new aws_events_targets_1.CodeBuildProject(validationProject, {
                event: aws_events_1.RuleTargetInput.fromObject({
                    environmentVariablesOverride: [
                        {
                            name: "SYNTH_BUILD_ID",
                            type: "PLAINTEXT",
                            value: aws_events_1.EventField.fromPath("$.detail.build-id"),
                        },
                    ],
                }),
            }),
        });
        new aws_cdk_lib_1.CfnOutput(this, "SonarqubeSecretArn", {
            exportName: "SonarqubeSecretArn",
            value: sonarQubeToken.secretArn,
        });
    }
}
exports.SonarCodeScanner = SonarCodeScanner;
_a = JSII_RTTI_SYMBOL_1;
SonarCodeScanner[_a] = { fqn: "aws-prototyping-sdk.pipeline.SonarCodeScanner", version: "0.3.9" };
//# sourceMappingURL=data:application/json;base64,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