"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PipelineWorkflow = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const helpers_internal_1 = require("aws-cdk-lib/pipelines/lib/helpers-internal");
const github_actions_cdk_1 = require("github-actions-cdk");
const jobs_1 = require("./jobs");
const steps_1 = require("./steps");
/**
 * Represents a GitHub Actions workflow to manage CDK pipeline jobs for synthesizing, publishing, and deploying AWS resources.
 *
 * @remarks
 * Extends `Workflow` from `github-actions-cdk`, and provides structured job orchestration based on the AWS CDK pipeline graph.
 */
class PipelineWorkflow extends github_actions_cdk_1.Workflow {
    static [JSII_RTTI_SYMBOL_1] = { fqn: "@github-actions-cdk/aws-cdk.PipelineWorkflow", version: "0.0.20" };
    awsCredentials;
    versionOverrides;
    cdkoutDir;
    stackOptions;
    assetHashMap = {};
    /**
     * Initializes a new `PipelineWorkflow`.
     *
     * @param scope - The scope within which this construct is created.
     * @param id - The unique identifier for this workflow.
     * @param props - Configuration properties for the pipeline workflow.
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        this.awsCredentials = props.awsCredentials;
        this.versionOverrides = props.versionOverrides;
        this.cdkoutDir = props.cdkoutDir;
        this.stackOptions = props.stackOptions;
        const structure = new helpers_internal_1.PipelineGraph(props.pipeline, {
            selfMutation: false,
            publishTemplate: true,
            prepareStep: false,
            singlePublisherPerAssetType: props.singlePublisherPerAssetType ?? false,
        });
        for (const stageNode of flatten(structure.graph.sortedChildren())) {
            if (!(0, helpers_internal_1.isGraph)(stageNode)) {
                throw new Error(`Top-level children must be graphs, received: '${stageNode}'`);
            }
            const tranches = stageNode.sortedLeaves();
            for (const tranche of tranches) {
                for (const node of tranche) {
                    switch (node.data?.type) {
                        case "step":
                            if (node.data?.isBuildStep && node.data?.step instanceof steps_1.Synth) {
                                this.createSynthJob(node.uniqueId, this.getDependencies(node), node.data.step, props.preBuild, props.postBuild);
                            }
                            else if (node.data?.step instanceof steps_1.StageJob) {
                                this.createStageJob(node.uniqueId, this.getDependencies(node), node.data.step);
                            }
                            break;
                        case "publish-assets":
                            this.createPublishJob(node.uniqueId, this.getDependencies(node), node.data.assets);
                            break;
                        case "execute":
                            this.createDeployJob(node.uniqueId, this.getDependencies(node), node.data.stack);
                            break;
                        default:
                            throw new Error(`Unknown node type: ${node.data?.type}`);
                    }
                }
            }
        }
    }
    /**
     * Creates a job for synthesizing the CDK application.
     *
     * @param id - Unique identifier for the synth job.
     * @param needs - List of dependencies for this job.
     * @param synth - Synth step configuration.
     * @param preBuild - Optional jobs to run before the synth job.
     * @param postBuild - Optional jobs to run after the synth job.
     */
    createSynthJob(id, needs, synth, preBuild, postBuild) {
        new jobs_1.SynthPipelineJob(this, id, {
            name: "Synthesize",
            needs,
            permissions: {
                contents: github_actions_cdk_1.PermissionLevel.READ,
                idToken: this.awsCredentials.permissionLevel(),
            },
            env: synth.env,
            preBuild,
            postBuild,
            installCommands: synth.installCommands,
            commands: synth.commands,
            awsCredentials: this.awsCredentials,
            versionOverrides: this.versionOverrides,
            cdkoutDir: this.cdkoutDir,
        });
    }
    /**
     * Creates a job for publishing stack assets.
     *
     * @param id - Unique identifier for the publish job.
     * @param needs - List of dependencies for this job.
     * @param assets - List of assets to publish.
     */
    createPublishJob(id, needs, assets) {
        new jobs_1.PublishPipelineJob(this, id, {
            name: `Publish Assets ${id}`,
            needs,
            permissions: {
                contents: github_actions_cdk_1.PermissionLevel.READ,
                idToken: this.awsCredentials.permissionLevel(),
            },
            assets,
            assetHashMap: this.assetHashMap,
            awsCredentials: this.awsCredentials,
            versionOverrides: this.versionOverrides,
            cdkoutDir: this.cdkoutDir,
        });
    }
    /**
     * Creates a job for deploying a stack to AWS.
     *
     * @param id - Unique identifier for the deploy job.
     * @param needs - List of dependencies for this job.
     * @param stack - Stack deployment information.
     */
    createDeployJob(id, needs, stack) {
        const options = this.stackOptions[stack.stackArtifactId];
        new jobs_1.DeployPipelineJob(this, id, {
            name: `Deploy ${stack.stackArtifactId}`,
            needs,
            environment: options?.environment,
            permissions: {
                contents: github_actions_cdk_1.PermissionLevel.READ,
                idToken: this.awsCredentials.permissionLevel(),
            },
            stack,
            assetHashMap: this.assetHashMap,
            stackOptions: options,
            awsCredentials: this.awsCredentials,
            versionOverrides: this.versionOverrides,
            cdkoutDir: this.cdkoutDir,
        });
    }
    /**
     * Creates a job for running a stage job in the pipeline.
     *
     * @param id - Unique identifier for the stage job.
     * @param needs - List of dependencies for this job.
     * @param job - Configuration of the stage job.
     */
    createStageJob(id, needs, job) {
        new jobs_1.StagePipelineJob(this, id, {
            name: job.id,
            needs,
            phase: job.props,
            awsCredentials: this.awsCredentials,
            versionOverrides: this.versionOverrides,
            cdkoutDir: this.cdkoutDir,
            ...job.props,
        });
    }
    /**
     * Retrieves a list of dependencies for a given graph node.
     *
     * @param node - The graph node to analyze for dependencies.
     * @returns An array of unique IDs representing dependencies of the node.
     */
    getDependencies(node) {
        const deps = [];
        for (const dep of node.allDeps) {
            if (dep instanceof helpers_internal_1.Graph) {
                deps.push(...dep.allLeaves().nodes);
            }
            else {
                deps.push(dep);
            }
        }
        return deps.map((dependency) => dependency.uniqueId);
    }
}
exports.PipelineWorkflow = PipelineWorkflow;
/**
 * Utility function to flatten an iterable of arrays into a single iterable.
 *
 * @param xs - The input iterable containing arrays.
 * @returns A flattened iterable.
 */
function* flatten(xs) {
    for (const x of xs) {
        yield* x;
    }
}
//# sourceMappingURL=data:application/json;base64,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