import type { PipelineBase, StackAsset, StackDeployment } from "aws-cdk-lib/pipelines";
import type { Construct } from "constructs";
import { Workflow, type WorkflowProps } from "github-actions-cdk";
import type { AwsCredentialsProvider } from "./aws-credentials";
import { type IJobPhase, type StackOptions } from "./jobs";
import { StageJob, Synth } from "./steps";
/**
 * Properties for defining a Pipeline Workflow.
 *
 * @remarks
 * This interface extends WorkflowProps and adds properties specific to AWS CDK Pipelines and job execution.
 */
export interface PipelineWorkflowProps extends WorkflowProps {
    /**
     * The CDK pipeline, including its stages and job configuration.
     * Defines the sequence and structure of actions for synthesizing, publishing, and deploying.
     */
    readonly pipeline: PipelineBase;
    /**
     * Whether to use a single publisher job for each type of asset.
     *
     * @remarks
     * If `true`, each asset type (e.g., file assets, Docker images) will be published by a single job in the workflow,
     * consolidating multiple asset publication steps into one job. This can reduce the total number of jobs needed,
     * making the workflow more efficient when dealing with large numbers of assets.
     *
     * Defaults to `false`, meaning each asset is published in its own job.
     */
    readonly singlePublisherPerAssetType?: boolean;
    /** Configuration options for individual stacks in the pipeline. */
    readonly stackOptions: Record<string, StackOptions>;
    /** Optional job phase to run before the main build jobs. */
    readonly preBuild?: IJobPhase;
    /** Optional job phase to run after the main build jobs. */
    readonly postBuild?: IJobPhase;
    /** Provider for AWS credentials required to interact with AWS services. */
    readonly awsCredentials: AwsCredentialsProvider;
    /** Overrides for specific action versions in GitHub Actions. */
    readonly versionOverrides?: Record<string, string>;
    /** Directory where CDK generates CloudFormation templates. */
    readonly cdkoutDir: string;
}
/**
 * Represents a GitHub Actions workflow to manage CDK pipeline jobs for synthesizing, publishing, and deploying AWS resources.
 *
 * @remarks
 * Extends `Workflow` from `github-actions-cdk`, and provides structured job orchestration based on the AWS CDK pipeline graph.
 */
export declare class PipelineWorkflow extends Workflow {
    readonly awsCredentials: AwsCredentialsProvider;
    readonly versionOverrides?: Record<string, string>;
    readonly cdkoutDir: string;
    private readonly stackOptions;
    private readonly assetHashMap;
    /**
     * Initializes a new `PipelineWorkflow`.
     *
     * @param scope - The scope within which this construct is created.
     * @param id - The unique identifier for this workflow.
     * @param props - Configuration properties for the pipeline workflow.
     */
    constructor(scope: Construct, id: string, props: PipelineWorkflowProps);
    /**
     * Creates a job for synthesizing the CDK application.
     *
     * @param id - Unique identifier for the synth job.
     * @param needs - List of dependencies for this job.
     * @param synth - Synth step configuration.
     * @param preBuild - Optional jobs to run before the synth job.
     * @param postBuild - Optional jobs to run after the synth job.
     */
    protected createSynthJob(id: string, needs: string[], synth: Synth, preBuild?: IJobPhase, postBuild?: IJobPhase): void;
    /**
     * Creates a job for publishing stack assets.
     *
     * @param id - Unique identifier for the publish job.
     * @param needs - List of dependencies for this job.
     * @param assets - List of assets to publish.
     */
    protected createPublishJob(id: string, needs: string[], assets: StackAsset[]): void;
    /**
     * Creates a job for deploying a stack to AWS.
     *
     * @param id - Unique identifier for the deploy job.
     * @param needs - List of dependencies for this job.
     * @param stack - Stack deployment information.
     */
    protected createDeployJob(id: string, needs: string[], stack: StackDeployment): void;
    /**
     * Creates a job for running a stage job in the pipeline.
     *
     * @param id - Unique identifier for the stage job.
     * @param needs - List of dependencies for this job.
     * @param job - Configuration of the stage job.
     */
    protected createStageJob(id: string, needs: string[], job: StageJob): void;
    /**
     * Retrieves a list of dependencies for a given graph node.
     *
     * @param node - The graph node to analyze for dependencies.
     * @returns An array of unique IDs representing dependencies of the node.
     */
    private getDependencies;
}
