"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitHubActionsPipeline = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const pipelines_1 = require("aws-cdk-lib/pipelines");
const constructs_1 = require("constructs");
const adapter_1 = require("./adapter");
const wave_1 = require("./wave");
const workflow_1 = require("./workflow");
/**
 * Constructs a GitHub Actions pipeline for deploying AWS resources.
 *
 * @remarks
 * The `GitHubActionsPipeline` provides methods to define and manage deployment stages and job waves in
 * a GitHub Actions pipeline, utilizing AWS credentials and CDK output for cloud infrastructure automation.
 */
class GitHubActionsPipeline extends constructs_1.Construct {
    static [JSII_RTTI_SYMBOL_1] = { fqn: "@github-actions-cdk/aws-cdk.GitHubActionsPipeline", version: "0.0.20" };
    innerPipeline;
    /**
     * Constructs a new instance of `GitHubActionsPipeline`.
     *
     * @param scope - The parent construct scope.
     * @param id - Unique identifier for this pipeline construct.
     * @param props - Configuration properties for the pipeline.
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.innerPipeline = new InnerPipeline(this, id, props);
    }
    /**
     * Returns the name of the workflow.
     */
    get workflowName() {
        return this.innerPipeline.workflowName;
    }
    /**
     * Returns the output directory path for the workflow files.
     */
    get workflowOutdir() {
        return this.innerPipeline.workflowOutdir;
    }
    /**
     * Returns the filename for the workflow file.
     */
    get workflowFilename() {
        return this.innerPipeline.workflowFilename;
    }
    /**
     * Adds a stage to the pipeline with GitHub-specific configuration options.
     *
     * @param stage - The CDK Stage to add to the pipeline.
     * @param options - Optional configuration for the stage.
     * @returns Deployment details for the added stage.
     */
    addStage(stage, options = {}) {
        return this.innerPipeline.addStageWithGitHubOptions(stage, options);
    }
    /**
     * Adds a wave of jobs to the pipeline.
     *
     * @param id - Unique identifier for the wave.
     * @param options - Options for configuring the wave.
     * @returns The created GitHub wave instance.
     */
    addWave(id, options = {}) {
        return this.innerPipeline.addGitHubWave(id, options);
    }
}
exports.GitHubActionsPipeline = GitHubActionsPipeline;
/**
 * Inner class extending `PipelineBase` to manage core functionalities of the GitHub Actions pipeline.
 */
class InnerPipeline extends pipelines_1.PipelineBase {
    workflowName;
    workflowOutdir;
    workflowFilename;
    singlePublisherPerAssetType;
    workflowEnv;
    preBuild;
    postBuild;
    awsCredentials;
    versionOverrides;
    stackOptions = {};
    adapter;
    /**
     * Constructs a new instance of `InnerPipeline`.
     *
     * @param scope - The parent construct.
     * @param id - Unique identifier for this inner pipeline instance.
     * @param props - Configuration properties for the pipeline.
     */
    constructor(scope, id, props) {
        super(scope, id, { synth: props.synth });
        this.workflowName = props.workflowName ?? "Deploy";
        this.workflowOutdir = props.workflowOutdir ?? ".github/workflows";
        this.workflowFilename = props.workflowFilename ?? "deploy";
        this.singlePublisherPerAssetType = props.singlePublisherPerAssetType;
        this.workflowEnv = props.workflowEnv;
        this.preBuild = props.preBuild;
        this.postBuild = props.postBuild;
        this.awsCredentials = props.awsCredentials;
        this.versionOverrides = props.versionOverrides;
        this.adapter = new adapter_1.AwsCdkAdapter(this, { outdir: this.workflowOutdir });
    }
    /**
     * Adds a stage deployment from a wave with optional configuration.
     *
     * @param stageDeployment - The stage deployment to add.
     * @param options - Configuration options for the stage.
     */
    addStageFromWave(stageDeployment, options) {
        const stacks = stageDeployment.stacks;
        this.addStackProps(stacks, "capabilities", options?.stackCapabilities);
    }
    /**
     * Adds a stage to the pipeline with GitHub-specific options.
     *
     * @param stage - The CDK Stage to add.
     * @param options - Configuration options for the stage.
     * @returns Deployment details for the added stage.
     */
    addStageWithGitHubOptions(stage, options = {}) {
        const stageDeployment = this.addStage(stage, {
            pre: options.preJobs,
            post: options.postJobs,
        });
        const stacks = stageDeployment.stacks;
        this.addStackProps(stacks, "environment", options?.gitHubEnvironment);
        this.addStackProps(stacks, "capabilities", options?.stackCapabilities);
        return stageDeployment;
    }
    /**
     * Adds a wave of jobs to the pipeline with GitHub-specific options.
     *
     * @param id - Unique identifier for the wave.
     * @param options - Configuration options for the wave.
     * @returns The created GitHub wave instance.
     */
    addGitHubWave(id, options = {}) {
        const wave = new wave_1.GitHubWave(id, this, {
            pre: options.preJobs,
            post: options.postJobs,
        });
        this.waves.push(wave._innerWave);
        return wave;
    }
    /**
     * Builds the pipeline workflow, generating workflow files during CDK synthesis.
     *
     * @remarks
     * This method is invoked to create workflow files required by GitHub Actions, integrating CDK stack details.
     */
    doBuildPipeline() {
        const app = aws_cdk_lib_1.Stage.of(this);
        if (!app) {
            throw new Error("The GitHub Workflow must be defined within an App scope.");
        }
        const names = app.node
            .findAll()
            .filter((node) => aws_cdk_lib_1.Stack.isStack(node))
            .map((stack) => stack.stackName);
        new workflow_1.PipelineWorkflow(this.adapter, this.workflowFilename, {
            name: this.workflowName,
            singlePublisherPerAssetType: this.singlePublisherPerAssetType,
            commentAtTop: this.renderYamlComment(names),
            env: this.workflowEnv,
            pipeline: this,
            stackOptions: this.stackOptions,
            preBuild: this.preBuild,
            postBuild: this.postBuild,
            cdkoutDir: app.outdir,
            awsCredentials: this.awsCredentials,
            versionOverrides: this.versionOverrides,
        });
    }
    /**
     * Renders a YAML comment for the workflow file listing deployed stacks.
     *
     * @param stackNames - List of stack names to include in the comment.
     * @returns A formatted string for the YAML comment header.
     */
    renderYamlComment(stackNames) {
        const header = "Generated by github-actions-cdk, DO NOT EDIT DIRECTLY!\n\n";
        const stackListHeader = "Deployed stacks from this pipeline:\n";
        const stackList = stackNames.map((stack) => `- ${stack}`).join("\n");
        return header + stackListHeader + stackList;
    }
    /**
     * Adds properties to stack options for each stack in the deployment.
     *
     * @param stacks - Array of stack deployments.
     * @param key - Property key to set in the stack options.
     * @param value - Value to assign to the specified key.
     */
    addStackProps(stacks, key, value) {
        if (value === undefined)
            return;
        for (const stack of stacks) {
            this.stackOptions[stack.stackArtifactId] = {
                ...this.stackOptions[stack.stackArtifactId],
                [key]: value,
            };
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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