"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitHubActionsRole = exports.GitHubActionsOpenIdConnectProvider = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const constructs_1 = require("constructs");
/**
 * GitHub OIDC provider thumbprints, updated as of 2023-07-27.
 *
 * For details and future updates, see:
 * https://github.blog/changelog/2023-06-27-github-actions-update-on-oidc-integration-with-aws/
 */
const GITHUB_OIDC_THUMBPRINTS = [
    "6938fd4d98bab03faadb97b34396831e3780aea1",
    "1c58a3a8518e8759bf075b76b750d4f2df264fcd",
];
/** GitHub Actions OIDC token endpoint */
const RAW_ENDPOINT = "token.actions.githubusercontent.com";
const PROVIDER_URL = `https://${RAW_ENDPOINT}`;
/**
 * Represents an OpenID Connect (OIDC) provider for GitHub Actions.
 *
 * This provider allows GitHub Actions to assume roles in AWS by connecting through OpenID Connect.
 */
class GitHubActionsOpenIdConnectProvider extends constructs_1.Construct {
    static [JSII_RTTI_SYMBOL_1] = { fqn: "@github-actions-cdk/aws-cdk.GitHubActionsOpenIdConnectProvider", version: "0.0.20" };
    /**
     * Imports an existing GitHub Actions OpenID Connect provider by ARN.
     *
     * @param scope - The construct scope to define the provider within.
     * @returns The imported OIDC provider interface.
     */
    static fromGitHubActionsOpenIdConnectProvider(scope) {
        return aws_iam_1.OpenIdConnectProvider.fromOpenIdConnectProviderArn(scope, "GitHubActionProvider", `arn:${aws_cdk_lib_1.Aws.PARTITION}:iam::${aws_cdk_lib_1.Aws.ACCOUNT_ID}:oidc-provider/${RAW_ENDPOINT}`);
    }
    /**
     * Constructs a new instance of `GitHubActionsOpenIdConnectProvider`.
     *
     * @param scope - The construct scope to define the provider within.
     * @param id - The unique identifier for this provider.
     * @param props - Optional properties for the OpenID Connect provider.
     */
    constructor(scope, id, props = {}) {
        super(scope, id);
        new aws_iam_1.OpenIdConnectProvider(this, "GitHubProvider", {
            url: PROVIDER_URL,
            clientIds: ["sts.amazonaws.com"],
            thumbprints: props.thumbprints ?? GITHUB_OIDC_THUMBPRINTS,
        });
    }
}
exports.GitHubActionsOpenIdConnectProvider = GitHubActionsOpenIdConnectProvider;
/**
 * Creates an IAM Role for GitHub Actions workflows using an OpenID Connect provider.
 *
 * The role includes policies allowing the assumption of bootstrap roles and access to ECR authorization.
 */
class GitHubActionsRole extends aws_iam_1.Role {
    static [JSII_RTTI_SYMBOL_1] = { fqn: "@github-actions-cdk/aws-cdk.GitHubActionsRole", version: "0.0.20" };
    /**
     * Constructs a new instance of `GitHubActionsRole`.
     *
     * @param scope - The construct scope to define the role within.
     * @param id - The unique identifier for this role.
     * @param props - The properties for configuring the GitHub Actions role.
     */
    constructor(scope, id, props) {
        super(scope, id, {
            roleName: props.roleName ?? "GitHubActionsRole",
            assumedBy: new aws_iam_1.FederatedPrincipal(props.provider.openIdConnectProviderArn, {
                StringLike: {
                    [`${RAW_ENDPOINT}:sub`]: formatRepos(props.repos ?? []).concat(props.subjectClaims ?? []),
                },
            }, "sts:AssumeRoleWithWebIdentity"),
            inlinePolicies: {
                AssumeBootstrapRoles: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: ["sts:AssumeRole"],
                            resources: ["*"],
                            conditions: {
                                "ForAnyValue:StringEquals": {
                                    "iam:ResourceTag/aws-cdk:bootstrap-role": [
                                        "deploy",
                                        "lookup",
                                        "file-publishing",
                                        "image-publishing",
                                    ],
                                },
                            },
                        }),
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: ["ecr:GetAuthorizationToken"],
                            resources: ["*"],
                        }),
                    ],
                }),
            },
        });
    }
}
exports.GitHubActionsRole = GitHubActionsRole;
/**
 * Formats GitHub repository identifiers for the OIDC `sub` claim.
 *
 * Each entry is formatted as `repo:owner/repo:*`.
 *
 * @param repos - A list of GitHub repositories in the format `owner/repo`.
 * @returns A list of formatted repository subject claims.
 */
function formatRepos(repos) {
    return repos.map((repo) => `repo:${repo}:*`);
}
//# sourceMappingURL=data:application/json;base64,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