"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.StagePipelineJob = exports.DeployPipelineJob = exports.PublishPipelineJob = exports.SynthPipelineJob = exports.PipelineJob = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = __importStar(require("node:fs"));
const path = __importStar(require("node:path"));
const github_actions_cdk_1 = require("github-actions-cdk");
const ASSET_HASH_NAME = "asset-hash";
const CDKOUT_ARTIFACT = "cdk.out";
/**
 * Represents a job within the pipeline that requires AWS credentials and CDK output.
 *
 * @remarks
 * The `PipelineJob` class extends the `Job` class and includes specific properties and methods for managing
 * AWS authentication, CDK output references, and version control for GitHub Actions used in the pipeline.
 */
class PipelineJob extends github_actions_cdk_1.Job {
    static [JSII_RTTI_SYMBOL_1] = { fqn: "@github-actions-cdk/aws-cdk.PipelineJob", version: "0.0.20" };
    /** AWS credentials provider associated with this job. */
    awsCredentials;
    /** Specific version overrides for GitHub Actions, if any are provided. */
    versionOverrides;
    /** Directory containing the CDK output files for this job. */
    cdkoutDir;
    /**
     * Constructs a new instance of `PipelineJob`.
     *
     * @param scope - The scope in which to define this job construct.
     * @param id - Unique identifier for this job within the workflow.
     * @param props - Properties for configuring the pipeline job.
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        this.awsCredentials = props.awsCredentials;
        this.versionOverrides = props.versionOverrides ?? {};
        this.cdkoutDir = props.cdkoutDir;
    }
    /**
     * Looks up the version override for a given action identifier, if available.
     *
     * @param actionIdentifier - The identifier of the GitHub Action to retrieve the version for.
     * @returns The overridden version (or SHA) for the action, if specified; otherwise, `undefined`.
     */
    lookupVersion(actionIdentifier) {
        return this.versionOverrides[actionIdentifier] ?? undefined;
    }
}
exports.PipelineJob = PipelineJob;
/**
 * A job that synthesizes the CloudFormation template using CDK.
 */
class SynthPipelineJob extends PipelineJob {
    static [JSII_RTTI_SYMBOL_1] = { fqn: "@github-actions-cdk/aws-cdk.SynthPipelineJob", version: "0.0.20" };
    constructor(scope, id, props) {
        super(scope, id, props);
        new github_actions_cdk_1.actions.CheckoutV4(this, "checkout", {
            name: "Checkout",
            version: this.lookupVersion(github_actions_cdk_1.actions.CheckoutV4.IDENTIFIER),
        });
        if (props.preBuild)
            props.preBuild.steps(this);
        if (props.installCommands && props.installCommands.length > 0) {
            new github_actions_cdk_1.RunStep(this, "install", {
                name: "Install",
                run: props.installCommands,
            });
        }
        new github_actions_cdk_1.RunStep(this, "build", {
            name: "Build",
            run: props.commands,
        });
        if (props.postBuild)
            props.postBuild.steps(this);
        new github_actions_cdk_1.actions.UploadArtifactV4(this, "upload", {
            name: `Upload ${CDKOUT_ARTIFACT}`,
            artifactName: CDKOUT_ARTIFACT,
            path: props.cdkoutDir,
            includeHiddenFiles: true,
            version: this.lookupVersion(github_actions_cdk_1.actions.UploadArtifactV4.IDENTIFIER),
        });
    }
}
exports.SynthPipelineJob = SynthPipelineJob;
/**
 * A job that publishes stack assets to AWS.
 *
 * @remarks
 * The `PublishPipelineJob` class handles the process of publishing assets to AWS.
 * It defines the steps required to download artifacts, install necessary dependencies,
 * and execute the publish command for each asset. The job integrates with AWS
 * credentials for secure authentication and provides hooks for outputting asset hashes.
 */
class PublishPipelineJob extends PipelineJob {
    static [JSII_RTTI_SYMBOL_1] = { fqn: "@github-actions-cdk/aws-cdk.PublishPipelineJob", version: "0.0.20" };
    /**
     * Constructs a new instance of `PublishPipelineJob`.
     *
     * @param scope - The parent construct scope.
     * @param id - Unique identifier for this publish job.
     * @param props - Configuration properties for the publish job.
     *
     * @remarks
     * The constructor initializes the publish job by setting up the necessary steps
     * to download artifacts, install dependencies, and publish assets. It iterates
     * through each asset and creates the appropriate publish steps.
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        // Download artifact step
        new github_actions_cdk_1.actions.DownloadArtifactV4(this, "DownloadArtifact", {
            name: `Download ${CDKOUT_ARTIFACT}`,
            artifactName: CDKOUT_ARTIFACT,
            path: props.cdkoutDir,
            version: this.lookupVersion(github_actions_cdk_1.actions.DownloadArtifactV4.IDENTIFIER),
        });
        // Install CDK assets
        const installSuffix = props.cdkCliVersion ? `@${props.cdkCliVersion}` : "";
        new github_actions_cdk_1.RunStep(this, "install", {
            name: "Install",
            run: `npm install --no-save cdk-assets${installSuffix}`,
        });
        // AWS credentials configuration
        props.awsCredentials.credentialSteps(this, "us-east-1");
        // Helper functions to manage file paths and asset names
        const relativeToAssembly = (p) => posixPath(path.join(props.cdkoutDir, path.relative(path.resolve(props.cdkoutDir), p)));
        const assetName = (idx) => `${ASSET_HASH_NAME}-${idx + 1}`;
        // Array to hold paths of publish step files
        const publishStepFiles = [];
        // Loop through each asset and create publish step files
        props.assets.forEach((asset, idx) => {
            const { assetId, assetManifestPath, assetSelector } = asset;
            const fileContents = [
                "set -ex",
                `npx cdk-assets --path "${relativeToAssembly(assetManifestPath)}" --verbose publish "${assetSelector}"`,
            ];
            fileContents.push(`echo '${ASSET_HASH_NAME}=${assetId}' >> $GITHUB_OUTPUT`);
            props.assetHashMap[assetId] = `\${{ needs.${this.id}.outputs.${assetName(idx)} }}`;
            // Define a unique publish step file path for each asset
            const publishStepFile = posixPath(path.join(path.dirname(relativeToAssembly(assetManifestPath)), `publish-${id}-step${idx + 1}.sh`));
            fs.mkdirSync(path.dirname(publishStepFile), { recursive: true });
            fs.writeFileSync(publishStepFile, fileContents.join("\n"), { encoding: "utf-8" });
            publishStepFiles.push(publishStepFile);
        });
        // Create a RunSteps for each publish step file
        publishStepFiles.forEach((publishStepFile, idx) => {
            const publishStep = new github_actions_cdk_1.RunStep(this, `publish-${idx + 1}`, {
                name: `Publish ${id}`,
                run: `/bin/bash ./cdk.out/${posixPath(path.relative(props.cdkoutDir, publishStepFile))}`,
            });
            this.addOutput(assetName(idx), publishStep.outputExpression(ASSET_HASH_NAME));
        });
    }
}
exports.PublishPipelineJob = PublishPipelineJob;
/**
 * A job that deploys a CloudFormation stack.
 *
 * @remarks
 * The `DeployPipelineJob` class is responsible for executing the deployment of a
 * specified CloudFormation stack. It integrates with AWS credentials for authentication
 * and ensures that the stack is deployed with the correct template and asset replacements.
 * The job will throw errors if required properties are not provided, ensuring
 * robustness in the deployment process.
 */
class DeployPipelineJob extends PipelineJob {
    static [JSII_RTTI_SYMBOL_1] = { fqn: "@github-actions-cdk/aws-cdk.DeployPipelineJob", version: "0.0.20" };
    /**
     * Constructs a new instance of `DeployPipelineJob`.
     *
     * @param scope - The parent construct scope.
     * @param id - Unique identifier for this deployment job.
     * @param props - Configuration properties for the deployment job.
     *
     * @remarks
     * The constructor validates required properties for the stack and sets up the
     * necessary steps to deploy the CloudFormation stack using the provided asset hash
     * mappings and options. It initializes the deployment action with AWS CloudFormation.
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        // Validate required properties
        if (!props.stack.region) {
            throw new Error('"region" is required');
        }
        if (!props.stack.templateUrl) {
            throw new Error(`unable to determine template URL for stack ${props.stack.stackArtifactId}`);
        }
        // Configure AWS credentials for deployment
        props.awsCredentials.credentialSteps(this, props.stack.region, props.stack.assumeRoleArn);
        // Function to replace asset hash in the template
        const replaceAssetHash = (template) => {
            const hash = path.parse(template.split("/").pop() ?? "").name;
            if (props.assetHashMap[hash] === undefined) {
                throw new Error(`Template asset hash ${hash} not found.`);
            }
            return template.replace(hash, props.assetHashMap[hash]);
        };
        // Create the CloudFormation deployment action
        new github_actions_cdk_1.actions.AwsCloudFormationGitHubDeployV1(this, "deploy", {
            stackName: props.stack.stackName,
            template: replaceAssetHash(props.stack.templateUrl),
            noFailOnEmptyChangeset: "1",
            roleArn: props.stack.executionRoleArn,
            capabilities: props.stackOptions?.capabilities?.join(","),
            version: this.lookupVersion(github_actions_cdk_1.actions.AwsCloudFormationGitHubDeployV1.IDENTIFIER),
        });
    }
}
exports.DeployPipelineJob = DeployPipelineJob;
/**
 * A job that executes a specific phase of steps in the pipeline.
 */
class StagePipelineJob extends PipelineJob {
    static [JSII_RTTI_SYMBOL_1] = { fqn: "@github-actions-cdk/aws-cdk.StagePipelineJob", version: "0.0.20" };
    constructor(scope, id, props) {
        super(scope, id, props);
        props.phase.steps(this);
    }
}
exports.StagePipelineJob = StagePipelineJob;
/**
 * Converts a Windows or POSIX path to a POSIX path format.
 *
 * @param windowsOrPosixPath - The input path in either Windows or POSIX format.
 * @returns The normalized POSIX path.
 */
function posixPath(windowsOrPosixPath) {
    return windowsOrPosixPath.split(path.sep).join(path.posix.sep);
}
//# sourceMappingURL=data:application/json;base64,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