"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsCredentials = exports.GitHubActionsRoleArn = exports.AwsCredentialsProvider = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const github_actions_cdk_1 = require("github-actions-cdk");
/**
 * Abstract class representing a provider for AWS credentials.
 *
 * Implementations of this class are responsible for defining how
 * AWS credentials are obtained and how they are configured within
 * GitHub Actions jobs.
 */
class AwsCredentialsProvider {
    static [JSII_RTTI_SYMBOL_1] = { fqn: "@github-actions-cdk/aws-cdk.AwsCredentialsProvider", version: "0.0.20" };
}
exports.AwsCredentialsProvider = AwsCredentialsProvider;
/**
 * AWS credentials provider that retrieves credentials from GitHub Secrets.
 */
class GitHubSecretsProvider extends AwsCredentialsProvider {
    accessKeyId;
    secretAccessKey;
    sessionToken;
    /**
     * Constructs a new instance of `GitHubSecretsProvider`.
     *
     * @param props - Optional properties for configuring the GitHub Secrets provider.
     */
    constructor(props) {
        super();
        this.accessKeyId = props?.accessKeyId ?? "AWS_ACCESS_KEY_ID";
        this.secretAccessKey = props?.secretAccessKey ?? "AWS_SECRET_ACCESS_KEY";
        this.sessionToken = props?.sessionToken;
    }
    /**
     * Returns the permission level required by this provider.
     * In this case, no special permissions are required.
     */
    permissionLevel() {
        return github_actions_cdk_1.PermissionLevel.NONE;
    }
    /**
     * Configures AWS credentials for a GitHub Actions job using GitHub Secrets.
     *
     * @param job - The job in which to configure the credentials.
     * @param region - The AWS region where the credentials are applicable.
     * @param assumeRoleArn - An optional ARN for a role to assume.
     * @returns An array of steps to authenticate with AWS using the provided secrets.
     */
    credentialSteps(job, region, assumeRoleArn) {
        return [
            new github_actions_cdk_1.actions.ConfigureAwsCredentialsV4(job, "id", {
                name: "Authenticate Via GitHub Secrets",
                awsRegion: region,
                awsAccessKeyId: github_actions_cdk_1.Expression.fromSecrets(this.accessKeyId),
                awsSecretAccessKey: github_actions_cdk_1.Expression.fromSecrets(this.secretAccessKey),
                ...(this.sessionToken
                    ? {
                        sessionToken: github_actions_cdk_1.Expression.fromSecrets(this.sessionToken),
                    }
                    : undefined),
                ...(assumeRoleArn
                    ? {
                        roleToAssume: assumeRoleArn,
                        roleExternalId: "Pipeline",
                    }
                    : undefined),
            }),
        ];
    }
}
/**
 * AWS credentials provider that uses OpenID Connect for authentication.
 */
class OpenIdConnectProvider extends AwsCredentialsProvider {
    gitHubActionsRoleArn;
    roleSessionName;
    /**
     * Constructs a new instance of `OpenIdConnectProvider`.
     *
     * @param props - Properties for configuring the OpenID Connect provider.
     * @throws Error if `gitHubActionsRoleArn` is unresolved.
     */
    constructor(props) {
        super();
        if (aws_cdk_lib_1.Token.isUnresolved(props.gitHubActionsRoleArn)) {
            throw new Error(`The provided gitHubActionsRoleArn (${props.gitHubActionsRoleArn}) is unresolved. Please provide a concrete value.`);
        }
        this.gitHubActionsRoleArn = props.gitHubActionsRoleArn;
        this.roleSessionName = props.roleSessionName;
    }
    /**
     * Returns the permission level required by this provider.
     * This provider requires write permissions.
     */
    permissionLevel() {
        return github_actions_cdk_1.PermissionLevel.WRITE;
    }
    /**
     * Configures AWS credentials for a GitHub Actions job using OpenID Connect.
     *
     * @param job - The job in which to configure the credentials.
     * @param region - The AWS region where the credentials are applicable.
     * @param assumeRoleArn - An optional ARN for a role to assume with elevated permissions.
     * @returns An array of steps to authenticate with AWS using OpenID Connect.
     */
    credentialSteps(job, region, assumeRoleArn) {
        const steps = [];
        steps.push(new github_actions_cdk_1.actions.ConfigureAwsCredentialsV4(job, "authenticate", {
            name: "Authenticate Via OIDC",
            awsRegion: region,
            roleToAssume: this.gitHubActionsRoleArn,
            roleSessionName: this.roleSessionName,
        }));
        if (assumeRoleArn) {
            const getDeployRole = (arn) => arn.replace("cfn-exec", "deploy");
            steps.push(new github_actions_cdk_1.actions.ConfigureAwsCredentialsV4(job, "assume-role", {
                name: "Assume CDK Deploy Role",
                awsRegion: region,
                awsAccessKeyId: github_actions_cdk_1.Expression.fromEnv("AWS_ACCESS_KEY_ID"),
                awsSecretAccessKey: github_actions_cdk_1.Expression.fromEnv("AWS_SECRET_ACCESS_KEY"),
                awsSessionToken: github_actions_cdk_1.Expression.fromEnv("AWS_SESSION_TOKEN"),
                roleToAssume: getDeployRole(assumeRoleArn),
                roleExternalId: "Pipeline",
            }));
        }
        return steps;
    }
}
/**
 * Helper class for generating ARNs for GitHub Actions roles.
 */
class GitHubActionsRoleArn {
    static [JSII_RTTI_SYMBOL_1] = { fqn: "@github-actions-cdk/aws-cdk.GitHubActionsRoleArn", version: "0.0.20" };
    /**
     * Creates an ARN for a GitHub Actions role based on the account ID.
     *
     * @param accountId - The AWS account ID.
     * @param roleName - The name of the IAM role (defaults to "GitHubActionsRole").
     * @returns The full ARN of the specified role.
     */
    static fromAccount(accountId, roleName = "GitHubActionsRole") {
        return `arn:aws:iam::${accountId}:role/${roleName}`;
    }
}
exports.GitHubActionsRoleArn = GitHubActionsRoleArn;
/**
 * Factory class for creating instances of AWS credentials providers.
 */
class AwsCredentials {
    static [JSII_RTTI_SYMBOL_1] = { fqn: "@github-actions-cdk/aws-cdk.AwsCredentials", version: "0.0.20" };
    /**
     * Creates an AWS credentials provider that uses GitHub secrets.
     *
     * @param props - Optional properties for configuring the GitHub Secrets provider.
     * @returns An instance of `GitHubSecretsProvider`.
     */
    static fromGitHubSecrets(props) {
        return new GitHubSecretsProvider(props);
    }
    /**
     * Creates an AWS credentials provider that uses OpenID Connect.
     *
     * @param props - Properties for configuring the OpenID Connect provider.
     * @returns An instance of `OpenIdConnectProvider`.
     */
    static fromOpenIdConnect(props) {
        return new OpenIdConnectProvider(props);
    }
}
exports.AwsCredentials = AwsCredentials;
//# sourceMappingURL=data:application/json;base64,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