"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsCdkAdapter = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const github_actions_cdk_1 = require("github-actions-cdk");
/**
 * The `AwsCdkAdapter` class integrates GitHub Actions workflows with AWS CDK constructs,
 * inheriting from the `Project` base class in `github-actions-cdk`.
 *
 * This adapter binds the lifecycle of a GitHub Actions workflow to an AWS CDK Construct,
 * allowing workflow creation, error handling, and annotation of errors and warnings
 * during the CDK synthesis process.
 */
class AwsCdkAdapter extends github_actions_cdk_1.Project {
    static [JSII_RTTI_SYMBOL_1] = { fqn: "@github-actions-cdk/aws-cdk.AwsCdkAdapter", version: "0.0.20" };
    awsCdkScope;
    hasValidationErrors;
    /**
     * Constructs a new `AwsCdkAdapter` instance.
     *
     * @param awsCdkScope - The AWS CDK construct scope associated with this adapter.
     * This scope is used as a base for adding validations, annotations, and managing synthesis errors.
     * @param props - Project properties for configuring GitHub Actions workflows.
     */
    constructor(awsCdkScope, props = {}) {
        super(props);
        this.awsCdkScope = awsCdkScope;
        this.hasValidationErrors = false;
        // Add an aspect to automatically synthesize workflows within the CDK scope.
        aws_cdk_lib_1.Aspects.of(this.awsCdkScope).add({
            visit: (node) => {
                if (node === this.awsCdkScope) {
                    this.synth();
                }
            },
        });
    }
    /**
     * Handles synthesis errors encountered during workflow generation.
     * If the error is a validation error, it registers the error message as a validation
     * message on the associated CDK scope.
     *
     * @param error - The error encountered during synthesis.
     * @throws Error - If the error is not a validation error, it will be re-thrown.
     */
    handleSynthesisError(error) {
        if ((0, github_actions_cdk_1.isValidationError)(error)) {
            this.hasValidationErrors = true;
            this.awsCdkScope.node.addValidation({
                validate: () => {
                    return error.errors.map(({ message, source }) => `- [${source.node.path}]: ${message}`);
                },
            });
        }
        else {
            throw error;
        }
    }
    /**
     * Finalizes the synthesis process by transferring workflow annotations to
     * the CDK context as appropriate.
     *
     * This method checks each annotation's severity level (info, warning, error) and
     * adds it to the CDK scope using the `Annotations` utility.
     *
     * Additionally, this method stops synthesis if there are blocking errors,
     * unless overridden by `continueOnErrorAnnotations`.
     */
    finalizeSynthesis() {
        const workflows = Object.values(this.manifest.workflows);
        for (const workflow of workflows) {
            for (const annotation of workflow.annotations) {
                switch (annotation.level) {
                    case github_actions_cdk_1.AnnotationMetadataEntryType.INFO:
                        aws_cdk_lib_1.Annotations.of(this.awsCdkScope).addInfo(annotation.message);
                        break;
                    case github_actions_cdk_1.AnnotationMetadataEntryType.WARN:
                        aws_cdk_lib_1.Annotations.of(this.awsCdkScope).addWarning(annotation.message);
                        break;
                    case github_actions_cdk_1.AnnotationMetadataEntryType.ERROR:
                        aws_cdk_lib_1.Annotations.of(this.awsCdkScope).addError(annotation.message);
                        break;
                    default:
                        throw new Error(`Unknown annotation level: ${annotation.level}`);
                }
            }
        }
        // Halt synthesis if errors exist and should not be ignored.
        if (!this.continueOnErrorAnnotations && this.manifest.hasErrorAnnotation()) {
            return;
        }
        // Halt synthesis if validation errors are present.
        if (this.hasValidationErrors) {
            return;
        }
        // Add informational message upon successful synthesis of workflows.
        aws_cdk_lib_1.Annotations.of(this.awsCdkScope).addInfo(`GitHub Actions workflows generated at ${this.outdir}`);
    }
}
exports.AwsCdkAdapter = AwsCdkAdapter;
//# sourceMappingURL=data:application/json;base64,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