from django.conf import settings
from django.db import models

from crud_framework.errors import Error


class BaseChoices:
    @classmethod
    def get_choices(cls):
        res = []
        for k, v in cls.__dict__.items():
            if k not in ['__module__', '__dict__', '__weakref__', '__doc__']:
                res.append((v, v))
        return res

    @classmethod
    def is_choice_or_err(cls, value):
        for _, v in cls.get_choices():
            if value == v:
                return True
        raise Error(field_name=cls.__name__, message=f'({value}) is not a choice')

    @classmethod
    def get_field(cls, default, max_length=None, null=False, blank=True):
        choices = cls.get_choices()
        if not max_length:
            max_length = max([len(v[0]) for v in choices])
        return models.CharField(max_length=max_length, choices=choices, default=default, null=null, blank=blank)


class BaseManager(models.Manager):
    def __init__(self, *args, **kwargs):
        self.getter_model = kwargs.pop('getter_model', ValueError('model is required'))
        super(BaseManager, self).__init__(*args, **kwargs)

    def get_queryset(self):
        return self.getter_model.objects.get_queryset()


class BaseModel(models.Model):
    class Meta:
        abstract = True

    UNIQUE_FIELDS = []

    def clean_unique(self):
        if not self.UNIQUE_FIELDS:
            return True
        filters = {f: getattr(self, f) for f in self.UNIQUE_FIELDS}
        if self.__class__.objects.filter(is_deleted=False, **filters).filter(~models.Q(pk=self.pk)).exists():
            data = ', '.join([f'{field_name} {getattr(self, field_name)}' for field_name in self.UNIQUE_FIELDS])
            raise Error(message=f'{data} already exist!', field_name=self.UNIQUE_FIELDS[-1])

    def clean(self):
        self.clean_unique()
        super(BaseModel, self).clean()

    def handle_create(self):
        pass

    def handle_update(self):
        pass

    def save(self, *args, **kwargs):
        is_new = not self.pk
        super(BaseModel, self).save(*args, **kwargs)
        if is_new:
            self.handle_create()
        else:
            self.handle_update()


class BaseTrackedModel(BaseModel):
    class Meta:
        abstract = True

    created_at = models.DateTimeField(null=False, auto_now_add=True)
    updated_at = models.DateTimeField(null=False, auto_now=True)
    editor = models.ForeignKey(settings.AUTH_USER_MODEL, on_delete=models.SET_NULL,
                               null=True, blank=True, related_name='%(class)s_editor_user')
    creator = models.ForeignKey(settings.AUTH_USER_MODEL, on_delete=models.SET_NULL,
                                null=True, blank=True, related_name='%(class)s_creator_user')

    def clean_creator(self):
        if self.editor and not self.creator:  # TODO move this to save
            self.creator = self.editor

    def clean(self):
        self.clean_creator()
        super(BaseTrackedModel, self).clean()

    @classmethod
    def get_user_field(cls):
        raise NotImplemented('Refer user_field for filtering')
