# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pyairvisual']

package_data = \
{'': ['*']}

install_requires = \
['aiohttp>=3.6.2,<4.0.0', 'numpy>=1.18.2,<2.0.0', 'pysmb>=1.1.28,<2.0.0']

setup_kwargs = {
    'name': 'pyairvisual',
    'version': '4.4.0',
    'description': 'A simple API for AirVisual air quality data',
    'long_description': '# ☀️ pyairvisual: a thin Python wrapper for the AirVisual© API\n\n[![CI](https://github.com/bachya/pyairvisual/workflows/CI/badge.svg)](https://github.com/bachya/pyairvisual/actions)\n[![PyPi](https://img.shields.io/pypi/v/pyairvisual.svg)](https://pypi.python.org/pypi/pyairvisual)\n[![Version](https://img.shields.io/pypi/pyversions/pyairvisual.svg)](https://pypi.python.org/pypi/pyairvisual)\n[![License](https://img.shields.io/pypi/l/pyairvisual.svg)](https://github.com/bachya/pyairvisual/blob/master/LICENSE)\n[![Code Coverage](https://codecov.io/gh/bachya/pyairvisual/branch/master/graph/badge.svg)](https://codecov.io/gh/bachya/pyairvisual)\n[![Maintainability](https://api.codeclimate.com/v1/badges/948e4e3c84e5c49826f1/maintainability)](https://codeclimate.com/github/bachya/pyairvisual/maintainability)\n[![Say Thanks](https://img.shields.io/badge/SayThanks-!-1EAEDB.svg)](https://saythanks.io/to/bachya)\n\n`pyairvisual` is a simple, clean, well-tested library for interacting with\n[AirVisual](https://www.airvisual.com/) to retrieve air quality information.\n\n- [Python Versions](#python-versions)\n- [Installation](#installation)\n- [API Key](#api-key)\n  * [Community](#community)\n  * [Startup](#startup)\n  * [Enterprise](#enterprise)\n- [Usage](#usage)\n  * [Using the Cloud API](#using-the-cloud-api)\n  * [Working with Node/Pro Units](#working-with-node-pro-units)\n- [Contributing](#contributing)\n\n# Python Versions\n\n`pyairvisual` is currently supported on:\n\n* Python 3.6\n* Python 3.7\n* Python 3.8\n\n# Installation\n\n```python\npip install pyairvisual\n```\n\n# API Key\n\nYou can get an AirVisual API key from\n[the AirVisual API site](https://www.airvisual.com/user/api). Depending on\nthe plan you choose, more functionality will be available from the API:\n\n## Community\n\nThe Community Plan gives access to:\n\n* List supported countries\n* List supported states\n* List supported cities\n* Get data from the nearest city based on IP address\n* Get data from the nearest city based on latitude/longitude\n* Get data from a specific city\n\n## Startup\n\nThe Startup Plan gives access to:\n\n* List supported stations in a city\n* Get data from the nearest station based on IP address\n* Get data from the nearest station based on latitude/longitude\n* Get data from a specific station\n\n## Enterprise\n\nThe Enterprise Plan gives access to:\n\n* Get a global city ranking of air quality\n\n# Usage\n\n## Using the Cloud API\n\n```python\nimport asyncio\n\nfrom aiohttp import ClientSession\n\nfrom pyairvisual import Client\n\n\nasync def main() -> None:\n    """Create the aiohttp session and run the example."""\n    async with ClientSession() as session:\n        # If an API key isn\'t provided, only Nodes can be queried; everything else\n        # requires an API key:\n        client = Client(api_key="<YOUR_AIRVISUAL_API_KEY>", session=session)\n\n        # Get data based on the city nearest to your IP address:\n        data = await client.data.nearest_city()\n\n        # ...or get data based on the city nearest to a latitude/longitude:\n        data = await client.data.nearest_city(\n            latitude=39.742599, longitude=-104.9942557)\n\n        # ...or get it explicitly:\n        data = await client.data.city(\n            city="Los Angeles", state="California", country="USA")\n\n        # If you have the appropriate API key, you can also get data based on\n        # station (nearest or explicit):\n        data = await client.data.nearest_station()\n        data = await client.data.nearest_station(\n            latitude=39.742599, longitude=-104.9942557)\n        data = await client.data.station(\n            station="US Embassy in Beijing",\n            city="Beijing",\n            state="Beijing",\n            country="China")\n\n        # With the appropriate API key, you can get an air quality ranking:\n        data = await client.data.ranking()\n\n        # pyairvisual gives you several methods to look locations up:\n        countries = await client.supported.countries()\n        states = await client.supported.states("USA")\n        cities = await client.supported.cities("USA", "Colorado")\n        stations = await client.supported.stations("USA", "Colorado", "Denver")\n\n\nasyncio.run(main())\n```\n\nBy default, the library creates a new connection to AirVisual with each coroutine. If\nyou are calling a large number of coroutines (or merely want to squeeze out every second\nof runtime savings possible), an\n[`aiohttp`](https://github.com/aio-libs/aiohttp) `ClientSession` can be used for connection\npooling:\n\n```python\nimport asyncio\n\nfrom aiohttp import ClientSession\n\nfrom pyairvisual import Client\n\n\nasync def main() -> None:\n    """Create the aiohttp session and run the example."""\n    async with ClientSession() as session:\n        client = Client(api_key="<YOUR_AIRVISUAL_API_KEY>", session=session)\n\n        # ...\n\n\nasyncio.run(main())\n```\n\n## Working with Node/Pro Units\n\n`pyairvisual` also allows users to interact with\n[Node/Pro units](https://www.airvisual.com/air-quality-monitor), both via the cloud API:\n\n```python\nimport asyncio\n\nfrom aiohttp import ClientSession\n\nfrom pyairvisual import Client\n\n\nasync def main() -> None:\n    """Create the aiohttp session and run the example."""\n    client = Client()\n\n    # The Node/Pro unit ID can be retrieved from the "API" section of the cloud\n    # dashboard:\n    data = await client.node.from_cloud_api("<NODE_ID>")\n\n\nasyncio.run(main())\n```\n\n...or over the local network (the unit password can be found on the device itself):\n\n```python\nimport asyncio\n\nfrom aiohttp import ClientSession\n\nfrom pyairvisual import Client\nfrom pyairvisual.node import NodeSamba\n\n\nasync def main() -> None:\n    """Create the aiohttp session and run the example."""\n    client = Client()\n\n    # Can take several optional parameters:\n    #   1. include_history: include historical measurements (defaults to True)\n    #   2. include_trends: include trends (defaults to True)\n    #   3. measurements_to_use: the number of measurements to use when calculating\n    #      trends (defaults to -1, which means "use all measurements")\n    data = await client.node.from_samba(\n        "<IP_ADDRESS_OR_HOST>",\n        "<PASSWORD>",\n        include_history=True,\n        include_trends=True,\n        measurements_to_use=10,\n    )\n\n\nasyncio.run(main())\n```\n\nCheck out the examples, the tests, and the source files themselves for method\nsignatures and more examples.\n\n# Contributing\n\n1. [Check for open features/bugs](https://github.com/bachya/pyairvisual/issues)\n  or [initiate a discussion on one](https://github.com/bachya/pyairvisual/issues/new).\n2. [Fork the repository](https://github.com/bachya/pyairvisual/fork).\n3. (_optional, but highly recommended_) Create a virtual environment: `python3 -m venv .venv`\n4. (_optional, but highly recommended_) Enter the virtual environment: `source ./.venv/bin/activate`\n5. Install the dev environment: `script/setup`\n6. Code your new feature or bug fix.\n7. Write tests that cover your new functionality.\n8. Run tests and ensure 100% code coverage: `script/test`\n9. Update `README.md` with any new documentation.\n10. Add yourself to `AUTHORS.md`.\n11. Submit a pull request!\n',
    'author': 'Aaron Bach',
    'author_email': 'bachya1208@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/bachya/pyairvisual',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6.0,<4.0.0',
}


setup(**setup_kwargs)
