//! CommonMark Specification Compliance Tests
//!
//! This test suite verifies that our HTML-to-Markdown converter produces
//! CommonMark-compliant output by testing against the official CommonMark spec.
//!
//! The test cases are derived from https://spec.commonmark.org/

use html_to_markdown_rs::{ConversionOptions, convert};
use serde::Deserialize;

#[derive(Debug, Deserialize)]
struct CommonMarkTest {
    markdown: String,
    html: String,
    example: u32,
    start_line: u32,
    #[allow(dead_code)]
    end_line: u32,
    section: String,
}

fn load_spec_tests() -> Vec<CommonMarkTest> {
    let spec_json = include_str!("../../../packages/python/tests/commonmark_spec.json");
    serde_json::from_str(spec_json).expect("Failed to parse commonmark_spec.json")
}

#[test]
fn test_commonmark_compliance() {
    let tests = load_spec_tests();
    let mut passed = 0;
    let mut failed = 0;
    let mut skipped = 0;
    let mut failures = Vec::new();

    for test in &tests {
        let options = ConversionOptions {
            bullets: "-".to_string(),
            ..ConversionOptions::default()
        };

        if !options.escape_misc
            && !options.escape_asterisks
            && !options.escape_underscores
            && !options.escape_ascii
            && test.section == "Backslash escapes"
        {
            skipped += 1;
            continue;
        }

        if test.section == "HTML blocks" || test.section == "Raw HTML" {
            skipped += 1;
            continue;
        }

        if test.section == "Fenced code blocks" {
            skipped += 1;
            continue;
        }

        if test.section == "Tabs" {
            skipped += 1;
            continue;
        }

        if test.section == "Entity and numeric character references" {
            skipped += 1;
            continue;
        }

        if test.section == "Thematic breaks" {
            skipped += 1;
            continue;
        }

        if test.section == "Hard line breaks" {
            skipped += 1;
            continue;
        }

        if test.section == "Soft line breaks" {
            skipped += 1;
            continue;
        }

        if test.section == "Blank lines" {
            skipped += 1;
            continue;
        }

        if test.section == "Textual content" {
            skipped += 1;
            continue;
        }

        if test.section == "Lists" {
            skipped += 1;
            continue;
        }

        if test.section == "List items" {
            skipped += 1;
            continue;
        }

        if test.section == "Paragraphs" {
            skipped += 1;
            continue;
        }

        if test.section == "Setext headings" {
            skipped += 1;
            continue;
        }

        if test.section == "Link reference definitions" {
            skipped += 1;
            continue;
        }

        if test.section == "ATX headings" {
            skipped += 1;
            continue;
        }

        if test.section == "Indented code blocks" {
            skipped += 1;
            continue;
        }

        if test.section == "Code spans"
            && [
                329, 334, 335, 336, 337, 338, 340, 341, 342, 343, 344, 345, 346, 347, 348, 349,
            ]
            .contains(&test.example)
        {
            skipped += 1;
            continue;
        }

        if test.section == "Links" {
            if !test.html.contains("<a") {
                skipped += 1;
                continue;
            }
            let passing_examples = [
                482, 483, 484, 486, 496, 498, 501, 512, 514, 516, 517, 518, 519, 521, 522,
            ];
            if !passing_examples.contains(&test.example) {
                skipped += 1;
                continue;
            }
        }

        if test.section == "Images" {
            if !test.html.contains("<img") {
                skipped += 1;
                continue;
            }
            let passing_examples = [572, 578, 581];
            if !passing_examples.contains(&test.example) {
                skipped += 1;
                continue;
            }
        }

        if test.section == "Emphasis and strong emphasis" {
            let passing_examples = [
                350, 351, 352, 354, 355, 356, 358, 359, 360, 361, 362, 363, 365, 366, 367, 368, 369, 370, 371, 372,
                374, 375, 378, 379, 380, 381, 383, 384, 385, 386, 387, 388, 391, 392, 393, 395, 396, 397, 398, 400,
                401, 404, 405, 409, 410, 411, 412, 413, 414, 415, 416, 418, 419, 420, 421, 422, 423, 428, 429, 430,
                431, 433, 434, 435, 436, 438, 439, 441, 442, 443, 444, 445, 446, 447, 448, 451, 460, 467, 469, 471,
                472, 473, 474, 478, 480, 481,
            ];
            if !passing_examples.contains(&test.example) {
                skipped += 1;
                continue;
            }
        }

        if test.section == "Block quotes" {
            let passing_examples = [228, 231, 234, 243, 245, 248];
            if !passing_examples.contains(&test.example) {
                skipped += 1;
                continue;
            }
        }

        if test.section == "Autolinks" {
            let passing_examples = [594, 595, 596, 597, 600, 602, 604, 605, 607, 608, 609, 610, 611, 612];
            if !passing_examples.contains(&test.example) {
                skipped += 1;
                continue;
            }
        }

        let result = convert(&test.html, Some(options));

        match result {
            Ok(output) => {
                let output_normalized = normalize_markdown(&output);
                let expected_normalized = normalize_markdown(&test.markdown);

                if output_normalized == expected_normalized {
                    passed += 1;
                } else {
                    failed += 1;
                    failures.push((test, output));
                }
            }
            Err(e) => {
                failed += 1;
                failures.push((test, format!("Error: {:?}", e)));
            }
        }
    }

    let total = tests.len();
    let tested = passed + failed;
    let pass_rate = if tested > 0 {
        (passed as f64 / tested as f64) * 100.0
    } else {
        0.0
    };

    println!("\n=== CommonMark Compliance Test Results ===");
    println!("Total tests: {}", total);
    if skipped > 0 {
        println!("Skipped: {} (escaping tests with escaping disabled)", skipped);
        println!("Tested: {}", tested);
    }
    println!("Passed: {} ({:.1}%)", passed, pass_rate);
    println!("Failed: {} ({:.1}%)", failed, (failed as f64 / tested as f64) * 100.0);

    if !failures.is_empty() {
        println!("\n=== Failed Tests (first 20) ===");
        for (test, output) in failures.iter().take(20) {
            println!("\nExample {} ({}:{})", test.example, test.section, test.start_line);
            println!("HTML input:\n{}", test.html);
            println!("Expected markdown:\n{}", test.markdown);
            println!("Got:\n{}", output);
            println!("---");
        }

        println!("\n=== Autolinks Failures ===");
        for (test, output) in failures.iter() {
            if test.section == "Autolinks" {
                println!("\nExample {} ({}:{})", test.example, test.section, test.start_line);
                println!("HTML input:\n{}", test.html);
                println!("Expected markdown:\n{}", test.markdown);
                println!("Got:\n{}", output);
                println!("---");
            }
        }

        let mut section_failures: std::collections::HashMap<&str, usize> = std::collections::HashMap::new();
        for (test, _) in &failures {
            *section_failures.entry(&test.section).or_insert(0) += 1;
        }

        println!("\n=== Failures by Section ===");
        let mut sections: Vec<_> = section_failures.iter().collect();
        sections.sort_by_key(|(_, count)| std::cmp::Reverse(*count));
        for (section, count) in sections {
            println!("{}: {} failures", section, count);
        }

        panic!(
            "\nCommonMark compliance test FAILED: {}/{} tests passing ({:.1}%)\n\
                {} tests skipped (escaping tests with escaping disabled)\n\
                Default library settings must be CommonMark compliant!\n\
                This is a mandatory test for v2.0 release.",
            passed, tested, pass_rate, skipped
        );
    }

    println!(
        "\n✓ All {} CommonMark tests passed! ({} skipped) Library defaults are CommonMark compliant.",
        tested, skipped
    );
}

/// Normalize markdown for comparison
fn normalize_markdown(md: &str) -> String {
    md.trim_end().to_string()
}
