"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StatefulSet = exports.PodManagementPolicy = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const base_1 = require("./base");
const k8s = require("./imports/k8s");
const pod_1 = require("./pod");
/**
 * Controls how pods are created during initial scale up, when replacing pods on nodes,
 * or when scaling down.
 *
 * The default policy is `OrderedReady`, where pods are created in increasing order
 * (pod-0, then pod-1, etc) and the controller will wait until each pod is ready before
 * continuing. When scaling down, the pods are removed in the opposite order.
 *
 * The alternative policy is `Parallel` which will create pods in parallel to match the
 * desired scale without waiting, and on scale down will delete all pods at once.
 */
var PodManagementPolicy;
(function (PodManagementPolicy) {
    PodManagementPolicy["ORDERED_READY"] = "OrderedReady";
    PodManagementPolicy["PARALLEL"] = "Parallel";
})(PodManagementPolicy = exports.PodManagementPolicy || (exports.PodManagementPolicy = {}));
/**
 * StatefulSet is the workload API object used to manage stateful applications.
 *
 * Manages the deployment and scaling of a set of Pods, and provides guarantees
 * about the ordering and uniqueness of these Pods.
 *
 * Like a Deployment, a StatefulSet manages Pods that are based on an identical
 * container spec. Unlike a Deployment, a StatefulSet maintains a sticky identity
 * for each of their Pods. These pods are created from the same spec, but are not
 * interchangeable: each has a persistent identifier that it maintains across any
 * rescheduling.
 *
 * If you want to use storage volumes to provide persistence for your workload, you
 * can use a StatefulSet as part of the solution. Although individual Pods in a StatefulSet
 * are susceptible to failure, the persistent Pod identifiers make it easier to match existing
 * volumes to the new Pods that replace any that have failed.
 *
 * Using StatefulSets
 * ------------------
 * StatefulSets are valuable for applications that require one or more of the following.
 *
 * - Stable, unique network identifiers.
 * - Stable, persistent storage.
 * - Ordered, graceful deployment and scaling.
 * - Ordered, automated rolling updates.
 */
class StatefulSet extends base_1.Resource {
    constructor(scope, id, props) {
        var _b, _c, _d;
        super(scope, id);
        this.apiObject = new k8s.KubeStatefulSet(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
        this._service = props.service;
        this.apiObject.addDependency(this._service);
        this.replicas = (_b = props.replicas) !== null && _b !== void 0 ? _b : 1;
        this.podManagementPolicy = (_c = props.podManagementPolicy) !== null && _c !== void 0 ? _c : PodManagementPolicy.ORDERED_READY;
        this._podTemplate = new pod_1.PodTemplate(props);
        this._labelSelector = {};
        if ((_d = props.defaultSelector) !== null && _d !== void 0 ? _d : true) {
            const selector = 'cdk8s.statefulset';
            const matcher = cdk8s_1.Names.toLabelValue(this);
            this.podMetadata.addLabel(selector, matcher);
            this.selectByLabel(selector, matcher);
        }
        const selectors = Object.entries(this.labelSelector);
        for (const [k, v] of selectors) {
            this._service.addSelector(k, v);
        }
    }
    get podMetadata() {
        return this._podTemplate.podMetadata;
    }
    /**
      * The labels this statefulset will match against in order to select pods.
      *
      * Returns a a copy. Use `selectByLabel()` to add labels.
      */
    get labelSelector() {
        return { ...this._labelSelector };
    }
    get containers() {
        return this._podTemplate.containers;
    }
    get volumes() {
        return this._podTemplate.volumes;
    }
    get restartPolicy() {
        return this._podTemplate.restartPolicy;
    }
    get serviceAccount() {
        return this._podTemplate.serviceAccount;
    }
    /**
      * Configure a label selector to this deployment.
      * Pods that have the label will be selected by deployments configured with this spec.
      *
      * @param key - The label key.
      * @param value - The label value.
      */
    selectByLabel(key, value) {
        this._labelSelector[key] = value;
    }
    addContainer(container) {
        return this._podTemplate.addContainer(container);
    }
    addVolume(volume) {
        return this._podTemplate.addVolume(volume);
    }
    /**
      * @internal
      */
    _toKube() {
        return {
            serviceName: this._service.name,
            replicas: this.replicas,
            template: this._podTemplate._toPodTemplateSpec(),
            selector: {
                matchLabels: this._labelSelector,
            },
            podManagementPolicy: this.podManagementPolicy,
        };
    }
}
exports.StatefulSet = StatefulSet;
_a = JSII_RTTI_SYMBOL_1;
StatefulSet[_a] = { fqn: "cdk8s-plus-22.StatefulSet", version: "1.0.0-beta.142" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic3RhdGVmdWxzZXQuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvc3RhdGVmdWxzZXQudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSxpQ0FBNEU7QUFFNUUsaUNBQWlEO0FBRWpELHFDQUFxQztBQUNyQywrQkFBbUY7QUFNbkY7Ozs7Ozs7Ozs7R0FVRztBQUNILElBQVksbUJBR1g7QUFIRCxXQUFZLG1CQUFtQjtJQUM3QixxREFBOEIsQ0FBQTtJQUM5Qiw0Q0FBcUIsQ0FBQTtBQUN2QixDQUFDLEVBSFcsbUJBQW1CLEdBQW5CLDJCQUFtQixLQUFuQiwyQkFBbUIsUUFHOUI7QUFvQ0Q7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7R0F5Qkc7QUFDSCxNQUFhLFdBQVksU0FBUSxlQUFRO0lBcUJ2QyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXVCOztRQUMvRCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLElBQUksQ0FBQyxTQUFTLEdBQUcsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDekQsUUFBUSxFQUFFLEtBQUssQ0FBQyxRQUFRO1lBQ3hCLElBQUksRUFBRSxZQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUFDO1NBQ2xELENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxRQUFRLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQztRQUU5QixJQUFJLENBQUMsU0FBUyxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUM7UUFFNUMsSUFBSSxDQUFDLFFBQVEsU0FBRyxLQUFLLENBQUMsUUFBUSxtQ0FBSSxDQUFDLENBQUM7UUFDcEMsSUFBSSxDQUFDLG1CQUFtQixTQUFHLEtBQUssQ0FBQyxtQkFBbUIsbUNBQUksbUJBQW1CLENBQUMsYUFBYSxDQUFDO1FBQzFGLElBQUksQ0FBQyxZQUFZLEdBQUcsSUFBSSxpQkFBVyxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQzNDLElBQUksQ0FBQyxjQUFjLEdBQUcsRUFBRSxDQUFDO1FBRXpCLFVBQUksS0FBSyxDQUFDLGVBQWUsbUNBQUksSUFBSSxFQUFFO1lBQ2pDLE1BQU0sUUFBUSxHQUFHLG1CQUFtQixDQUFDO1lBQ3JDLE1BQU0sT0FBTyxHQUFHLGFBQUssQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLENBQUM7WUFDekMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxRQUFRLENBQUMsUUFBUSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQzdDLElBQUksQ0FBQyxhQUFhLENBQUMsUUFBUSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1NBQ3ZDO1FBRUQsTUFBTSxTQUFTLEdBQUcsTUFBTSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUM7UUFDckQsS0FBSyxNQUFNLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxJQUFJLFNBQVMsRUFBRTtZQUM5QixJQUFJLENBQUMsUUFBUSxDQUFDLFdBQVcsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUM7U0FDakM7SUFDSCxDQUFDO0lBRUQsSUFBVyxXQUFXO1FBQ3BCLE9BQU8sSUFBSSxDQUFDLFlBQVksQ0FBQyxXQUFXLENBQUM7SUFDdkMsQ0FBQztJQUVEOzs7O1FBSUk7SUFDSixJQUFXLGFBQWE7UUFDdEIsT0FBTyxFQUFFLEdBQUcsSUFBSSxDQUFDLGNBQWMsRUFBRSxDQUFDO0lBQ3BDLENBQUM7SUFFRCxJQUFXLFVBQVU7UUFDbkIsT0FBTyxJQUFJLENBQUMsWUFBWSxDQUFDLFVBQVUsQ0FBQztJQUN0QyxDQUFDO0lBRUQsSUFBVyxPQUFPO1FBQ2hCLE9BQU8sSUFBSSxDQUFDLFlBQVksQ0FBQyxPQUFPLENBQUM7SUFDbkMsQ0FBQztJQUVELElBQVcsYUFBYTtRQUN0QixPQUFPLElBQUksQ0FBQyxZQUFZLENBQUMsYUFBYSxDQUFDO0lBQ3pDLENBQUM7SUFFRCxJQUFXLGNBQWM7UUFDdkIsT0FBTyxJQUFJLENBQUMsWUFBWSxDQUFDLGNBQWMsQ0FBQztJQUMxQyxDQUFDO0lBRUQ7Ozs7OztRQU1JO0lBQ0csYUFBYSxDQUFDLEdBQVcsRUFBRSxLQUFhO1FBQzdDLElBQUksQ0FBQyxjQUFjLENBQUMsR0FBRyxDQUFDLEdBQUcsS0FBSyxDQUFDO0lBQ25DLENBQUM7SUFFTSxZQUFZLENBQUMsU0FBeUI7UUFDM0MsT0FBTyxJQUFJLENBQUMsWUFBWSxDQUFDLFlBQVksQ0FBQyxTQUFTLENBQUMsQ0FBQztJQUNuRCxDQUFDO0lBRU0sU0FBUyxDQUFDLE1BQWM7UUFDN0IsT0FBTyxJQUFJLENBQUMsWUFBWSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsQ0FBQztJQUM3QyxDQUFDO0lBRUQ7O1FBRUk7SUFDRyxPQUFPO1FBQ1osT0FBTztZQUNMLFdBQVcsRUFBRSxJQUFJLENBQUMsUUFBUSxDQUFDLElBQUk7WUFDL0IsUUFBUSxFQUFFLElBQUksQ0FBQyxRQUFRO1lBQ3ZCLFFBQVEsRUFBRSxJQUFJLENBQUMsWUFBWSxDQUFDLGtCQUFrQixFQUFFO1lBQ2hELFFBQVEsRUFBRTtnQkFDUixXQUFXLEVBQUUsSUFBSSxDQUFDLGNBQWM7YUFDakM7WUFDRCxtQkFBbUIsRUFBRSxJQUFJLENBQUMsbUJBQW1CO1NBQzlDLENBQUM7SUFDSixDQUFDOztBQS9HSCxrQ0FnSEMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBBcGlPYmplY3QsIEFwaU9iamVjdE1ldGFkYXRhRGVmaW5pdGlvbiwgTGF6eSwgTmFtZXMgfSBmcm9tICdjZGs4cyc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IFJlc291cmNlLCBSZXNvdXJjZVByb3BzIH0gZnJvbSAnLi9iYXNlJztcbmltcG9ydCB7IENvbnRhaW5lciwgQ29udGFpbmVyUHJvcHMgfSBmcm9tICcuL2NvbnRhaW5lcic7XG5pbXBvcnQgKiBhcyBrOHMgZnJvbSAnLi9pbXBvcnRzL2s4cyc7XG5pbXBvcnQgeyBSZXN0YXJ0UG9saWN5LCBQb2RUZW1wbGF0ZSwgSVBvZFRlbXBsYXRlLCBQb2RUZW1wbGF0ZVByb3BzIH0gZnJvbSAnLi9wb2QnO1xuaW1wb3J0IHsgU2VydmljZSB9IGZyb20gJy4vc2VydmljZSc7XG5pbXBvcnQgeyBJU2VydmljZUFjY291bnQgfSBmcm9tICcuL3NlcnZpY2UtYWNjb3VudCc7XG5pbXBvcnQgeyBWb2x1bWUgfSBmcm9tICcuL3ZvbHVtZSc7XG5cblxuLyoqXG4gKiBDb250cm9scyBob3cgcG9kcyBhcmUgY3JlYXRlZCBkdXJpbmcgaW5pdGlhbCBzY2FsZSB1cCwgd2hlbiByZXBsYWNpbmcgcG9kcyBvbiBub2RlcyxcbiAqIG9yIHdoZW4gc2NhbGluZyBkb3duLlxuICpcbiAqIFRoZSBkZWZhdWx0IHBvbGljeSBpcyBgT3JkZXJlZFJlYWR5YCwgd2hlcmUgcG9kcyBhcmUgY3JlYXRlZCBpbiBpbmNyZWFzaW5nIG9yZGVyXG4gKiAocG9kLTAsIHRoZW4gcG9kLTEsIGV0YykgYW5kIHRoZSBjb250cm9sbGVyIHdpbGwgd2FpdCB1bnRpbCBlYWNoIHBvZCBpcyByZWFkeSBiZWZvcmVcbiAqIGNvbnRpbnVpbmcuIFdoZW4gc2NhbGluZyBkb3duLCB0aGUgcG9kcyBhcmUgcmVtb3ZlZCBpbiB0aGUgb3Bwb3NpdGUgb3JkZXIuXG4gKlxuICogVGhlIGFsdGVybmF0aXZlIHBvbGljeSBpcyBgUGFyYWxsZWxgIHdoaWNoIHdpbGwgY3JlYXRlIHBvZHMgaW4gcGFyYWxsZWwgdG8gbWF0Y2ggdGhlXG4gKiBkZXNpcmVkIHNjYWxlIHdpdGhvdXQgd2FpdGluZywgYW5kIG9uIHNjYWxlIGRvd24gd2lsbCBkZWxldGUgYWxsIHBvZHMgYXQgb25jZS5cbiAqL1xuZXhwb3J0IGVudW0gUG9kTWFuYWdlbWVudFBvbGljeSB7XG4gIE9SREVSRURfUkVBRFkgPSAnT3JkZXJlZFJlYWR5JyxcbiAgUEFSQUxMRUwgPSAnUGFyYWxsZWwnLFxufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGluaXRpYWxpemF0aW9uIG9mIGBTdGF0ZWZ1bFNldGAuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgU3RhdGVmdWxTZXRQcm9wcyBleHRlbmRzIFJlc291cmNlUHJvcHMsIFBvZFRlbXBsYXRlUHJvcHMge1xuICAvKipcbiAgICogU2VydmljZSB0byBhc3NvY2lhdGUgd2l0aCB0aGUgc3RhdGVmdWxzZXQuXG4gICAqL1xuICByZWFkb25seSBzZXJ2aWNlOiBTZXJ2aWNlO1xuXG4gIC8qKlxuICAgICogTnVtYmVyIG9mIGRlc2lyZWQgcG9kcy5cbiAgICAqXG4gICAgKiBAZGVmYXVsdCAxXG4gICAgKi9cbiAgcmVhZG9ubHkgcmVwbGljYXM/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAgKiBBdXRvbWF0aWNhbGx5IGFsbG9jYXRlcyBhIHBvZCBzZWxlY3RvciBmb3IgdGhpcyBzdGF0ZWZ1bHNldC5cbiAgICAqXG4gICAgKiBJZiB0aGlzIGlzIHNldCB0byBgZmFsc2VgIHlvdSBtdXN0IGRlZmluZSB5b3VyIHNlbGVjdG9yIHRocm91Z2hcbiAgICAqIGBzdGF0ZWZ1bHNldC5wb2RNZXRhZGF0YS5hZGRMYWJlbCgpYCBhbmQgYHN0YXRlZnVsc2V0LnNlbGVjdEJ5TGFiZWwoKWAuXG4gICAgKlxuICAgICogQGRlZmF1bHQgdHJ1ZVxuICAgICovXG4gIHJlYWRvbmx5IGRlZmF1bHRTZWxlY3Rvcj86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAgKiBQb2QgbWFuYWdlbWVudCBwb2xpY3kgdG8gdXNlIGZvciB0aGlzIHN0YXRlZnVsc2V0LlxuICAgICpcbiAgICAqIEBkZWZhdWx0IFBvZE1hbmFnZW1lbnRQb2xpY3kuT1JERVJFRF9SRUFEWVxuICAgICovXG4gIHJlYWRvbmx5IHBvZE1hbmFnZW1lbnRQb2xpY3k/OiBQb2RNYW5hZ2VtZW50UG9saWN5O1xufVxuXG4vKipcbiAqIFN0YXRlZnVsU2V0IGlzIHRoZSB3b3JrbG9hZCBBUEkgb2JqZWN0IHVzZWQgdG8gbWFuYWdlIHN0YXRlZnVsIGFwcGxpY2F0aW9ucy5cbiAqXG4gKiBNYW5hZ2VzIHRoZSBkZXBsb3ltZW50IGFuZCBzY2FsaW5nIG9mIGEgc2V0IG9mIFBvZHMsIGFuZCBwcm92aWRlcyBndWFyYW50ZWVzXG4gKiBhYm91dCB0aGUgb3JkZXJpbmcgYW5kIHVuaXF1ZW5lc3Mgb2YgdGhlc2UgUG9kcy5cbiAqXG4gKiBMaWtlIGEgRGVwbG95bWVudCwgYSBTdGF0ZWZ1bFNldCBtYW5hZ2VzIFBvZHMgdGhhdCBhcmUgYmFzZWQgb24gYW4gaWRlbnRpY2FsXG4gKiBjb250YWluZXIgc3BlYy4gVW5saWtlIGEgRGVwbG95bWVudCwgYSBTdGF0ZWZ1bFNldCBtYWludGFpbnMgYSBzdGlja3kgaWRlbnRpdHlcbiAqIGZvciBlYWNoIG9mIHRoZWlyIFBvZHMuIFRoZXNlIHBvZHMgYXJlIGNyZWF0ZWQgZnJvbSB0aGUgc2FtZSBzcGVjLCBidXQgYXJlIG5vdFxuICogaW50ZXJjaGFuZ2VhYmxlOiBlYWNoIGhhcyBhIHBlcnNpc3RlbnQgaWRlbnRpZmllciB0aGF0IGl0IG1haW50YWlucyBhY3Jvc3MgYW55XG4gKiByZXNjaGVkdWxpbmcuXG4gKlxuICogSWYgeW91IHdhbnQgdG8gdXNlIHN0b3JhZ2Ugdm9sdW1lcyB0byBwcm92aWRlIHBlcnNpc3RlbmNlIGZvciB5b3VyIHdvcmtsb2FkLCB5b3VcbiAqIGNhbiB1c2UgYSBTdGF0ZWZ1bFNldCBhcyBwYXJ0IG9mIHRoZSBzb2x1dGlvbi4gQWx0aG91Z2ggaW5kaXZpZHVhbCBQb2RzIGluIGEgU3RhdGVmdWxTZXRcbiAqIGFyZSBzdXNjZXB0aWJsZSB0byBmYWlsdXJlLCB0aGUgcGVyc2lzdGVudCBQb2QgaWRlbnRpZmllcnMgbWFrZSBpdCBlYXNpZXIgdG8gbWF0Y2ggZXhpc3RpbmdcbiAqIHZvbHVtZXMgdG8gdGhlIG5ldyBQb2RzIHRoYXQgcmVwbGFjZSBhbnkgdGhhdCBoYXZlIGZhaWxlZC5cbiAqXG4gKiBVc2luZyBTdGF0ZWZ1bFNldHNcbiAqIC0tLS0tLS0tLS0tLS0tLS0tLVxuICogU3RhdGVmdWxTZXRzIGFyZSB2YWx1YWJsZSBmb3IgYXBwbGljYXRpb25zIHRoYXQgcmVxdWlyZSBvbmUgb3IgbW9yZSBvZiB0aGUgZm9sbG93aW5nLlxuICpcbiAqIC0gU3RhYmxlLCB1bmlxdWUgbmV0d29yayBpZGVudGlmaWVycy5cbiAqIC0gU3RhYmxlLCBwZXJzaXN0ZW50IHN0b3JhZ2UuXG4gKiAtIE9yZGVyZWQsIGdyYWNlZnVsIGRlcGxveW1lbnQgYW5kIHNjYWxpbmcuXG4gKiAtIE9yZGVyZWQsIGF1dG9tYXRlZCByb2xsaW5nIHVwZGF0ZXMuXG4gKi9cbmV4cG9ydCBjbGFzcyBTdGF0ZWZ1bFNldCBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSVBvZFRlbXBsYXRlIHtcbiAgLyoqXG4gICAgKiBOdW1iZXIgb2YgZGVzaXJlZCBwb2RzLlxuICAgICovXG4gIHB1YmxpYyByZWFkb25seSByZXBsaWNhczogbnVtYmVyO1xuXG4gIC8qKlxuICAgICogTWFuYWdlbWVudCBwb2xpY3kgdG8gdXNlIGZvciB0aGUgc2V0LlxuICAgICovXG4gIHB1YmxpYyByZWFkb25seSBwb2RNYW5hZ2VtZW50UG9saWN5OiBQb2RNYW5hZ2VtZW50UG9saWN5O1xuXG4gIC8qKlxuICAgICogQHNlZSBiYXNlLlJlc291cmNlLmFwaU9iamVjdFxuICAgICovXG4gIHByb3RlY3RlZCByZWFkb25seSBhcGlPYmplY3Q6IEFwaU9iamVjdDtcblxuICBwcml2YXRlIHJlYWRvbmx5IF9wb2RUZW1wbGF0ZTogUG9kVGVtcGxhdGU7XG4gIHByaXZhdGUgcmVhZG9ubHkgX2xhYmVsU2VsZWN0b3I6IFJlY29yZDxzdHJpbmcsIHN0cmluZz47XG5cbiAgcHJpdmF0ZSByZWFkb25seSBfc2VydmljZTogU2VydmljZTtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogU3RhdGVmdWxTZXRQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICB0aGlzLmFwaU9iamVjdCA9IG5ldyBrOHMuS3ViZVN0YXRlZnVsU2V0KHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgIG1ldGFkYXRhOiBwcm9wcy5tZXRhZGF0YSxcbiAgICAgIHNwZWM6IExhenkuYW55KHsgcHJvZHVjZTogKCkgPT4gdGhpcy5fdG9LdWJlKCkgfSksXG4gICAgfSk7XG4gICAgdGhpcy5fc2VydmljZSA9IHByb3BzLnNlcnZpY2U7XG5cbiAgICB0aGlzLmFwaU9iamVjdC5hZGREZXBlbmRlbmN5KHRoaXMuX3NlcnZpY2UpO1xuXG4gICAgdGhpcy5yZXBsaWNhcyA9IHByb3BzLnJlcGxpY2FzID8/IDE7XG4gICAgdGhpcy5wb2RNYW5hZ2VtZW50UG9saWN5ID0gcHJvcHMucG9kTWFuYWdlbWVudFBvbGljeSA/PyBQb2RNYW5hZ2VtZW50UG9saWN5Lk9SREVSRURfUkVBRFk7XG4gICAgdGhpcy5fcG9kVGVtcGxhdGUgPSBuZXcgUG9kVGVtcGxhdGUocHJvcHMpO1xuICAgIHRoaXMuX2xhYmVsU2VsZWN0b3IgPSB7fTtcblxuICAgIGlmIChwcm9wcy5kZWZhdWx0U2VsZWN0b3IgPz8gdHJ1ZSkge1xuICAgICAgY29uc3Qgc2VsZWN0b3IgPSAnY2RrOHMuc3RhdGVmdWxzZXQnO1xuICAgICAgY29uc3QgbWF0Y2hlciA9IE5hbWVzLnRvTGFiZWxWYWx1ZSh0aGlzKTtcbiAgICAgIHRoaXMucG9kTWV0YWRhdGEuYWRkTGFiZWwoc2VsZWN0b3IsIG1hdGNoZXIpO1xuICAgICAgdGhpcy5zZWxlY3RCeUxhYmVsKHNlbGVjdG9yLCBtYXRjaGVyKTtcbiAgICB9XG5cbiAgICBjb25zdCBzZWxlY3RvcnMgPSBPYmplY3QuZW50cmllcyh0aGlzLmxhYmVsU2VsZWN0b3IpO1xuICAgIGZvciAoY29uc3QgW2ssIHZdIG9mIHNlbGVjdG9ycykge1xuICAgICAgdGhpcy5fc2VydmljZS5hZGRTZWxlY3RvcihrLCB2KTtcbiAgICB9XG4gIH1cblxuICBwdWJsaWMgZ2V0IHBvZE1ldGFkYXRhKCk6IEFwaU9iamVjdE1ldGFkYXRhRGVmaW5pdGlvbiB7XG4gICAgcmV0dXJuIHRoaXMuX3BvZFRlbXBsYXRlLnBvZE1ldGFkYXRhO1xuICB9XG5cbiAgLyoqXG4gICAgKiBUaGUgbGFiZWxzIHRoaXMgc3RhdGVmdWxzZXQgd2lsbCBtYXRjaCBhZ2FpbnN0IGluIG9yZGVyIHRvIHNlbGVjdCBwb2RzLlxuICAgICpcbiAgICAqIFJldHVybnMgYSBhIGNvcHkuIFVzZSBgc2VsZWN0QnlMYWJlbCgpYCB0byBhZGQgbGFiZWxzLlxuICAgICovXG4gIHB1YmxpYyBnZXQgbGFiZWxTZWxlY3RvcigpOiBSZWNvcmQ8c3RyaW5nLCBzdHJpbmc+IHtcbiAgICByZXR1cm4geyAuLi50aGlzLl9sYWJlbFNlbGVjdG9yIH07XG4gIH1cblxuICBwdWJsaWMgZ2V0IGNvbnRhaW5lcnMoKTogQ29udGFpbmVyW10ge1xuICAgIHJldHVybiB0aGlzLl9wb2RUZW1wbGF0ZS5jb250YWluZXJzO1xuICB9XG5cbiAgcHVibGljIGdldCB2b2x1bWVzKCk6IFZvbHVtZVtdIHtcbiAgICByZXR1cm4gdGhpcy5fcG9kVGVtcGxhdGUudm9sdW1lcztcbiAgfVxuXG4gIHB1YmxpYyBnZXQgcmVzdGFydFBvbGljeSgpOiBSZXN0YXJ0UG9saWN5IHwgdW5kZWZpbmVkIHtcbiAgICByZXR1cm4gdGhpcy5fcG9kVGVtcGxhdGUucmVzdGFydFBvbGljeTtcbiAgfVxuXG4gIHB1YmxpYyBnZXQgc2VydmljZUFjY291bnQoKTogSVNlcnZpY2VBY2NvdW50IHwgdW5kZWZpbmVkIHtcbiAgICByZXR1cm4gdGhpcy5fcG9kVGVtcGxhdGUuc2VydmljZUFjY291bnQ7XG4gIH1cblxuICAvKipcbiAgICAqIENvbmZpZ3VyZSBhIGxhYmVsIHNlbGVjdG9yIHRvIHRoaXMgZGVwbG95bWVudC5cbiAgICAqIFBvZHMgdGhhdCBoYXZlIHRoZSBsYWJlbCB3aWxsIGJlIHNlbGVjdGVkIGJ5IGRlcGxveW1lbnRzIGNvbmZpZ3VyZWQgd2l0aCB0aGlzIHNwZWMuXG4gICAgKlxuICAgICogQHBhcmFtIGtleSAtIFRoZSBsYWJlbCBrZXkuXG4gICAgKiBAcGFyYW0gdmFsdWUgLSBUaGUgbGFiZWwgdmFsdWUuXG4gICAgKi9cbiAgcHVibGljIHNlbGVjdEJ5TGFiZWwoa2V5OiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcpOiB2b2lkIHtcbiAgICB0aGlzLl9sYWJlbFNlbGVjdG9yW2tleV0gPSB2YWx1ZTtcbiAgfVxuXG4gIHB1YmxpYyBhZGRDb250YWluZXIoY29udGFpbmVyOiBDb250YWluZXJQcm9wcyk6IENvbnRhaW5lciB7XG4gICAgcmV0dXJuIHRoaXMuX3BvZFRlbXBsYXRlLmFkZENvbnRhaW5lcihjb250YWluZXIpO1xuICB9XG5cbiAgcHVibGljIGFkZFZvbHVtZSh2b2x1bWU6IFZvbHVtZSk6IHZvaWQge1xuICAgIHJldHVybiB0aGlzLl9wb2RUZW1wbGF0ZS5hZGRWb2x1bWUodm9sdW1lKTtcbiAgfVxuXG4gIC8qKlxuICAgICogQGludGVybmFsXG4gICAgKi9cbiAgcHVibGljIF90b0t1YmUoKTogazhzLlN0YXRlZnVsU2V0U3BlYyB7XG4gICAgcmV0dXJuIHtcbiAgICAgIHNlcnZpY2VOYW1lOiB0aGlzLl9zZXJ2aWNlLm5hbWUsXG4gICAgICByZXBsaWNhczogdGhpcy5yZXBsaWNhcyxcbiAgICAgIHRlbXBsYXRlOiB0aGlzLl9wb2RUZW1wbGF0ZS5fdG9Qb2RUZW1wbGF0ZVNwZWMoKSxcbiAgICAgIHNlbGVjdG9yOiB7XG4gICAgICAgIG1hdGNoTGFiZWxzOiB0aGlzLl9sYWJlbFNlbGVjdG9yLFxuICAgICAgfSxcbiAgICAgIHBvZE1hbmFnZW1lbnRQb2xpY3k6IHRoaXMucG9kTWFuYWdlbWVudFBvbGljeSxcbiAgICB9O1xuICB9XG59XG4iXX0=