import { ApiObject, ApiObjectMetadata, ApiObjectMetadataDefinition } from 'cdk8s';
import { Construct } from 'constructs';
import { ResourceProps, Resource } from './base';
import { Container, ContainerProps } from './container';
import * as k8s from './imports/k8s';
import { IServiceAccount } from './service-account';
import { Volume } from './volume';
/**
 * Represents a resource that can be configured with a kuberenets pod spec. (e.g `Deployment`, `Job`, `Pod`, ...).
 *
 * Use the `PodSpec` class as an implementation helper.
 */
export interface IPodSpec {
    /**
     * The containers belonging to the pod.
     *
     * Use `addContainer` to add containers.
     */
    readonly containers: Container[];
    /**
     * The volumes associated with this pod.
     *
     * Use `addVolume` to add volumes.
     */
    readonly volumes: Volume[];
    /**
     * Restart policy for all containers within the pod.
     */
    readonly restartPolicy?: RestartPolicy;
    /**
     * The service account used to run this pod.
     */
    readonly serviceAccount?: IServiceAccount;
    /**
     * Add a container to the pod.
     *
     * @param container The container.
     */
    addContainer(container: ContainerProps): Container;
    /**
     * Add a volume to the pod.
     *
     * @param volume The volume.
     */
    addVolume(volume: Volume): void;
}
/**
 * Represents a resource that can be configured with a kuberenets pod template. (e.g `Deployment`, `Job`, ...).
 *
 * Use the `PodTemplate` class as an implementation helper.
 */
export interface IPodTemplate extends IPodSpec {
    /**
     * Provides read/write access to the underlying pod metadata of the resource.
     */
    readonly podMetadata: ApiObjectMetadataDefinition;
}
/**
 * Provides read/write capabilities ontop of a `PodSpecProps`.
 */
export declare class PodSpec implements IPodSpec {
    readonly restartPolicy?: RestartPolicy;
    readonly serviceAccount?: IServiceAccount;
    private readonly _containers;
    private readonly _volumes;
    constructor(props?: PodSpecProps);
    get containers(): Container[];
    get volumes(): Volume[];
    addContainer(container: ContainerProps): Container;
    addVolume(volume: Volume): void;
    /**
     * @internal
     */
    _toPodSpec(): k8s.PodSpec;
}
/**
 * Properties of a `PodTemplate`.
 *
 * Adds metadata information on top of the spec.
 */
export interface PodTemplateProps extends PodSpecProps {
    /**
     * The pod metadata.
     */
    readonly podMetadata?: ApiObjectMetadata;
}
/**
 * Provides read/write capabilities ontop of a `PodTemplateProps`.
 */
export declare class PodTemplate extends PodSpec implements IPodTemplate {
    readonly podMetadata: ApiObjectMetadataDefinition;
    constructor(props?: PodTemplateProps);
    /**
     * @internal
     */
    _toPodTemplateSpec(): k8s.PodTemplateSpec;
}
/**
 * Properties for initialization of `Pod`.
 */
export interface PodProps extends ResourceProps, PodSpecProps {
}
/**
 * Properties of a `PodSpec`.
 */
export interface PodSpecProps {
    /**
     * List of containers belonging to the pod. Containers cannot currently be
     * added or removed. There must be at least one container in a Pod.
     *
     * You can add additionnal containers using `podSpec.addContainer()`
     *
     * @default - No containers. Note that a pod spec must include at least one container.
     */
    readonly containers?: ContainerProps[];
    /**
     * List of volumes that can be mounted by containers belonging to the pod.
     *
     * You can also add volumes later using `podSpec.addVolume()`
     *
     * @see https://kubernetes.io/docs/concepts/storage/volumes
     *
     * @default - No volumes.
     */
    readonly volumes?: Volume[];
    /**
     * Restart policy for all containers within the pod.
     *
     * @see https://kubernetes.io/docs/concepts/workloads/pods/pod-lifecycle/#restart-policy
     *
     * @default RestartPolicy.ALWAYS
     */
    readonly restartPolicy?: RestartPolicy;
    /**
     * A service account provides an identity for processes that run in a Pod.
     *
     * When you (a human) access the cluster (for example, using kubectl), you are
     * authenticated by the apiserver as a particular User Account (currently this
     * is usually admin, unless your cluster administrator has customized your
     * cluster). Processes in containers inside pods can also contact the
     * apiserver. When they do, they are authenticated as a particular Service
     * Account (for example, default).
     *
     * @see https://kubernetes.io/docs/tasks/configure-pod-container/configure-service-account/
     *
     * @default - No service account.
     */
    readonly serviceAccount?: IServiceAccount;
}
/**
 * Pod is a collection of containers that can run on a host. This resource is
 * created by clients and scheduled onto hosts.
 */
export declare class Pod extends Resource implements IPodSpec {
    /**
     * @see base.Resource.apiObject
     */
    protected readonly apiObject: ApiObject;
    private readonly _spec;
    constructor(scope: Construct, id: string, props?: PodProps);
    get containers(): Container[];
    get volumes(): Volume[];
    get restartPolicy(): RestartPolicy | undefined;
    get serviceAccount(): IServiceAccount | undefined;
    addContainer(container: ContainerProps): Container;
    addVolume(volume: Volume): void;
}
/**
 * Restart policy for all containers within the pod.
 */
export declare enum RestartPolicy {
    /**
     * Always restart the pod after it exits.
     */
    ALWAYS = "Always",
    /**
     * Only restart if the pod exits with a non-zero exit code.
     */
    ON_FAILURE = "OnFailure",
    /**
     * Never restart the pod.
     */
    NEVER = "Never"
}
