"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.IngressBackend = exports.Ingress = exports.HttpIngressPathType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const base_1 = require("./base");
const k8s = require("./imports/k8s");
/**
 * Specify how the path is matched against request paths.
 *
 * @see https://kubernetes.io/docs/concepts/services-networking/ingress/#path-types
 */
var HttpIngressPathType;
(function (HttpIngressPathType) {
    /**
     * Matches the URL path exactly.
     */
    HttpIngressPathType["PREFIX"] = "Prefix";
    /**
     * Matches based on a URL path prefix split by '/'.
     */
    HttpIngressPathType["EXACT"] = "Exact";
    /**
     * Matching is specified by the underlying IngressClass.
     */
    HttpIngressPathType["IMPLEMENTATION_SPECIFIC"] = "ImplementationSpecific";
})(HttpIngressPathType = exports.HttpIngressPathType || (exports.HttpIngressPathType = {}));
/**
 * Ingress is a collection of rules that allow inbound connections to reach the
 * endpoints defined by a backend. An Ingress can be configured to give services
 * externally-reachable urls, load balance traffic, terminate SSL, offer name
 * based virtual hosting etc.
 */
class Ingress extends base_1.Resource {
    constructor(scope, id, props = {}) {
        var _c;
        super(scope, id);
        this._rulesPerHost = {};
        this._tlsConfig = [];
        this.apiObject = new k8s.KubeIngress(this, 'Resource', {
            metadata: props.metadata,
            spec: {
                defaultBackend: cdk8s_1.Lazy.any({ produce: () => { var _c; return (_c = this._defaultBackend) === null || _c === void 0 ? void 0 : _c._toKube(); } }),
                rules: cdk8s_1.Lazy.any({ produce: () => this.synthRules() }),
                tls: cdk8s_1.Lazy.any({ produce: () => this.tlsConfig() }),
            },
        });
        if (props.defaultBackend) {
            this.addDefaultBackend(props.defaultBackend);
        }
        this.addRules(...(_c = props.rules) !== null && _c !== void 0 ? _c : []);
        if (props.tls) {
            this.addTls(props.tls);
        }
    }
    onValidate() {
        if (!this._defaultBackend && Object.keys(this._rulesPerHost).length === 0) {
            return ['ingress with no rules or default backend'];
        }
        return [];
    }
    /**
     * Defines the default backend for this ingress. A default backend capable of
     * servicing requests that don't match any rule.
     *
     * @param backend The backend to use for requests that do not match any rule.
     */
    addDefaultBackend(backend) {
        this.addRules({ backend });
    }
    /**
     * Specify a default backend for a specific host name. This backend will be used as a catch-all for requests
     * targeted to this host name (the `Host` header matches this value).
     *
     * @param host The host name to match
     * @param backend The backend to route to
     */
    addHostDefaultBackend(host, backend) {
        if (!host) {
            throw new Error('host must not be an empty string');
        }
        this.addRules({ host, backend });
    }
    /**
     * Adds an ingress rule applied to requests to a specific host and a specific
     * HTTP path (the `Host` header matches this value).
     *
     * @param host The host name
     * @param path The HTTP path
     * @param backend The backend to route requests to
     * @param pathType How the path is matched against request paths
     */
    addHostRule(host, path, backend, pathType) {
        if (!host) {
            throw new Error('host must not be an empty string');
        }
        this.addRules({ host, backend, path, pathType });
    }
    /**
     * Adds an ingress rule applied to requests sent to a specific HTTP path.
     *
     * @param path The HTTP path
     * @param backend The backend to route requests to
     * @param pathType How the path is matched against request paths
     */
    addRule(path, backend, pathType) {
        this.addRules({ backend, path, pathType });
    }
    /**
     * Adds rules to this ingress.
     * @param rules The rules to add
     */
    addRules(...rules) {
        var _c, _d, _e, _f;
        for (const rule of rules) {
            // default backend is not really a rule
            if (!rule.host && !rule.path) {
                if (this._defaultBackend) {
                    throw new Error('a default backend is already defined for this ingress');
                }
                this._defaultBackend = rule.backend;
                continue;
            }
            const host = (_c = rule.host) !== null && _c !== void 0 ? _c : '';
            const backend = rule.backend;
            const path = (_d = rule.path) !== null && _d !== void 0 ? _d : '/';
            const pathType = (_e = rule.pathType) !== null && _e !== void 0 ? _e : HttpIngressPathType.PREFIX;
            if (path && !path.startsWith('/')) {
                throw new Error(`ingress paths must begin with a "/": ${path}`);
            }
            const routes = this._rulesPerHost[host] = (_f = this._rulesPerHost[host]) !== null && _f !== void 0 ? _f : [];
            // check if we already have a rule for this host/path
            if (routes.find(r => r.path === path)) {
                throw new Error(`there is already an ingress rule for ${host}${path}`);
            }
            routes.push({
                backend: backend._toKube(),
                path,
                pathType,
            });
        }
    }
    synthRules() {
        const rules = new Array();
        for (const [host, paths] of Object.entries(this._rulesPerHost)) {
            rules.push({
                host: host ? host : undefined,
                http: { paths: paths.sort(sortByPath) },
            });
        }
        return rules.length > 0 ? rules : undefined;
    }
    addTls(tls) {
        this._tlsConfig.push(...tls);
    }
    tlsConfig() {
        var _c;
        if (this._tlsConfig.length == 0) {
            return undefined;
        }
        const tls = new Array();
        for (const entry of this._tlsConfig) {
            tls.push({
                hosts: entry.hosts,
                secretName: (_c = entry.secret) === null || _c === void 0 ? void 0 : _c.name,
            });
        }
        return tls;
    }
}
exports.Ingress = Ingress;
_a = JSII_RTTI_SYMBOL_1;
Ingress[_a] = { fqn: "cdk8s-plus-22.Ingress", version: "1.0.0-beta.142" };
/**
 * The backend for an ingress path.
 */
class IngressBackend {
    constructor(backend) {
        this.backend = backend;
    }
    /**
     * A Kubernetes `Service` to use as the backend for this path.
     * @param service The service object.
     */
    static fromService(service, options = {}) {
        if (service.ports.length === 0) {
            throw new Error('service does not expose any ports');
        }
        let servicePort;
        if (service.ports.length === 1) {
            servicePort = service.ports[0].port;
        }
        else {
            if (options.port !== undefined) {
                const found = service.ports.find(p => p.port === options.port);
                if (found) {
                    servicePort = found.port;
                }
                else {
                    throw new Error(`service exposes ports ${service.ports.map(p => p.port).join(',')} but backend is defined to use port ${options.port}`);
                }
            }
            else {
                throw new Error(`unable to determine service port since service exposes multiple ports: ${service.ports.map(x => x.port).join(',')}`);
            }
        }
        if (options.port !== undefined && servicePort !== options.port) {
            throw new Error(`backend defines port ${options.port} but service exposes port ${servicePort}`);
        }
        return new IngressBackend({
            service: {
                name: service.name,
                port: { number: servicePort },
            },
        });
    }
    /**
     * @internal
     */
    _toKube() { return this.backend; }
}
exports.IngressBackend = IngressBackend;
_b = JSII_RTTI_SYMBOL_1;
IngressBackend[_b] = { fqn: "cdk8s-plus-22.IngressBackend", version: "1.0.0-beta.142" };
function sortByPath(lhs, rhs) {
    var _c, _d;
    const p1 = (_c = lhs.path) !== null && _c !== void 0 ? _c : '';
    const p2 = (_d = rhs.path) !== null && _d !== void 0 ? _d : '';
    return p1.localeCompare(p2);
}
//# sourceMappingURL=data:application/json;base64,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