import tqdm
import numpy as np
import matplotlib.pyplot as plt
from matplotlib.animation import FuncAnimation, writers
from matplotlib.collections import LineCollection
from sklearn.neighbors import NearestNeighbors
from scipy.stats import norm as normal


class Streamlines(object):
    """
    Copyright (c) 2011 Raymond Speth.
    Permission is hereby granted, free of charge, to any person obtaining a
    copy of this software and associated documentation files (the "Software"),
    to deal in the Software without restriction, including without limitation
    the rights to use, copy, modify, merge, publish, distribute, sublicense,
    and/or sell copies of the Software, and to permit persons to whom the
    Software is furnished to do so, subject to the following conditions:
    The above copyright notice and this permission notice shall be included in
    all copies or substantial portions of the Software.
    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
    FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
    DEALINGS IN THE SOFTWARE.
    See: http://web.mit.edu/speth/Public/streamlines.py
    """
    def __init__(self, X, Y, U, V, res=0.125,
                 spacing=2, maxLen=2500, detectLoops=False):
        """
        Compute a set of streamlines covering the given velocity field.
        X and Y - 1D or 2D (e.g. generated by np.meshgrid) arrays of the
                  grid points. The mesh spacing is assumed to be uniform
                  in each dimension.
        U and V - 2D arrays of the velocity field.
        res - Sets the distance between successive points in each
              streamline (same units as X and Y)
        spacing - Sets the minimum density of streamlines, in grid points.
        maxLen - The maximum length of an individual streamline segment.
        detectLoops - Determines whether an attempt is made to stop extending
                      a given streamline before reaching maxLen points if
                      it forms a closed loop or reaches a velocity node.
        Plots are generated with the 'plot' or 'plotArrows' methods.
        """
        self.spacing = spacing
        self.detectLoops = detectLoops
        self.maxLen = maxLen
        self.res = res
        xa = np.asanyarray(X)
        ya = np.asanyarray(Y)
        self.x = xa if xa.ndim == 1 else xa[0]
        self.y = ya if ya.ndim == 1 else ya[:, 0]
        self.u = U
        self.v = V
        self.dx = (self.x[-1] - self.x[0]) / (self.x.size - 1)  # assume a regular grid
        self.dy = (self.y[-1] - self.y[0]) / (self.y.size - 1)  # assume a regular grid
        self.dr = self.res * np.sqrt(self.dx * self.dy)
        # marker for which regions have contours
        self.used = np.zeros(self.u.shape, dtype=bool)
        self.used[0] = True
        self.used[-1] = True
        self.used[:, 0] = True
        self.used[:, -1] = True
        # Don't try to compute streamlines in regions where there is no velocity data
        for i in range(self.x.size):
            for j in range(self.y.size):
                if self.u[j, i] == 0.0 and self.v[j, i] == 0.0:
                    self.used[j, i] = True
        # Make the streamlines
        self.streamlines = []
        while not self.used.all():
            nz = np.transpose(np.logical_not(self.used).nonzero())
            # Make a streamline starting at the first unrepresented grid point
            self.streamlines.append(self._makeStreamline(self.x[nz[0][1]],
                                                         self.y[nz[0][0]]))

    def _interp(self, x, y):
        """ Compute the velocity at point (x,y) """
        i = (x - self.x[0]) / self.dx
        ai = i % 1
        j = (y - self.y[0]) / self.dy
        aj = j % 1
        i, j = int(i), int(j)
        # Bilinear interpolation
        u = (self.u[j, i] * (1 - ai) * (1 - aj) +
             self.u[j, i + 1] * ai * (1 - aj) +
             self.u[j + 1, i] * (1 - ai) * aj +
             self.u[j + 1, i + 1] * ai * aj)
        v = (self.v[j, i] * (1 - ai) * (1 - aj) +
             self.v[j, i + 1] * ai * (1 - aj) +
             self.v[j + 1, i] * (1 - ai) * aj +
             self.v[j + 1, i + 1] * ai * aj)
        self.used[j:j + self.spacing, i:i + self.spacing] = True
        return u, v

    def _makeStreamline(self, x0, y0):
        """
        Compute a streamline extending in both directions from the given point.
        """
        sx, sy = self._makeHalfStreamline(x0, y0, 1)  # forwards
        rx, ry = self._makeHalfStreamline(x0, y0, -1)  # backwards
        rx.reverse()
        ry.reverse()
        return rx + [x0] + sx, ry + [y0] + sy

    def _makeHalfStreamline(self, x0, y0, sign):
        """
        Compute a streamline extending in one direction from the given point.
        """
        xmin = self.x[0]
        xmax = self.x[-1]
        ymin = self.y[0]
        ymax = self.y[-1]
        sx = []
        sy = []
        x = x0
        y = y0
        i = 0
        while xmin < x < xmax and ymin < y < ymax:
            u, v = self._interp(x, y)
            theta = np.arctan2(v, u)
            x += sign * self.dr * np.cos(theta)
            y += sign * self.dr * np.sin(theta)
            sx.append(x)
            sy.append(y)
            i += 1
            if self.detectLoops and i % 10 == 0 and self._detectLoop(sx, sy):
                break
            if i > self.maxLen / 2:
                break
        return sx, sy

    def _detectLoop(self, xVals, yVals):
        """ Detect closed loops and nodes in a streamline. """
        x = xVals[-1]
        y = yVals[-1]
        D = np.array([np.hypot(x - xj, y - yj)
                      for xj, yj in zip(xVals[:-1], yVals[:-1])])
        return (D < 0.9 * self.dr).any()


def compute_velocity_on_grid(
        X_emb,
        V_emb,
        density=None,
        smooth=None,
        n_neighbors=None,
        min_mass=None,
        autoscale=True,
        adjust_for_stream=True,
        cutoff_perc=None,
):
    """
    Compute velocity on a grid from embedded coordinates and velocities.
    Adapted from scVelo Volker Bergen Nature Biotechnology 2020
    """
    # remove invalid cells
    idx_valid = np.isfinite(X_emb.sum(1) + V_emb.sum(1))
    X_emb = X_emb[idx_valid]
    V_emb = V_emb[idx_valid]
    
    # prepare grid
    n_obs, n_dim = X_emb.shape
    density = 1 if density is None else density
    smooth = 0.5 if smooth is None else smooth
    grs = []
    for dim_i in range(n_dim):
        m, M = np.min(X_emb[:, dim_i]), np.max(X_emb[:, dim_i])
        m = m - 0.01 * np.abs(M - m)
        M = M + 0.01 * np.abs(M - m)
        gr = np.linspace(m, M, int(50 * density))
        grs.append(gr)
    
    meshes_tuple = np.meshgrid(*grs)
    X_grid = np.vstack([i.flat for i in meshes_tuple]).T
    
    # estimate grid velocities
    if n_neighbors is None:
        n_neighbors = min(int(n_obs / 50), 20)
    nn = NearestNeighbors(n_neighbors=n_neighbors, n_jobs=-1)
    nn.fit(X_emb)
    dists, neighs = nn.kneighbors(X_grid)
    scale = np.mean([(g[1] - g[0]) for g in grs]) * smooth
    weight = normal.pdf(x=dists, scale=scale)
    p_mass = weight.sum(1)
    V_grid = (V_emb[neighs] * weight[:, :, None]).sum(1)
    V_grid /= np.maximum(1, p_mass)[:, None]
    
    if min_mass is None:
        min_mass = 1
    if adjust_for_stream == True:
        X_grid = np.stack([np.unique(X_grid[:, 0]), np.unique(X_grid[:, 1])])
        ns = int(np.sqrt(len(V_grid[:, 0])))
        V_grid = V_grid.T.reshape(2, ns, ns)
        mass = np.sqrt((V_grid ** 2).sum(0))
        min_mass = 10 ** (min_mass - 6)
        min_mass = np.clip(min_mass, None, np.max(mass) * 0.9)
        cutoff = mass.reshape(V_grid[0].shape) < min_mass
        if cutoff_perc is None:
            cutoff_perc = 5
        length = np.sum(np.mean(np.abs(V_emb[neighs]), axis=1), axis=1).T
        length = length.reshape(ns, ns)
        cutoff |= length < np.percentile(length, cutoff_perc)
        V_grid[0][cutoff] = np.nan
    else:
        min_mass *= np.percentile(p_mass, 99) / 100
        X_grid, V_grid = X_grid[p_mass > min_mass], V_grid[p_mass > min_mass]
    
    return X_grid, V_grid


def nan_helper(y):
    """Helper to handle indices and logical indices of NaNs"""
    return np.isnan(y), lambda z: z.nonzero()[0]


def animate_streamplot(X_grid, V_grid, adata=None, 
                      # Animation parameters
                      n_frames=20, interval=40, fps=25,
                      # Visual parameters  
                      alpha_animate=0.7, cmap_stream='Blues', linewidth=0.5,
                      segment_length=1, figsize=(4, 4),
                      # Background plotting parameters
                      basis='X_umap', color='celltype', palette=None,
                      background_color='black',
                      # Save parameters
                      saveto='animation.gif', show_plot=True,
                      # Streamline parameters
                      streamline_res=0.125, streamline_spacing=2, streamline_maxLen=2500):
    """
    Create an animated streamplot visualization.
    
    Parameters:
    -----------
    X_grid : array-like
        Grid coordinates [X, Y] where X and Y are 2D arrays
    V_grid : array-like  
        Velocity field [U, V] where U and V are 2D arrays
    adata : AnnData object, optional
        Annotated data object for background scatter plot
    n_frames : int, default=20
        Number of animation frames
    interval : int, default=40
        Delay between frames in milliseconds
    fps : int, default=25
        Frames per second for saved animation
    alpha_animate : float, default=0.7
        Alpha transparency for streamlines
    cmap_stream : str, default='Blues'
        Colormap for streamlines
    linewidth : float, default=0.5
        Width of streamlines
    segment_length : float, default=1
        Length scaling for streamline segments
    figsize : tuple, default=(4, 4)
        Figure size
    basis : str, default='X_umap'
        Basis for background scatter plot
    color : str, default='celltype'
        Color variable for background scatter plot
    palette : dict, optional
        Color palette for background scatter plot
    saveto : str, default='animation.gif'
        Output filename for animation
    show_plot : bool, default=True
        Whether to display the plot
    streamline_res : float, default=0.125
        Resolution for streamline computation
    streamline_spacing : int, default=2
        Spacing for streamline computation
    streamline_maxLen : int, default=2500
        Maximum length for streamlines
        
    Returns:
    --------
    animation : FuncAnimation
        The matplotlib animation object
    """
    
    print("Creating streamlines...")
    streamlines = Streamlines(X_grid[0], X_grid[1], V_grid[0], V_grid[1],
                             res=streamline_res, spacing=streamline_spacing, 
                             maxLen=streamline_maxLen)
    
    # Set up the figure
    fig = plt.figure(figsize=figsize)
    fig.patch.set_facecolor(background_color)
    ax = plt.subplot(1, 1, 1)
    ax.set_facecolor(background_color)
    
    # Add background scatter plot if data provided
    if adata is not None:
        try:
            import scanpy as sc
            sc.pl.embedding(
                adata,
                basis=basis,
                color=color,
                palette=palette,
                ax=ax,
                show=False,
                legend_loc=None,
                frameon=False,
                title=''
            )
        except ImportError:
            print("Warning: scanpy not available, skipping background scatter plot")
    
    # Prepare animation data
    lengths = []
    colors = []
    lines = []
    linewidths = []
    
    print(f"Processing {len(streamlines.streamlines)} streamlines...")
    for i, (x, y) in enumerate(streamlines.streamlines):
        # Handle NaN values
        x_ = np.array(x)
        nans, func_ = nan_helper(x_)
        if np.any(nans) and np.any(~nans):
            x_[nans] = np.interp(func_(nans), func_(~nans), x_[~nans])
        
        y_ = np.array(y)
        nans, func_ = nan_helper(y_)
        if np.any(nans) and np.any(~nans):
            y_[nans] = np.interp(func_(nans), func_(~nans), y_[~nans])
        
        # Create line segments
        points = np.array([x_, y_]).T.reshape(-1, 1, 2)
        if len(points) < 2:
            continue
            
        segments = np.concatenate([points[:-1], points[1:]], axis=1)
        n = len(segments)
        
        # Calculate segment lengths and cumulative distances
        D = np.sqrt(((points[1:] - points[:-1]) ** 2).sum(axis=-1)) / segment_length
        np.random.seed(i + 42)
        L = D.cumsum().reshape(n, 1) + np.random.uniform(0, 1)
        
        # Initialize colors
        C = np.zeros((n, 4))
        C[::-1] = (L * 1.5) % 1
        C[:, 3] = alpha_animate
        
        lw = L.flatten().tolist()
        
        # Create line collection
        line = LineCollection(segments, color=C, linewidth=linewidth)
        
        lengths.append(L)
        colors.append(C)
        linewidths.append(lw)
        lines.append(line)
        
        ax.add_collection(line)
    
    # Set axis limits
    ax.set_xlim(X_grid[0].min(), X_grid[0].max())
    ax.set_ylim(X_grid[1].min(), X_grid[1].max())
    ax.set_xticks([])
    ax.set_yticks([])
    plt.tight_layout()
    
    print(f"Created {len(lines)} animated streamlines")
    
    # Animation update function
    def update(frame_no):
        cmap = plt.cm.get_cmap(cmap_stream)
        if background_color == 'black':
            cmap_colors = cmap(np.linspace(0, 0.1, 100))  # Light portion of colormap
        else:
            cmap_colors = cmap(np.linspace(0, 1, 100))  # Light portion of colormap
        
        for i in range(len(lines)):
            if len(lengths[i]) == 0:
                continue
                
            lengths[i] -= 0.05
            
            # Update colors with clipping to avoid hard blacks
            colors[i][::-1] = np.clip(0.1 + (lengths[i] * 1.5) % 1, 0.2, 0.9)
            colors[i][:, 3] = alpha_animate
            
            # Update line widths
            temp = (lengths[i] * 1) % 2
            linewidths[i] = temp.flatten().tolist()
            
            # Apply colormap
            for row in range(colors[i].shape[0]):
                color_idx = int(colors[i][row, 0] * 99)  # Map to colormap index
                color_idx = np.clip(color_idx, 0, 99)
                colors[i][row, :3] = cmap_colors[color_idx][:3]
            
            colors[i][:, 3] = alpha_animate
            lines[i].set_linewidth(linewidths[i])
            lines[i].set_color(colors[i])
    
    # Create animation
    print(f"Creating animation with {n_frames} frames...")
    animation = FuncAnimation(fig, update, frames=n_frames, interval=interval, repeat=True)
    
    # Save animation
    if saveto:
        try:
            animation.save(saveto, writer='pillow', fps=fps,
                          savefig_kwargs={'facecolor': 'black'})
            print(f"Animation saved to {saveto}")
        except Exception as e:
            print(f"Could not save animation: {e}")
            print("You can still view the animation interactively")
    
    if show_plot:
        plt.show()
    
    return animation


# Example usage:
# animation = animate_streamplot(X_grid, V_grid, adata, 
#                               color='celltype', palette=color_dict,
#                               saveto='my_animation.gif')