"""
General data access tools
"""

import os
import json
from pathlib import Path
from multiprocessing import Pool

import numpy as np
import pandas as pd
import networkx as nx


def load_cells(DATA_DIR, reflect=True):
    """
    Load dataframe of cells with soma position, and graphs
    reflect_about_midline (bool) Reflects cells about midline
    """
    filePaths, genotypeDict = get_filepaths(DATA_DIR)
    graphs = load_graphs(filePaths)
    dataDF = load_data_to_dataframe(filePaths, genotypeDict, graphs)
    if reflect:
        dataDF, graph = reflect_about_midline(dataDF, graphs)
    return dataDF, graphs


def get_filepaths(DATA_DIR):
    """
    Loads a list of filepaths of cells, and a dictionary of genotypes
    """
    genotypeDict = {}
    folderPath = Path(DATA_DIR)
    filePaths = []
    for folder in os.listdir(folderPath):
        dataPath = folderPath / folder / "Complete_annotated"
        if os.path.exists(dataPath):
            filePaths.append(
                [
                    os.path.join(dataPath, fn)
                    for fn in os.listdir(dataPath)
                    if fn.endswith(".json")
                ]
            )
            subjectPath = dataPath.parent / "subject.json"
            with open(subjectPath) as f:
                subjectJSON = json.load(f)
            genotypeDict[subjectJSON["subject_id"]] = subjectJSON["genotype"]

    filePaths = [file for files in filePaths for file in files]
    return filePaths, genotypeDict


def load_data_to_dataframe(filePaths, genotypeDict, graphs):
    """
    Build a dataframe of cells
    filePaths - list of file locations
    genotypeDict - dictionary of genotype information
    graphs - networkx graphs
    """
    i = 0
    datasetDicts = {}
    for key, val in graphs.items():
        neuronID, sample, annotator = key.split("-")
        try:
            soma = [
                node
                for node in val.nodes()
                if val.nodes[node]["structure_id"] == 1
            ]  # Get soma nodes
            assert len(soma) == 1
            genotype = genotypeDict[sample]
            x, y, z = val.nodes[soma[0]]["pos"]
        except Exception:
            print(
                f"Error finding structures for: {key}, dropping from dataframe"
            )
            continue
        neuronDict = {
            "Graph": key,
            "ID": neuronID,
            "Sample": sample,
            "Annotator": annotator,
            "Genotype": genotype,
            "somaAP": x,
            "somaDV": y,
            "somaML": z,
        }
        datasetDicts[i] = neuronDict
        i = i + 1

    dataDF = pd.DataFrame.from_dict(datasetDicts, orient="index")

    # Sort by the name of the graph, because mutlipool loading is stochastic
    dataDF = dataDF.sort_values(by="Graph")
    return dataDF


def euclidean_distance(node1, node2):
    """
    Calculate the Euclidean distance between two nodes.

    Parameters:
    node1, node2 (dict): Nodes with 'pos' key containing x, y, z coordinates.

    Returns:
    float: Euclidean distance between node1 and node2.
    """
    pos1 = np.array(node1["pos"])
    pos2 = np.array(node2["pos"])
    return np.linalg.norm(pos1 - pos2)


def add_node_to_graph(graph, node):
    """
    Add a node with attributes to the graph.

    Parameters:
    graph (nx.DiGraph): The graph to which the node will be added.
    node (dict): Node data.
    """
    graph.add_node(
        node["sampleNumber"],
        pos=(node["x"], node["y"], node["z"]),
        radius=node["radius"],
        structure_id=node["structureIdentifier"],
        allen_id=node["allenId"],
    )


def add_edge_to_graph(graph, parent, child):
    """
    Add an edge between parent and child nodes in the graph,
    with weight as Euclidean distance.

    Parameters:
    graph (nx.DiGraph): The graph to which the edge will be added.
    parent, child (int): The sampleNumbers of the parent and child nodes.
    """
    graph.add_edge(
        parent,
        child,
        weight=euclidean_distance(graph.nodes()[parent], graph.nodes()[child]),
    )


def json_to_digraph(file_path):
    """
    Load a neuronal reconstruction from a JSON file into a NetworkX graph.

    The JSON file contains SWC data with additional brain region
    information for each node. The graph will be a directed tree.

    Parameters:
    file_path (str): Path to the JSON file containing reconstruction data.

    Returns:
    nx.DiGraph: A directed graph representing the neuronal tree.
    """
    try:
        with open(file_path, "r") as file:
            data = json.load(file)
    except IOError as e:
        print(f"Error opening file: {e}")
        return None

    # Certain JSON files may have a single 'neuron' object
    # instead of a 'neurons' array
    neuron_data = data["neuron"] if "neuron" in data else data["neurons"][0]

    axon_graph, dendrite_graph = nx.DiGraph(), nx.DiGraph()

    for structure, graph in [
        ("dendrite", dendrite_graph),
        ("axon", axon_graph),
    ]:
        if structure not in neuron_data:
            # Some reconstructions may be missing an axon or dendrite tracing
            print(f"Missing structure {structure} for {file_path}")
            continue
        for node in sorted(
            neuron_data[structure], key=lambda x: x["sampleNumber"]
        ):
            add_node_to_graph(graph, node)
            if node["parentNumber"] != -1:
                add_edge_to_graph(
                    graph, node["parentNumber"], node["sampleNumber"]
                )

    if dendrite_graph.nodes() and axon_graph.nodes():
        # Remove duplicate soma node from axon graph
        axon_graph.remove_node(1)

    # The sampleNumber starts at 1 for both axon and dendrite, so
    # relabel axon nodes to avoid key collisions when merging the graphs,.
    first_axon_label = (
        max(dendrite_graph.nodes()) + 1 if dendrite_graph.nodes() else 1
    )
    joined_graph = nx.union(
        dendrite_graph,
        nx.convert_node_labels_to_integers(
            axon_graph, first_label=first_axon_label
        ),
    )
    roots = [n for n in joined_graph if joined_graph.in_degree(n) == 0]
    # Link the dendrite to the axon
    if len(roots) == 2:
        add_edge_to_graph(joined_graph, roots[0], roots[1])

    return file_path, joined_graph


# Define a function for filtering the graph based on attribute values
def get_subgraph(G, attribute, values):
    """
    Extract a subgraph from the given graph based on
    specified attribute values.

    Parameters:
    G (nx.Graph): The original graph from which to extract the subgraph.
    attribute (str): The node attribute used for filtering.
    values (tuple): A tuple of attribute values to include in the subgraph.

    Returns:
    nx.Graph: A subgraph of G containing only nodes with the
    specified attribute values.
    """
    filtered_nodes = [
        node
        for node, attr in G.nodes(data=True)
        if attr.get(attribute) in values
    ]
    return G.subgraph(filtered_nodes)


def load_graphs(filepaths):
    """
    Load all JSON files in the given directory as graphs using multiprocessing.

    Parameters:
    directory_path (str): Path to the directory containing JSON files.

    Returns:
    list of nx.Graph: A list of graphs loaded from the JSON files.
    """
    # Use multiprocessing pool to load graphs in parallel
    with Pool() as pool:
        graphs = pool.map(json_to_digraph, filepaths)

    # Organize into dictionary
    return {
        os.path.splitext(os.path.split(fn)[1])[0]: graph
        for fn, graph in graphs
    }


def get_cells_in_regions(manifest_path, acronyms):
    """
    Get cells located in a region
    """
    # Load the CSV file
    df = pd.read_csv(manifest_path)

    # If a single acronym is provided, convert it to a list
    if isinstance(acronyms, str):
        acronyms = [acronyms]

    # Filter the dataframe for the specified acronyms and get the filenames
    filtered_df = df[df["soma_acronym"].isin(acronyms)]
    filenames = filtered_df["filename"].tolist()
    filtered_acronyms = filtered_df["soma_acronym"].tolist()

    return filenames, filtered_acronyms


def reflect_about_midline(dataDF, graphs):
    """
    reflect cells about brain midline
    """
    mlMidline = 5700
    mlReflection = mlMidline * 2

    # Add column for original soma side
    dataDF["somaOnRight"] = dataDF["somaML"] > mlMidline

    # Reflect rightside graphs across midline
    for index, row in dataDF.iterrows():
        # Grab neurons with somas on right
        if row["somaOnRight"]:
            graph = graphs[row["Graph"]]
            # Update soma position in dataframe
            dataDF.loc[index, "somaML"] = (
                mlReflection - dataDF.loc[index, "somaML"]
            )
            # Reflect every node's position along ML axis
            for node in graph.nodes:
                graph.nodes[node]["pos"] = (
                    graph.nodes[node]["pos"][0],
                    graph.nodes[node]["pos"][1],
                    mlReflection - graph.nodes[node]["pos"][2],
                )
    return dataDF, graph
