"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PipelineWorkflow = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const helpers_internal_1 = require("aws-cdk-lib/pipelines/lib/helpers-internal");
const github_actions_cdk_1 = require("github-actions-cdk");
const jobs_1 = require("./jobs");
const steps_1 = require("./steps");
/**
 * Represents a GitHub Actions workflow to manage CDK pipeline jobs for synthesizing, publishing, and deploying AWS resources.
 *
 * @remarks
 * Extends `Workflow` from `github-actions-cdk`, and provides structured job orchestration based on the AWS CDK pipeline graph.
 */
class PipelineWorkflow extends github_actions_cdk_1.Workflow {
    static [JSII_RTTI_SYMBOL_1] = { fqn: "@github-actions-cdk/aws-cdk.PipelineWorkflow", version: "0.0.22" };
    pipelineJobProps;
    stackOptions;
    assetHashMap = {};
    /**
     * Initializes a new `PipelineWorkflow`.
     *
     * @param scope - The scope within which this construct is created.
     * @param id - The unique identifier for this workflow.
     * @param props - Configuration properties for the pipeline workflow.
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        this.pipelineJobProps = {
            awsCredentials: props.awsCredentials,
            dockerCredentials: props.dockerCredentials,
            versionOverrides: props.versionOverrides,
            cdkoutDir: props.cdkoutDir,
        };
        this.stackOptions = props.stackOptions;
        const structure = new helpers_internal_1.PipelineGraph(props.pipeline, {
            selfMutation: false,
            publishTemplate: true,
            prepareStep: false,
            singlePublisherPerAssetType: props.singlePublisherPerAssetType ?? false,
        });
        for (const stageNode of flatten(structure.graph.sortedChildren())) {
            if (!(0, helpers_internal_1.isGraph)(stageNode)) {
                throw new Error(`Top-level children must be graphs, received: '${stageNode}'`);
            }
            const tranches = stageNode.sortedLeaves();
            for (const tranche of tranches) {
                for (const node of tranche) {
                    switch (node.data?.type) {
                        case "step":
                            if (node.data?.isBuildStep && node.data?.step instanceof steps_1.Synth) {
                                this.createSynthJob(node.uniqueId, this.getDependencies(node), node.data.step, props.phases?.preBuild, props.phases?.postBuild);
                            }
                            else if (node.data?.step instanceof steps_1.StageJob) {
                                this.createStageJob(node.uniqueId, this.getDependencies(node), node.data.step);
                            }
                            break;
                        case "publish-assets":
                            this.createPublishJob(node.uniqueId, this.getDependencies(node), node.data.assets, props.phases?.prePublish, props.phases?.postPublish);
                            break;
                        case "execute":
                            this.createDeployJob(node.uniqueId, this.getDependencies(node), node.data.stack);
                            break;
                        default:
                            throw new Error(`Unknown node type: ${node.data?.type}`);
                    }
                }
            }
        }
    }
    /**
     * Creates a job for synthesizing the CDK application.
     *
     * @param id - Unique identifier for the synth job.
     * @param needs - List of dependencies for this job.
     * @param synth - Synth step configuration.
     * @param preBuild - Optional jobs to run before the synth job.
     * @param postBuild - Optional jobs to run after the synth job.
     */
    createSynthJob(id, needs, synth, preBuild, postBuild) {
        new jobs_1.SynthPipelineJob(this, id, {
            name: "Synthesize",
            needs,
            permissions: {
                contents: github_actions_cdk_1.PermissionLevel.READ,
                idToken: this.pipelineJobProps.awsCredentials.permissionLevel(),
            },
            installCommands: synth.installCommands,
            commands: synth.commands,
            env: synth.env,
            preBuild,
            postBuild,
            ...this.pipelineJobProps,
        });
    }
    /**
     * Creates a job for publishing stack assets.
     *
     * @param id - Unique identifier for the publish job.
     * @param needs - List of dependencies for this job.
     * @param assets - List of assets to publish.
     */
    createPublishJob(id, needs, assets, prePublish, postPublish) {
        new jobs_1.PublishPipelineJob(this, id, {
            name: `Publish Assets ${id}`,
            needs,
            permissions: {
                contents: github_actions_cdk_1.PermissionLevel.READ,
                idToken: this.pipelineJobProps.awsCredentials.permissionLevel(),
            },
            assets,
            prePublish,
            postPublish,
            assetHashMap: this.assetHashMap,
            ...this.pipelineJobProps,
        });
    }
    /**
     * Creates a job for deploying a stack to AWS.
     *
     * @param id - Unique identifier for the deploy job.
     * @param needs - List of dependencies for this job.
     * @param stack - Stack deployment information.
     */
    createDeployJob(id, needs, stack) {
        const options = this.stackOptions[stack.stackArtifactId];
        new jobs_1.DeployPipelineJob(this, id, {
            name: `Deploy ${stack.stackArtifactId}`,
            needs,
            environment: options?.environment,
            permissions: {
                contents: github_actions_cdk_1.PermissionLevel.READ,
                idToken: this.pipelineJobProps.awsCredentials.permissionLevel(),
            },
            stack,
            assetHashMap: this.assetHashMap,
            stackOptions: options,
            ...this.pipelineJobProps,
        });
    }
    /**
     * Creates a job for running a stage job in the pipeline.
     *
     * @param id - Unique identifier for the stage job.
     * @param needs - List of dependencies for this job.
     * @param job - Configuration of the stage job.
     */
    createStageJob(id, needs, job) {
        new jobs_1.StagePipelineJob(this, id, {
            name: job.id,
            needs,
            phase: job.props,
            ...job.props,
            ...this.pipelineJobProps,
        });
    }
    /**
     * Retrieves a list of dependencies for a given graph node.
     *
     * @param node - The graph node to analyze for dependencies.
     * @returns An array of unique IDs representing dependencies of the node.
     */
    getDependencies(node) {
        const deps = [];
        for (const dep of node.allDeps) {
            if (dep instanceof helpers_internal_1.Graph) {
                deps.push(...dep.allLeaves().nodes);
            }
            else {
                deps.push(dep);
            }
        }
        return deps.map((dependency) => dependency.uniqueId);
    }
}
exports.PipelineWorkflow = PipelineWorkflow;
/**
 * Utility function to flatten an iterable of arrays into a single iterable.
 *
 * @param xs - The input iterable containing arrays.
 * @returns A flattened iterable.
 */
function* flatten(xs) {
    for (const x of xs) {
        yield* x;
    }
}
//# sourceMappingURL=data:application/json;base64,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