"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitHubActionsPipeline = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const pipelines_1 = require("aws-cdk-lib/pipelines");
const constructs_1 = require("constructs");
const adapter_1 = require("./adapter");
const wave_1 = require("./wave");
const workflow_1 = require("./workflow");
/**
 * Constructs a GitHub Actions pipeline for deploying AWS resources.
 *
 * @remarks
 * The `GitHubActionsPipeline` provides methods to define and manage deployment stages and job waves in
 * a GitHub Actions pipeline, utilizing AWS credentials and CDK output for cloud infrastructure automation.
 */
class GitHubActionsPipeline extends constructs_1.Construct {
    static [JSII_RTTI_SYMBOL_1] = { fqn: "@github-actions-cdk/aws-cdk.GitHubActionsPipeline", version: "0.0.22" };
    innerPipeline;
    /**
     * Constructs a new instance of `GitHubActionsPipeline`.
     *
     * @param scope - The parent construct scope.
     * @param id - Unique identifier for this pipeline construct.
     * @param props - Configuration properties for the pipeline.
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.innerPipeline = new InnerPipeline(this, id, props);
    }
    /**
     * Returns the name of the workflow.
     */
    get workflowName() {
        return this.innerPipeline.workflowName;
    }
    /**
     * Returns the output directory path for the workflow files.
     */
    get workflowOutdir() {
        return this.innerPipeline.workflowOutdir;
    }
    /**
     * Returns the filename for the workflow file.
     */
    get workflowFilename() {
        return this.innerPipeline.workflowFilename;
    }
    /**
     * Adds a stage to the pipeline with GitHub-specific configuration options.
     *
     * @param stage - The CDK Stage to add to the pipeline.
     * @param options - Optional configuration for the stage.
     * @returns Deployment details for the added stage.
     */
    addStage(stage, options = {}) {
        return this.innerPipeline.addStageWithGitHubOptions(stage, options);
    }
    /**
     * Adds a wave of jobs to the pipeline.
     *
     * @param id - Unique identifier for the wave.
     * @param options - Options for configuring the wave.
     * @returns The created GitHub wave instance.
     */
    addWave(id, options = {}) {
        return this.innerPipeline.addGitHubWave(id, options);
    }
}
exports.GitHubActionsPipeline = GitHubActionsPipeline;
/**
 * Inner class extending `PipelineBase` to manage core functionalities of the GitHub Actions pipeline.
 */
class InnerPipeline extends pipelines_1.PipelineBase {
    workflowName;
    workflowOutdir;
    workflowFilename;
    singlePublisherPerAssetType;
    workflowEnv;
    preBuild;
    postBuild;
    prePublish;
    postPublish;
    dockerCredentials;
    versionOverrides;
    awsCredentials;
    stackOptions = {};
    adapter;
    /**
     * Constructs a new instance of `InnerPipeline`.
     *
     * @param scope - The parent construct.
     * @param id - Unique identifier for this inner pipeline instance.
     * @param props - Configuration properties for the pipeline.
     */
    constructor(scope, id, props) {
        super(scope, id, { synth: props.synth });
        this.workflowName = props.workflowName ?? "Deploy";
        this.workflowOutdir = props.workflowOutdir ?? ".github/workflows";
        this.workflowFilename = props.workflowFilename ?? "deploy";
        this.singlePublisherPerAssetType = props.singlePublisherPerAssetType;
        this.workflowEnv = props.workflowEnv;
        this.preBuild = props.preBuild;
        this.postBuild = props.postBuild;
        this.prePublish = props.prePublish;
        this.postPublish = props.postPublish;
        this.dockerCredentials = props.dockerCredentials;
        this.awsCredentials = props.awsCredentials;
        this.versionOverrides = props.versionOverrides;
        this.adapter = new adapter_1.AwsCdkAdapter(this, { outdir: this.workflowOutdir });
    }
    /**
     * Adds a stage deployment from a wave with optional configuration.
     *
     * @param stageDeployment - The stage deployment to add.
     * @param options - Configuration options for the stage.
     */
    addStageFromWave(stageDeployment, options) {
        const stacks = stageDeployment.stacks;
        this.addStackProps(stacks, "capabilities", options?.stackCapabilities);
    }
    /**
     * Adds a stage to the pipeline with GitHub-specific options.
     *
     * @param stage - The CDK Stage to add.
     * @param options - Configuration options for the stage.
     * @returns Deployment details for the added stage.
     */
    addStageWithGitHubOptions(stage, options = {}) {
        const stageDeployment = this.addStage(stage, {
            pre: options.preJobs,
            post: options.postJobs,
        });
        const stacks = stageDeployment.stacks;
        this.addStackProps(stacks, "environment", options?.gitHubEnvironment);
        this.addStackProps(stacks, "capabilities", options?.stackCapabilities);
        return stageDeployment;
    }
    /**
     * Adds a wave of jobs to the pipeline with GitHub-specific options.
     *
     * @param id - Unique identifier for the wave.
     * @param options - Configuration options for the wave.
     * @returns The created GitHub wave instance.
     */
    addGitHubWave(id, options = {}) {
        const wave = new wave_1.GitHubWave(id, this, {
            pre: options.preJobs,
            post: options.postJobs,
        });
        this.waves.push(wave._innerWave);
        return wave;
    }
    /**
     * Builds the pipeline workflow, generating workflow files during CDK synthesis.
     *
     * @remarks
     * This method is invoked to create workflow files required by GitHub Actions, integrating CDK stack details.
     */
    doBuildPipeline() {
        const app = aws_cdk_lib_1.Stage.of(this);
        if (!app) {
            throw new Error("The GitHub Workflow must be defined within an App scope.");
        }
        const names = app.node
            .findAll()
            .filter((node) => aws_cdk_lib_1.Stack.isStack(node))
            .map((stack) => stack.stackName);
        new workflow_1.PipelineWorkflow(this.adapter, this.workflowFilename, {
            name: this.workflowName,
            singlePublisherPerAssetType: this.singlePublisherPerAssetType,
            commentAtTop: this.renderYamlComment(names),
            env: this.workflowEnv,
            pipeline: this,
            stackOptions: this.stackOptions,
            phases: {
                preBuild: this.preBuild,
                postBuild: this.postBuild,
                prePublish: this.prePublish,
                postPublish: this.postPublish,
            },
            cdkoutDir: app.outdir,
            awsCredentials: this.awsCredentials,
            dockerCredentials: this.dockerCredentials,
            versionOverrides: this.versionOverrides,
        });
    }
    /**
     * Renders a YAML comment for the workflow file listing deployed stacks.
     *
     * @param stackNames - List of stack names to include in the comment.
     * @returns A formatted string for the YAML comment header.
     */
    renderYamlComment(stackNames) {
        const header = "Generated by github-actions-cdk, DO NOT EDIT DIRECTLY!\n\n";
        const stackListHeader = "Deployed stacks from this pipeline:\n";
        const stackList = stackNames.map((stack) => `- ${stack}`).join("\n");
        return header + stackListHeader + stackList;
    }
    /**
     * Adds properties to stack options for each stack in the deployment.
     *
     * @param stacks - Array of stack deployments.
     * @param key - Property key to set in the stack options.
     * @param value - Value to assign to the specified key.
     */
    addStackProps(stacks, key, value) {
        if (value === undefined)
            return;
        for (const stack of stacks) {
            this.stackOptions[stack.stackArtifactId] = {
                ...this.stackOptions[stack.stackArtifactId],
                [key]: value,
            };
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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